import logging
import datetime

from sandbox import sdk2
from sandbox import common
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.app_host.AutoBuildAppHostConfigBundle import get_dashboard_name, AutoBuildAppHostConfigBundle

IS_WORK_DAY = "is_work_day"
DESCRIPTION_SEPARATOR = " | "
DATE_FORMAT = "%Y-%m-%d"


class BuildAndReleaseProductionResources(sdk2.Task):
    """
    Task for building mapping and routing config in automated way
    """
    class Requirements(sdk2.Task.Requirements):
        disk_space = 1000

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(required=True)
        with sdk2.parameters.RadioGroup("Choose vertical", required=True) as choose_vertical:
            choose_vertical.values.WEB = choose_vertical.Value("WEB")
            choose_vertical.values.VIDEO = choose_vertical.Value("VIDEO")
            choose_vertical.values.IMGS = choose_vertical.Value("IMGS")
            choose_vertical.values.ATOM = choose_vertical.Value("ATOM")
            choose_vertical.values.COMMON = choose_vertical.Value("COMMON")
            choose_vertical.values.MAIL = choose_vertical.Value("MAIL")
            choose_vertical.values.MAILCORP = choose_vertical.Value("MAILCORP")
            choose_vertical.values.SHARED = choose_vertical.Value("SHARED", default=True)
            choose_vertical.values.NEWS = choose_vertical.Value("NEWS")

        deploy_ready_tasks = sdk2.parameters.Bool("Deploy ready tasks")
        with deploy_ready_tasks.value[True]:
            mapping_build_task = sdk2.parameters.Task("Mappings build task")
            routing_build_task = sdk2.parameters.Task("Routing config build task")

        use_dev_tasks = sdk2.parameters.Bool("Use dev task to deploy mappings")

    def on_execute(self):
        calendar_token = sdk2.Vault.data("APP_HOST", "calendar_token")
        current_date = datetime.datetime.now().strftime(DATE_FORMAT)
        is_work_day = AutoBuildAppHostConfigBundle.is_work_day(current_date, calendar_token)
        if not is_work_day:
            self.Parameters.description += "{}{} (non-working day)".format(DESCRIPTION_SEPARATOR, current_date)
            return

        if not self.Parameters.deploy_ready_tasks:
            self.create_and_release_tasks()
        else:
            self.Context.graph_name_mapping_id = self.Parameters.mapping_build_task.id if self.Parameters.mapping_build_task else None

        with self.memoize_stage.check_if_tasks_released_and_deploying_mapping:
            logging.info("Checking if tasks released")
            if self.Parameters.deploy_ready_tasks:
                sub_tasks = [t for t in (self.Parameters.mapping_build_task, self.Parameters.routing_build_task) if t]
            else:
                sub_tasks = self.find(status=ctt.Status.Group.BREAK + ctt.Status.Group.FINISH).limit(10)
            for task in sub_tasks:
                if task.status not in [ctt.Status.RELEASED]:
                    raise SandboxTaskFailureError("Task {} failed".format(task.id))

            logging.info("Done checking")
            if self.Context.graph_name_mapping_id:
                dashboard_name = get_dashboard_name(self.Parameters.choose_vertical)

                if self.Parameters.use_dev_tasks:
                    deploy_mapping_task = sdk2.Task["DEPLOY_NANNY_DASHBOARD_DEV"]
                    deployment_task_id = self.Context.graph_name_mapping_id
                else:
                    deploy_mapping_task = sdk2.Task["DEPLOY_NANNY_DASHBOARD"]
                    deployment_task_id = self.Context.graph_name_mapping_id

                deploy_mapping = deploy_mapping_task(
                    self,
                    deployment_task_id=deployment_task_id,
                    deployment_nanny_dashboard_filter="stable-app_host",
                    deployment_nanny_bool_wait=True,
                    deployment_nanny_dashboard_name=dashboard_name,
                    deployment_nanny_dashboard_recipe="app_host_ppsa_auto",
                    deployment_release_status="stable",
                    vault_name="nanny_oauth_token",
                    vault_owner="APP_HOST",
                    wait_deployment=120
                ).enqueue()

                self.Context.deploy_mapping_id = deploy_mapping.id

                logging.info(self.Context.deploy_mapping_id)

                raise sdk2.WaitTask(
                    [
                        self.Context.deploy_mapping_id
                    ],
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True
                )

        with self.memoize_stage.checking_deploy_tasks:
            logging.info("Release ready tasks")
            sub_tasks = self.find(status=ctt.Status.Group.BREAK + ctt.Status.Group.FINISH).limit(10)
            for task in sub_tasks:
                if task.status not in [ctt.Status.SUCCESS, ctt.Status.RELEASED]:
                    raise SandboxTaskFailureError("Task {} failed".format(task.id))

    def create_and_release_tasks(self):
        with self.memoize_stage.create_tasks:
            logging.info("Create tasks")

            graph_name_mapping_task = sdk2.Task["BUILD_APP_HOST_GRAPH_NAME_MAPPING"]

            graph_name_mapping = graph_name_mapping_task(
                self,
                description=self.Parameters.description,
                choose_vertical=self.Parameters.choose_vertical,
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            ).enqueue()

            self.Context.graph_name_mapping_id = graph_name_mapping.id

            logging.info(self.Context.graph_name_mapping_id)

            raise sdk2.WaitTask(
                [
                    self.Context.graph_name_mapping_id,
                ],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True
            )

        with self.memoize_stage.release_tasks:
            logging.info("Release ready tasks")
            sub_tasks = self.find(status=ctt.Status.Group.BREAK + ctt.Status.Group.FINISH).limit(10)
            for task in sub_tasks:
                if task.status not in [ctt.Status.SUCCESS]:
                    raise SandboxTaskFailureError("Task {} failed".format(task.id))

            self.server.release(
                task_id=self.Context.graph_name_mapping_id,
                type=ctt.ReleaseStatus.STABLE,
                subject=self.Parameters.description
            )

            logging.info("Done releasing")
            raise sdk2.WaitTask(
                [
                    self.Context.graph_name_mapping_id,
                ],
                common.utils.chain(ctt.Status.RELEASED, ctt.Status.NOT_RELEASED, ctt.Status.Group.BREAK),
                wait_all=True
            )

