# -*- coding: utf-8 -*-

import logging
import os
import sys

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.common import constants
from sandbox.projects.common.app_host.options import ChooseVertical, get_verticals, GraphGenerator
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.app_host import has_error

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxStringParameter, SandboxBoolParameter
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


resource_map = {
    'ATOM': app_host_resources.APP_HOST_BACKENDS_CONFIG_ATOM,
    'NEWS': app_host_resources.APP_HOST_BACKENDS_CONFIG_NEWS,
    'WEB': app_host_resources.APP_HOST_BACKENDS_CONFIG_WEB,
    'IMGS': app_host_resources.APP_HOST_BACKENDS_CONFIG_IMGS,
    'VIDEO': app_host_resources.APP_HOST_BACKENDS_CONFIG_VIDEO,
    'VIDEO_HOSTING': app_host_resources.APP_HOST_BACKENDS_CONFIG_VIDEO_HOSTING,
    'COMMON': app_host_resources.APP_HOST_BACKENDS_CONFIG_COMMON,
    'MAIL': app_host_resources.APP_HOST_BACKENDS_CONFIG_MAIL,
    'MAILCORP': app_host_resources.APP_HOST_BACKENDS_CONFIG_MAILCORP,
    'SHARED': app_host_resources.APP_HOST_BACKENDS_CONFIG_SHARED,
    'MUSIC': app_host_resources.APP_HOST_BACKENDS_CONFIG_MUSIC,
}


class ChooseAppBackendsToBuild(SandboxStringParameter):
    name = 'choose_app_to_build'
    description = 'Choose app to build'
    choices = (
        ('all', 'all'),
        ('app_host', 'app_host'),
        ('src_setup', 'src_setup'),
        ('http_adapter', 'http_adapter'),
    )

    default_value = 'all'


class Verbose(SandboxBoolParameter):
    name = 'verbose'
    description = "Add -vvv to build to be more verbose"
    default_value = False


class BuildAppHostBackendsConfig(nanny.ReleaseToNannyTask, SandboxTask):
    type = "BUILD_APP_HOST_BACKENDS_CONFIG"
    execution_space = 2048
    client_tags = ~ctc.Tag.IVA

    input_parameters = [ChooseVertical, ChooseAppBackendsToBuild,
                        GraphGenerator, SandboxArcadiaUrlParameter, Verbose]

    def create_backends_config_resource(self, src_path, vertical):
        path = src_path + '.copy'
        copy_path(src_path, path)
        self.create_resource("{} {}".format(self.descr, vertical),
                             path,
                             resource_map[vertical],
                             attributes={"vertical": vertical})

    def on_execute(self):
        with self.current_action('Downloading generator config'):
            url = self.ctx.get(constants.ARCADIA_URL_KEY)

            parsed_url = Arcadia.parse_url(url)

            logging.info("============= [BuildAppHostBackendsConfig.on_execute]\n\turl: {}\n\tparsed_url: {}".format(url, parsed_url))

            assert parsed_url.revision
            base_url = url.split("@")[0] + "/web/app_host/conf/graph_generator/"

            conf_dir = os.path.abspath("conf_dir")
            Arcadia.export("{}@{}".format(base_url, parsed_url.revision), conf_dir)
            sys.path.append(conf_dir)
            logging.debug(sys.path)

            graph_generator = self.sync_resource(GraphGenerator.get_resource_from_ctx(self.ctx))

        with self.current_action('Creating backend config'):
            vertical_dir = os.path.join(conf_dir, 'vertical')
            logging.debug("verticals to build {}".format(get_verticals(self.ctx)))
            verticals = [v for v in get_verticals(self.ctx) if os.path.isdir(os.path.join(vertical_dir, v))]
            msgs = []
            for vertical in verticals:
                path = '_backends.{}.json'.format(vertical)
                command = [graph_generator]

                if self.ctx.get(Verbose.name):
                    command.append('-vvv')

                command.extend([
                    'backends', vertical,
                    '-a', self.ctx.get(ChooseAppBackendsToBuild.name),
                    '-g', vertical_dir,
                    '-o', self.abs_path()
                ])

                result = run_process(
                    command,
                    shell=True,
                    check=True,
                    log_prefix="graph_generator"

                )

                stderr = open(result.stderr_path or result.stdout_path).readlines()

                if has_error(stderr):
                    msgs.append(vertical + '\n' + '\n'.join(stderr))

                self.create_resource("{} {}".format(self.descr, vertical),
                                     path,
                                     resource_types.APP_HOST_BACKENDS_CONFIG,
                                     attributes={"vertical": vertical})

                self.create_backends_config_resource(path, vertical)

            self.set_info("\n".join(msgs))
            if len(msgs):
                raise SandboxTaskFailureError("Not all backends resolved")
