import socket


class Location:
    UNKNOWN = 1
    MSK = 2
    SAS = 3
    AMS = 4
    MAN = 5
    SLB = 100


LOCSTR_TO_LOCATION_MAP = {
    'msk': Location.MSK,
    'sas': Location.SAS,
    'ams': Location.AMS,
    'man': Location.MAN,
    'slb': Location.SLB
}

LOCATION_TO_LOCSTR_MAP = {v: k for k, v in LOCSTR_TO_LOCATION_MAP.items()}

DCSTR_TO_LOCATION_MAP = {
    'ams': Location.AMS,
    'myt': Location.MSK,
    'ugrb': Location.MSK,
    'iva': Location.MSK,
    'fol': Location.MSK,
    'man': Location.MAN,
    'sas': Location.SAS,
    'slb': Location.SLB
}


def location_from_string(loc_str):
    if loc_str is None:
        return Location.UNKNOWN
    return LOCSTR_TO_LOCATION_MAP.get(loc_str.lower(), Location.UNKNOWN)


def location_to_string(loc):
    return LOCATION_TO_LOCSTR_MAP.get(loc, 'unknown')


def location_from_dc(dc):
    if dc is None:
        return Location.UNKNOWN
    return DCSTR_TO_LOCATION_MAP.get(dc.lower(), Location.UNKNOWN)


class Instance:
    def __init__(self, hostname, port):
        self.hostname = hostname
        self.port = port
        self.dc = None
        self.location = Location.UNKNOWN
        self.ipv4addr = None
        self.ipv6addr = None
        self.model = None
        self.min_weight = 0.00
        self.weight = 0.5
        self.max_weight = 1.0
        self.protocol = 'post'
        self.path = None

    def __repr__(self):
        return 'h@{}:{} ip:{} path:{}'.format(
            self.hostname,
            self.port,
            self.ipv6addr,
            self.path
        )

    def __eq__(self, other):
        return (isinstance(other, self.__class__) and
                self.hostname == other.hostname and
                self.port == other.port and
                self.protocol == other.protocol and
                self.path == other.path)

    def __ne__(self, other):
        return not self.__eq__(other)


def dns_resolve(host, ipv6=True):
    # TODO: retries, timeout
    proto = socket.AF_INET6
    if ipv6 is False:
        proto = socket.AF_INET
    ret = []
    res_data = socket.getaddrinfo(host, None, proto)
    for record in res_data:
        ret.append(record[4][0])
        return list(set(ret))


class Source:
    def __init__(self, name):
        self.name = name
        self.use_local = False
        self.use_remote = True
        self.local_balancing_scheme = 'weighted'
        self.remote_balancing_scheme = 'weighted'
        self.path = None
        self.instances = []

    def __repr__(self):
        return 'name: {} use_local {} use_remote {} local {} remote {} path {} instances {}'.format(
            self.name,
            self.use_local,
            self.use_remote,
            self.local_balancing_scheme,
            self.remote_balancing_scheme,
            self.path,
            self.instances
        )


class Configuration:
    def __init__(self, name):
        self.name = name
        self.location = Location.UNKNOWN
        # map of source_name -> list[Instance]
#        self.instances = {}
        # map of source_name -> Source
        self.sources = {}

    def __repr__(self):
        return 'name: {} location: {} sources: {}'.format(self.name, self.location, self.sources)
