import glob
import os.path
import os

import sandbox.sandboxsdk.parameters as sb_params
import sandbox.sdk2.parameters as parameters
import sandbox.common.types.client as ctc
import sandbox.projects.common.search.gdb as gdb
from sandbox.projects import resource_types
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.app_host import parse_release_version
from sandbox.projects.common.releases import Releaser, ShouldMakeNewRelease
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
import sandbox.projects.common.utils as utils
import sandbox.projects.common.build.parameters as build_params
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.app_host.options import ChooseVertical, get_verticals

from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.app_host.common import UseFuseRepositoryParameter


_APP_HOST_PATH = "apphost/daemons/app_host/app_host"
_RESOLVER_DAEMON_PATH = "apphost/daemons/resolver/resolver"

_APP_HOST_SERVICE_NAME = "app_host"
_APP_HOST_RELEASE_VAULT_NAME = "ah-releases"
_APP_HOST_RELEASE_VAULT_OWNER = "robot-ah-releases"
_APP_HOST_RELEASE_DIRECTORY = "arcadia:/arc/branches/app_host/releases"


def _tools_dir(package_dir):
    return os.path.join(package_dir, "tools")


def _add_tool(build_dir, package_dir, name):
    tool_path = os.path.join(package_dir, name)

    copy_path(
        os.path.join(build_dir, "apphost/tools/%s/%s" % (name, name)),
        tool_path
    )

    copy_path(tool_path, _tools_dir(package_dir))


class ClearBuild(sb_params.SandboxBoolParameter):
    name = consts.CLEAR_BUILD_KEY
    description = 'Clear build'
    default_value = False
    group = 'Base build params'


class BuildSystem(parameters.String):
    name = consts.BUILD_SYSTEM_KEY
    description = 'Build system'
    required = True
    default_value = consts.SEMI_DISTBUILD_BUILD_SYSTEM
    group = 'Build system params'
    choices = [
        ('Ya', consts.YMAKE_BUILD_SYSTEM),
        # TODO: remove after experiments
        ('Ya force', consts.YA_MAKE_FORCE_BUILD_SYSTEM),
        ('semi-distbuild', consts.SEMI_DISTBUILD_BUILD_SYSTEM),
        ('distbuild', consts.DISTBUILD_BUILD_SYSTEM),
    ]
    sub_fields = {
        consts.YMAKE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
        consts.YA_MAKE_FORCE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
    }


_TOOLS = [
    "event_log_dump",
    "servant_client",
    "grpc_client",
    "converter",
    "make_tank_ammo",
]


class BuildAppHostBundle(nanny.ReleaseToNannyTask, ArcadiaTask):

    """Builds a bundle with app_host and some related tools"""

    type = "BUILD_APP_HOST_BUNDLE"

    execution_space = 100000

    input_parameters = [
        ChooseVertical,
        ClearBuild,
        BuildSystem,
        UseFuseRepositoryParameter,
        SandboxArcadiaUrlParameter,
        ShouldMakeNewRelease,
        build_params.ThinLTO,
   ] + [k for k in build_params.get_build_system_params() if k.name != "clear_build" and k.name != "build_system"]

    _releaser = Releaser(_APP_HOST_SERVICE_NAME,
                         _APP_HOST_RELEASE_VAULT_NAME,
                         _APP_HOST_RELEASE_VAULT_OWNER,
                         _APP_HOST_RELEASE_DIRECTORY)

    resources_map = {
        "ATOM": resource_types.APP_HOST_BUNDLE_ATOM,
        "WEB": resource_types.APP_HOST_BUNDLE_WEB,
        "NEWS": resource_types.APP_HOST_BUNDLE_NEWS,
        "IMGS": resource_types.APP_HOST_BUNDLE_IMGS,
        "VIDEO": resource_types.APP_HOST_BUNDLE_VIDEO,
        "COMMON": resource_types.APP_HOST_BUNDLE_COMMON,
        "MAIL": app_host_resources.APP_HOST_BUNDLE_MAIL,
        "MAILCORP": app_host_resources.APP_HOST_BUNDLE_MAILCORP,
        "SHARED": app_host_resources.APP_HOST_BUNDLE_SHARED,
        "DISTRICT": app_host_resources.APP_HOST_BUNDLE_DISTRICT,
        "WEATHER": app_host_resources.APP_HOST_BUNDLE_WEATHER,
    }

    client_tags = ctc.Tag.Group.LINUX

    def add_app_host(self, build_dir, package_dir):
        copy_path(
            os.path.join(build_dir, _APP_HOST_PATH),
            os.path.join(package_dir, "app_host")
        )
        BuildAppHostBundle._releaser.add_release_file(package_dir)

    def do_execute(self):
        if self.ctx.get(UseFuseRepositoryParameter.name):
            with sdk.mount_arc_path(self.ctx[consts.ARCADIA_URL_KEY]) as arcadia_dir:
                self.build(arcadia_dir)
            return
        arcadia_dir = self.get_arcadia_src_dir()
        self.build(arcadia_dir)

    def build(self, arcadia_dir):
        gdb.append_to_stable_only(self)
        BuildAppHostBundle._releaser.set_task(self)
        BuildAppHostBundle._releaser.init_release_dir()
        # prepare build environment
        build_dir = self.abs_path("build")
        make_folder(build_dir)
        os.chdir(build_dir)

        sdk.do_build(
            utils.get_or_default(self.ctx, BuildSystem),
            arcadia_dir,
            ["apphost/tools", "apphost/daemons"],
            self.ctx['build_type'],
            clear_build=utils.get_or_default(self.ctx, ClearBuild),
            results_dir=build_dir,
            thinlto=utils.get_or_default(self.ctx, build_params.ThinLTO),
            def_flags=self.ctx['definition_flags']
        )

        # create package
        url = parse_release_version(self.ctx[consts.ARCADIA_URL_KEY])
        pkg_name = "app_host_bundle_%s" % url
        BuildAppHostBundle._releaser.make_new_release_if_needed(self.ctx[consts.ARCADIA_URL_KEY])

        package_dir = os.path.join(build_dir, pkg_name)
        os.chdir(build_dir)
        make_folder(package_dir)
        make_folder(_tools_dir(package_dir))

        copy_path(
            os.path.join(build_dir, _RESOLVER_DAEMON_PATH),
            os.path.join(package_dir, "resolver")
        )

        self.add_app_host(build_dir, package_dir)

        for tool in _TOOLS:
            _add_tool(build_dir, package_dir, tool)

        run_process("tar czvf %s.tar.gz %s" % (pkg_name, pkg_name), shell=True)

        tar = glob.glob(os.path.join(build_dir, "*.tar.gz"))[-1]

        # save created package as resource
        resource_dir = self.abs_path()
        copy_path(tar, resource_dir)

        tar_name = os.path.basename(tar)

        for vertical in get_verticals(self.ctx):
            new_path = os.path.join(resource_dir, tar_name + vertical + ".tar.gz")
            copy_path(os.path.join(resource_dir, tar_name), new_path)
            self.create_resource(
                description="app_host for {} from {}".format(vertical, str(Arcadia.parse_url(self.ctx[consts.ARCADIA_URL_KEY]))),
                resource_path=new_path,
                resource_type=self.resources_map[vertical],
                arch=self.arch
            )

        self.create_resource(
            description=tar_name,
            resource_path=os.path.join(resource_dir, tar_name),
            resource_type=resource_types.APP_HOST_BUNDLE,
            arch=self.arch,
        )

        self.create_resource(
            description="app_host_tools_%s" % url,
            resource_path=_tools_dir(package_dir),
            resource_type=resource_types.APP_HOST_TOOLS_BUNDLE,
            arch=self.arch,
        )


__Task__ = BuildAppHostBundle
