import json
import logging
import os

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.app_host import resources as ah_resources
from sandbox.projects.app_host.BuildAppHostConfigBundleDiff import ConfigBundleTaskOld, ConfigBundleTaskNew
from sandbox.projects.common import constants as consts
from sandbox.projects.common.app_host.options import get_verticals, ChooseVertical, GraphGenerator, SmokeTest, \
    LastReleasedAppHostBundle, get_last_stable_resource, PVA
from sandbox.projects.common.app_host import has_error
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.nanny import nanny

import sandbox.projects.app_host.BuildAppHostBackendsConfig as BuildAppHostBackendsConfig

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, TaskSelector, SandboxBoolParameter
from sandbox.sandboxsdk.paths import make_folder, copy_path, remove_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia


class BackendsConfigTask(TaskSelector):
    name = 'BUILD_APP_HOST_BACKENDS_CONFIG'
    task_type = 'BUILD_APP_HOST_BACKENDS_CONFIG'
    description = "OR select task with backends (optional)"
    required = False
    default_value = None


class BuildOnlyServiceResources(SandboxBoolParameter):
    name = 'build_only_service_resources'
    description = 'Build only common resources'
    default_value = False


class TransparentBuild(TaskSelector):
    name = 'transparent_build'
    task_type = 'HTTP_UPLOAD_2'
    description = 'Publish an existing bundle'
    default_value = None


class BuildServiceResources(SandboxBoolParameter):
    name = 'build_service_resources'
    description = 'Build cross service resources (loop.conf etc)'
    default_value = True
    required = True
    sub_fields = {'true': [BuildOnlyServiceResources.name]}


class BuildTemplatesDir(SandboxBoolParameter):
    name = 'build_templates_dir'
    description = 'Save templates dir'
    default_value = True


class EnrichGraphsWithRevisions(SandboxBoolParameter):
    name = 'enrich_graphs_revision'
    description = 'Enrich graphs dir with production graphs'
    default_value = False


class RunSmokeTest(SandboxBoolParameter):
    name = 'run_smoke_test'
    description = 'Run smoke test for each generated config bundle'
    default_value = False
    sub_fields = {'true': [LastReleasedAppHostBundle.name, SmokeTest.name]}


class NoBackEndsInGraph(SandboxBoolParameter):
    name = 'no_backends_in_graph'
    description = 'Generate bundle w/o backends in graphs'
    default_value = False


class AllowGraphRemoval(SandboxBoolParameter):
    name = 'allow_graph_removal'
    description = 'Enable graph removal'
    default_value = False


class SingleGraphRevision(SandboxBoolParameter):
    name = 'single_graph_revision'
    description = 'Single revison for all graphs'
    default_value = False


class ConfigBundleTaskOldOptional(ConfigBundleTaskOld):
    required = False
    description = "OLD config bundle task (optional)"


def _make_url(base_url, subpath, rev):
    return "%s/%s@%s" % (base_url, subpath, rev)


BACKENDS_BUILD_RETRIES = 3


class BuildAppHostConfigBundle(nanny.ReleaseToNannyTask, ArcadiaTask):
    """Builds a bundle with app_host config and graphs"""

    type = "BUILD_APP_HOST_CONFIG_BUNDLE"

    execution_space = 4000

    input_parameters = [
        ChooseVertical,
        BackendsConfigTask,
        TransparentBuild,
        BuildServiceResources,
        BuildOnlyServiceResources,
        GraphGenerator,
        PVA,
        ConfigBundleTaskOldOptional,
        SandboxArcadiaUrlParameter,
        BuildTemplatesDir,
        EnrichGraphsWithRevisions,
        LastReleasedAppHostBundle,
        SmokeTest,
        RunSmokeTest,
        NoBackEndsInGraph,
        SingleGraphRevision,
        AllowGraphRemoval
    ]

    global_pva_configs = {
        "horizon_app_host_instancectl.conf": ah_resources.HorizonAppHostInstancectlConf,
        "production_app_host_instancectl.conf": resource_types.APP_HOST_PRODUCTION_INSTANCECTL_CONF,
        "hamster_app_host_instancectl.conf": ah_resources.APP_HOST_HAMSTER_INSTANCECTL_CONF,
        "testing_app_host_instancectl.conf": resource_types.APP_HOST_TESTING_INSTANCECTL_CONF,
        "perftest_app_host_instancectl.conf": ah_resources.APP_HOST_PERFTEST_INSTANCECTL_CONF,
        "production_app_host_instancectl_isolated.conf": ah_resources.APP_HOST_PRODUCTION_INSTANCECTL_ISOLATED_CONF,
        "hamster_app_host_instancectl_isolated.conf": ah_resources.APP_HOST_HAMSTER_INSTANCECTL_ISOLATED_CONF,
        "testing_app_host_instancectl_isolated.conf": ah_resources.APP_HOST_TESTING_INSTANCECTL_ISOLATED_CONF,
        "perftest_app_host_instancectl_isolated.conf": ah_resources.APP_HOST_PERFTEST_INSTANCECTL_ISOLATED_CONF,
    }

    pva_configs = {
        'production_app_host_instancectl.conf': {
            'SHARED': ah_resources.APP_HOST_PRODUCTION_INSTANCECTL_CONF_SHARED,
        },
        'production_app_host_instancectl_isolated.conf': {
            'SHARED': ah_resources.APP_HOST_PRODUCTION_INSTANCECTL_ISOLATED_CONF_SHARED,
        },
        "testing_app_host_instancectl_isolated.conf": {
            'MAIL': ah_resources.APP_HOST_TESTING_INSTANCECTL_ISOLATED_CONF_MAIL,
            'MAILCORP': ah_resources.APP_HOST_TESTING_INSTANCECTL_ISOLATED_CONF_MAILCORP,
        }
    }

    # environment = (PipEnvironment('colour', '0.1.2', use_wheel=True),)

    resource_map = {
        'ATOM': resource_types.APP_HOST_CONFIG_BUNDLE_ATOM,
        'NEWS': resource_types.APP_HOST_CONFIG_BUNDLE_NEWS,
        'WEB': resource_types.APP_HOST_CONFIG_BUNDLE_WEB,
        'IMGS': resource_types.APP_HOST_CONFIG_BUNDLE_IMGS,
        'VIDEO': resource_types.APP_HOST_CONFIG_BUNDLE_VIDEO,
        'COMMON': resource_types.APP_HOST_CONFIG_BUNDLE_COMMON,
        'MAIL': ah_resources.APP_HOST_CONFIG_BUNDLE_MAIL,
        'MAILCORP': ah_resources.APP_HOST_CONFIG_BUNDLE_MAILCORP,
        'SHARED': ah_resources.APP_HOST_CONFIG_BUNDLE_SHARED,
    }

    graph_name_mapping_map = {
        'ATOM': resource_types.APP_HOST_GRAPH_NAME_MAPPING_ATOM,
        'NEWS': resource_types.APP_HOST_GRAPH_NAME_MAPPING_NEWS,
        'WEB': resource_types.APP_HOST_GRAPH_NAME_MAPPING_WEB,
        'IMGS': resource_types.APP_HOST_GRAPH_NAME_MAPPING_IMGS,
        'VIDEO': resource_types.APP_HOST_GRAPH_NAME_MAPPING_VIDEO,
        'COMMON': resource_types.APP_HOST_GRAPH_NAME_MAPPING_COMMON,
        'MAIL': ah_resources.APP_HOST_GRAPH_NAME_MAPPING_MAIL,
        'MAILCORP': ah_resources.APP_HOST_GRAPH_NAME_MAPPING_MAILCORP,
        'SHARED': ah_resources.APP_HOST_GRAPH_NAME_MAPPING_SHARED,
    }

    config_map = {
        'production_app_host.json': {
            'ATOM': resource_types.APP_HOST_PRODUCTION_APP_HOST_JSON_ATOM,
            'NEWS': resource_types.APP_HOST_PRODUCTION_APP_HOST_JSON_NEWS,
            'WEB': resource_types.APP_HOST_PRODUCTION_APP_HOST_JSON_WEB,
            'IMGS': resource_types.APP_HOST_PRODUCTION_APP_HOST_JSON_IMGS,
            'VIDEO': resource_types.APP_HOST_PRODUCTION_APP_HOST_JSON_VIDEO,
            'COMMON': resource_types.APP_HOST_PRODUCTION_APP_HOST_JSON_COMMON,
            'MAIL': ah_resources.APP_HOST_PRODUCTION_APP_HOST_JSON_MAIL,
            'MAILCORP': ah_resources.APP_HOST_PRODUCTION_APP_HOST_JSON_MAILCORP,
            'SHARED': ah_resources.APP_HOST_PRODUCTION_APP_HOST_JSON_SHARED,
            'DISTRICT': ah_resources.APP_HOST_PRODUCTION_APP_HOST_JSON_DISTRICT,
            'MUSIC': ah_resources.APP_HOST_PRODUCTION_APP_HOST_JSON_MUSIC,
        },
        'hamster_app_host.json': {
            'WEB': ah_resources.APP_HOST_HAMSTER_APP_HOST_JSON_WEB,
            'IMGS': ah_resources.APP_HOST_HAMSTER_APP_HOST_JSON_IMGS,
            'VIDEO': ah_resources.APP_HOST_HAMSTER_APP_HOST_JSON_VIDEO,
            'SHARED': ah_resources.APP_HOST_HAMSTER_APP_HOST_JSON_SHARED,
        },
        'dev_app_host.json': {
            'WEB': ah_resources.APP_HOST_DEV_APP_HOST_JSON_WEB,
            'IMGS': ah_resources.APP_HOST_DEV_APP_HOST_JSON_IMGS,
            'VIDEO': ah_resources.APP_HOST_DEV_APP_HOST_JSON_VIDEO,
            'COMMON': ah_resources.APP_HOST_DEV_APP_HOST_JSON_COMMON,
            'MAIL': ah_resources.APP_HOST_DEV_APP_HOST_JSON_MAIL,
            'MAILCORP': ah_resources.APP_HOST_DEV_APP_HOST_JSON_MAILCORP,
            'SHARED': ah_resources.APP_HOST_DEV_APP_HOST_JSON_SHARED,
        },
        'production_resolver.json': {
            'ATOM': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_ATOM,
            'NEWS': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_NEWS,
            'WEB': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_WEB,
            'IMGS': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_IMGS,
            'VIDEO': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_VIDEO,
            'COMMON': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_COMMON,
            'MAIL': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_MAIL,
            'MAILCORP': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_MAILCORP,
            'SHARED': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_SHARED,
            'MUSIC': ah_resources.APP_HOST_PRODUCTION_RESOLVER_JSON_MUSIC,
        },
        'hamster_resolver.json': {
            'WEB': ah_resources.APP_HOST_HAMSTER_RESOLVER_JSON_WEB,
            'IMGS': ah_resources.APP_HOST_HAMSTER_RESOLVER_JSON_IMGS,
            'VIDEO': ah_resources.APP_HOST_HAMSTER_RESOLVER_JSON_VIDEO,
            'SHARED': ah_resources.APP_HOST_HAMSTER_RESOLVER_JSON_SHARED,
        },
        'production_app_host_push_client.yml': {
            'WEB': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_CONFIG_WEB,
            'SHARED': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_CONFIG_SHARED,
            'COMMON': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_CONFIG_COMMON,
            'MAIL': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_CONFIG_MAIL,
            'MAILCORP': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_CONFIG_MAILCORP,
            'IMGS': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_CONFIG_IMGS,
            'VIDEO': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_CONFIG_VIDEO,
            'DISTRICT': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_CONFIG_DISTRICT,
            'MUSIC': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_CONFIG_MUSIC,
        },
        'production_app_host_push_client_fork.yml': {
            'SHARED': ah_resources.APP_HOST_PRODUCTION_PUSH_CLIENT_FORK_CONFIG_SHARED,
        }
    }

    push_client_fork_set = {
        'SHARED',
    }

    client_tags = ctc.Tag.Group.LINUX

    def __setup_paths(self, base_url, url):
        _GRAPH_GENERATOR_CONF_SUBPATH = "conf/graph_generator"

        self.__conf_dir = os.path.abspath("conf_dir")
        self.__split_dir = os.path.abspath("split")
        self.__logs_dir = os.path.abspath("logs")
        make_folder(self.__logs_dir)

        Arcadia.checkout(
            _make_url(base_url, _GRAPH_GENERATOR_CONF_SUBPATH, url.revision),
            self.__split_dir)
        Arcadia.export(
            _make_url(base_url, "conf", url.revision),
            self.__conf_dir)

    def __run_pva(self, config_name, vertical, output):
        pva = self.sync_resource(self.__get_pva())
        pva_config = os.path.join(self.__conf_dir, "pva", "config.json")

        logging.info("Building config {} for vertical {} into {} with PVA"
                     .format(config_name, vertical, output))

        logging.debug("PVA Config: {}".format(json.load(open(pva_config))))

        try:
            run_process([pva, pva_config, vertical or "''", config_name, output],
                        shell=True,
                        check=True,
                        outputs_to_one_file=False,
                        log_prefix="pva")
        except Exception as e:
            logging.error("Unexpected PVA error: ", e)
            raise

    @staticmethod
    def __remove_locations(src_path):
        for location in os.listdir(src_path):
            if location == '_ALL':
                continue
            graphs_dir = os.path.join(src_path, location, 'graphs')
            if os.path.isdir(graphs_dir):
                remove_path(os.path.join(src_path, location))

    @staticmethod
    def __collect_revisions(src_path):
        revisions = {}
        for location in os.listdir(src_path):
            if location == '_ALL':
                continue
            graphs_dir = os.path.join(src_path, location, 'graphs')
            if os.path.isdir(graphs_dir):
                for g in os.listdir(graphs_dir):
                    revisions[g] = os.path.join(graphs_dir, g)
        return revisions

    def __copy_revisions(self, pkg_path):
        revisions = self.__collect_revisions(pkg_path)
        logging.info("revisions: {}".format(revisions))

        dst_path = os.path.join(pkg_path, '_ALL', 'graphs')
        for name, p in revisions.viewitems():
            copy_path(p, os.path.join(dst_path, name))

    def __restore_trunk_graphs(self, graph_generator, pkg_path):
        dst_path = os.path.join(pkg_path, '_ALL', 'graphs')
        create_trunk_graphs_cmd = [
            graph_generator,
            '-vvvvv',
            'create_trunk_graphs',
            '-o', dst_path
        ]

        result = run_process(
            create_trunk_graphs_cmd,
            shell=True,
            check=True,
            log_prefix="graph_generator_create_trunk_graphs"
        )

        self.__save_output(result.stdout_path, os.path.join(pkg_path, 'create_trunk_graphs.stdout.txt'),
                           "create_trunk_graphs")

    def init_or_decrement_subtask_retry_count(self):
        retries_left = self.ctx.get("backends_config_task_retries", BACKENDS_BUILD_RETRIES + 1)

        self.ctx["backends_config_task_retries"] = retries_left - 1

        if retries_left < 0:
            raise SandboxTaskFailureError("child task failed, retry limit exceeded")

    def __get_graph_generator(self):
        if not hasattr(self, '__graph_generator'):
            self.__graph_generator = GraphGenerator.get_resource_from_ctx(self.ctx)
        return self.__graph_generator

    def __get_pva(self):
        if not hasattr(self, '__pva'):
            self.__pva = PVA.get_resource_from_ctx(self.ctx)
        return self.__pva

    def __copy_backends(self, src_path, vertical):
        name = os.path.basename(src_path)
        dst_path = self.abs_path(name)
        copy_path(src_path, dst_path)
        self.create_resource(
            description=name,
            resource_path=dst_path,
            resource_type=BuildAppHostBackendsConfig.resource_map[vertical],
            arch="any"
        )

    def __save_output(self, src_path, dst_path, description):
        copy_path(src_path, dst_path)
        self.create_resource(
            description=description,
            resource_path=dst_path,
            resource_type=resource_types.OTHER_RESOURCE,
            arch="any"
        )

    def __transparent_build(self, task_id):
        resources = channel.sandbox.list_resources(task_id=task_id, status="READY")

        for r in resources:
            vertical = r.attributes.get('vertical')
            if vertical:
                src_path = self.sync_resource(r.id)
                name = os.path.basename(src_path)
                dst_path = self.abs_path(name)
                copy_path(src_path, dst_path)

                self.create_resource(
                    description=name,
                    resource_path=dst_path,
                    resource_type=r.type,  # self.resource_map.get(vertical)
                    arch="any"
                )

                self.mark_resource_ready(r.id)

    def do_execute(self):
        if self.ctx.get(TransparentBuild.name):
            self.__transparent_build(self.ctx.get(TransparentBuild.name))
            return

        if not self.ctx.get(BackendsConfigTask.name) and \
                not self.ctx.get("backends_config_task_id") and \
                not self.ctx.get(BuildOnlyServiceResources.name):
            if not self.ctx.get(GraphGenerator.name):
                context = self.ctx.copy()
                context[GraphGenerator.name] = self.__get_graph_generator()
            else:
                context = self.ctx

            context["choose_app_to_build"] = "app_host"

            self.init_or_decrement_subtask_retry_count()
            subtask = self.create_subtask(
                task_type="BUILD_APP_HOST_BACKENDS_CONFIG",
                description=self.descr,
                input_parameters=context,
                priority=self.priority,
                inherit_notifications=True
            )
            self.ctx["backends_config_task_id"] = subtask.id
            logging.info(
                "Starting subtask BUILD_APP_HOST_BACKENDS_CONFIG {}".format(
                    subtask
                )
            )
            self.wait_tasks(
                [subtask],
                (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED,
                 self.Status.EXCEPTION, self.Status.TIMEOUT),
                True
            )

        # prepare build environment
        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)

        assert parsed_url.revision
        base_url = url.split("@")[0] + "/apphost"

        self.__setup_paths(base_url, parsed_url)

        if not self.ctx.get(BuildOnlyServiceResources.name):
            graph_generator = self.sync_resource(self.__get_graph_generator())

            # get backends config resource
            task_id = self.ctx.get("backends_config_task_id") or self.ctx.get(BackendsConfigTask.name)
            resource_ids = []
            if task_id is not None:
                resources = channel.sandbox.list_resources(resource_type="APP_HOST_BACKENDS_CONFIG",
                                                           task_id=task_id,
                                                           status="READY")
                if resources:
                    resource_ids = [(res.id, res.attributes) for res in resources]
                else:
                    del self.ctx["backends_config_task_id"]
                    self.do_execute()
            else:
                raise SandboxTaskFailureError("No task_id in backend task or selected config task")

            msgs = []
            pkg_version = parsed_url.revision
            logging.debug("resource_ids {}".format(resource_ids))
            for resid, attributes in resource_ids:
                vertical = attributes.get("vertical")
                if vertical is None:
                    continue

                pkg_name = "app_host_config_bundle_{}_{}".format(vertical, pkg_version)
                pkg_path = self.abs_path(pkg_name)
                tar = pkg_path + ".tar.gz"

                make_folder(pkg_name)
                os.chdir(pkg_name)

                backends_file = self.sync_resource(resid)

                if self.ctx.get(NoBackEndsInGraph.name):
                    self.__copy_backends(backends_file, vertical)

                revision = str(parsed_url.revision)
                gen_bundle_cmd = [
                    graph_generator,
                    '-vvvvv',
                    '--stdout',
                    'bundle', vertical,
                    '-g', os.path.join(self.__split_dir, "vertical")
                ]

                if not self.ctx.get(NoBackEndsInGraph.name):
                    gen_bundle_cmd.extend(['-b', backends_file, '-o', pkg_path])
                else:
                    gen_bundle_cmd.extend(['-o', os.path.join(pkg_path, '_ALL')])

                if self.ctx.get(SingleGraphRevision.name):
                    gen_bundle_cmd.extend(['-r', revision])

                result = run_process(
                    gen_bundle_cmd,
                    shell=True,
                    check=True,
                    log_prefix="graph_generator_bundle"
                )

                self.__save_output(
                    result.stdout_path,
                    os.path.join(pkg_path, 'bundle.stdout.txt'),
                    "%s bundle stdout" % pkg_name
                )

                stderr = open(result.stderr_path).readlines() if result.stderr_path else None
                if stderr:
                    msgs.append(vertical + '\n' + '\n'.join(stderr))

                stdout = open(result.stdout_path).readlines()
                if has_error(stdout):
                    msgs.append(vertical + '\n' + '\n'.join(stdout))

                if self.ctx.get(EnrichGraphsWithRevisions.name):
                    revision_generator_cmd = [
                        graph_generator,
                        '-vvvvv',
                        'revision', vertical,
                        '-G', '0',
                        '-o', pkg_path,
                        '-b', backends_file,
                        '-r', revision,
                        '-m', os.path.join(self.__split_dir, "vertical", vertical, "graph_name_mapping.txt"),
                        '--stdout',
                    ]

                    if self.ctx.get(NoBackEndsInGraph.name):
                        revision_generator_cmd.append('--clear-backends')

                    if self.ctx.get(AllowGraphRemoval.name):
                        cleanup_file_path = os.path.join(self.__split_dir, "vertical", vertical, "_graph_cleanup.txt")
                        revision_generator_cmd.extend(["--cleanup-file", cleanup_file_path])

                    result = run_process(
                        revision_generator_cmd,
                        shell=True,
                        check=True,
                        log_prefix="graph_generator_revision"
                    )

                    self.__save_output(
                        result.stdout_path,
                        os.path.join(pkg_path, 'enricher.stdout.txt'),
                        "%s enricher stdout" % pkg_name
                    )

                    if result.stderr_path:
                        self.__save_output(
                            result.stderr_path,
                            os.path.join(pkg_path, 'enricher.stderr.txt'),
                            "%s enricher stderr" % pkg_name
                        )

                    if self.ctx.get(NoBackEndsInGraph.name):
                        self.__copy_revisions(pkg_path)
                        self.__remove_locations(pkg_path)
                        self.__restore_trunk_graphs(graph_generator, pkg_path)

                for config_name, vertical_settings in self.config_map.items():
                    for _vertical in get_verticals(self.ctx):
                        config_type = vertical_settings.get(_vertical)
                        if config_type is not None:
                            self.__run_pva(config_name, _vertical, output=config_name)

                # copy _inverse_backends.json
                inverse_backends_path = os.path.join(self.__split_dir, "vertical", vertical, '_inverse_backends.json')
                if os.path.exists(inverse_backends_path):
                    copy_path(inverse_backends_path, pkg_path)

                # copy push_client.yml
                push_client_dstpath = os.path.join(pkg_path, 'push_client.yml')
                self.__run_pva("production_app_host_push_client.yml", vertical, push_client_dstpath)

                # copy push_client_fork.yml
                if vertical in self.push_client_fork_set:
                    push_client_fork_dstpath = os.path.join(pkg_path, 'push_client_fork.yml')
                    self.__run_pva("production_app_host_push_client_fork.yml", vertical, push_client_fork_dstpath)

                smoke_test_error = None

                if self.ctx.get(RunSmokeTest.name):
                    app_host_bundle_tar_gz = self.sync_resource(
                        LastReleasedAppHostBundle.get_resource_from_ctx(self.ctx)
                    )
                    run_process("tar xvf {}".format(app_host_bundle_tar_gz), shell=True, check=True, log_prefix="tar")
                    app_host_bundle = os.path.abspath(os.path.basename(app_host_bundle_tar_gz)[:-len(".tar.gz")])
                    app_host_binary = os.path.join(app_host_bundle, "app_host")
                    logging.debug("app_host bundle dir contents: {}".format(os.listdir(app_host_bundle)))
                    logging.debug("app_host binary: {}".format(os.path.isfile(app_host_binary)))

                    smoke_test = self.sync_resource(SmokeTest.get_resource_from_ctx(self.ctx))

                    stable_graph_name_mapping = get_last_stable_resource(self.graph_name_mapping_map[vertical])
                    graph_name_mapping = self.sync_resource(stable_graph_name_mapping)

                    self.ctx["smoke_retries"] = 0

                    run_smoke_test_cmd = [
                        smoke_test,
                        "--app_host", app_host_binary,
                        "--config_bundle", pkg_path,
                        "--graph_name_mapping", graph_name_mapping,
                        "--test_dir", os.path.join(self.__split_dir, "vertical", vertical),
                        "--log_path", os.path.join(self.__logs_dir, "app_host-")
                    ]

                    if self.ctx.get(NoBackEndsInGraph.name):
                        run_smoke_test_cmd.extend(['--backends_file', backends_file])

                    while self.ctx.get("smoke_retries") <= 4:
                        result = run_process(
                            run_smoke_test_cmd,
                            shell=True,
                            check=False,
                            log_prefix="smoke_test",
                            environment={"TVM_SECRET": "Qv3kM8vy450tOXl2XZCjWg"}
                        )

                        report_path = os.path.join(pkg_path,
                                                   "smoke_test_report-{}.html".format(self.ctx.get("smoke_retries")))
                        copy_path(result.stdout_path, report_path)
                        self.create_resource(
                            description="%s smoke_test report" % pkg_name,
                            resource_path=report_path,
                            resource_type=resource_types.APP_HOST_SMOKE_TEST_REPORT,
                            arch="any"
                        )

                        if result.returncode != 0:
                            self.ctx["smoke_retries"] += 1
                            smoke_test_error = "smoke_test failed, code=%s" % result.returncode
                        else:
                            smoke_test_error = None
                            break

                    logs_archive = self.abs_path("logs.tgz")
                    run_process("tar czvf {} {}".format(logs_archive, self.__logs_dir), shell=True, check=True,
                                log_prefix="tar")

                    self.create_resource(
                        description="smoke_test app_host logs",
                        resource_path=logs_archive,
                        resource_type=resource_types.OTHER_RESOURCE,
                        arch="any"
                    )

                    remove_path(app_host_bundle)

                os.chdir("..")

                run_process("tar czvf {} {}".format(tar, pkg_name), shell=True, check=True, log_prefix="tar")

                res_id = self.create_resource(
                    description=pkg_name,
                    resource_path=tar,
                    resource_type=self.resource_map.get(vertical),
                    arch="any"
                )

                if smoke_test_error:
                    raise SandboxTaskFailureError(smoke_test_error)

                self.mark_resource_ready(res_id.id)

            if len(msgs):
                self.set_info("\n".join(msgs))
                raise SandboxTaskFailureError("stderr is not empty")

        if self.ctx.get(BuildServiceResources.name):
            generated_dir = os.path.abspath("generated")
            make_folder(generated_dir)
            os.chdir(generated_dir)

            for config_name, config_type in self.global_pva_configs.items():
                output_path = os.path.join(generated_dir, config_name)

                self.__run_pva(config_name, "", output_path)

                self.create_resource(
                    description="{}@{}".format(config_name, parsed_url.revision),
                    resource_path=output_path,
                    resource_type=config_type,
                    arch="any")

            for config_name, vertical_settings in self.config_map.items() + self.pva_configs.items():
                for vertical in get_verticals(self.ctx):
                    config_type = vertical_settings.get(vertical)
                    if config_type is not None:
                        pos = config_name.rfind('.')
                        if pos != -1:
                            config_file_name = config_name[:pos]
                            config_file_ext = config_name[pos:]
                        else:
                            config_file_name = config_name
                            config_file_ext = ''

                        output_path = os.path.join(
                            generated_dir, "{}.{}{}".format(config_file_name, vertical, config_file_ext)
                        )

                        self.__run_pva(config_name, vertical, output_path)

                        self.create_resource(
                            description="{}@{}".format(config_name, parsed_url.revision),
                            resource_path=output_path,
                            resource_type=config_type,
                            arch="any")

            self.create_resource(
                description="iss_hook_notify@{}".format(parsed_url.revision),
                resource_path=os.path.join(self.__conf_dir, 'iss_hook_notify.py'),
                resource_type=resource_types.APP_HOST_ISS_HOOK_NOTIFY,
                arch="any")

        if self.ctx.get(ConfigBundleTaskOldOptional.name) and not self.ctx.get('diff_config_task_id', None):
            context = {
                ConfigBundleTaskOld.name: self.ctx.get(ConfigBundleTaskOldOptional.name),
                ConfigBundleTaskNew.name: self.id,
                GraphGenerator.name: self.__get_graph_generator()
            }
            subtask = self.create_subtask(
                task_type="BUILD_APP_HOST_CONFIG_BUNDLE_DIFF",
                description=self.descr,
                input_parameters=context,
                priority=self.priority,
                inherit_notifications=True
            )
            self.ctx["diff_config_task_id"] = subtask.id
            logging.info(
                "Starting subtask BUILD_APP_HOST_CONFIG_BUNDLE_DIFF {}".format(
                    subtask
                )
            )
            self.wait_task_completed(subtask)

        if self.ctx.get(BuildTemplatesDir.name):
            self.create_resource(
                description="{}@{}".format("templates", parsed_url.revision),
                resource_path=os.path.join(self.__split_dir, "vertical"),
                resource_type=resource_types.APP_HOST_GRAPH_TEMPLATES,
                arch="any"
            )


__Task__ = BuildAppHostConfigBundle
