# -*- coding: utf-8

from sandbox.sdk2 import Task, Resource, ResourceData, Vault
from sandbox.sdk2.parameters import RadioGroup as RadioGroupParameter, String as StringParameter
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants
import requests

TESTING_ID = 1878
TESTING_URL = 'http://internal.daas-backend-int.locdoc-test.yandex.net/v1/projects/{}/deploy'

PRODUCTION_ID = 1184
PRODUCTION_URL = 'http://internal.daas-backend-int.yandex.net/v1/projects/{}/deploy'


class AppHostDocumentation(Resource):
    pass


class BuildAppHostDoc(Task):
    class Parameters(Task.Parameters):
        vcs_url = StringParameter('VCS url', default='arcadia-arc:/#trunk')

        path = StringParameter('Path to doc', default='apphost/docs')

        with RadioGroupParameter("Target") as target:
            target.values['testing'] = target.Value("Testing", default=True)
            target.values['production'] = target.Value("Production")

    def on_execute(self):
        with arcadiasdk.mount_arc_path(self.Parameters.vcs_url) as arcadia:
            docs_path = self.build(arcadia)
            self.create_resource(docs_path)
            self.deploy(docs_path)
            pass

    def build(self, arcadia):
        path = self.path('docs').as_posix()
        arcadiasdk.do_build(
            build_system=sdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM,
            clear_build=False,
            source_root=arcadia,
            targets=[self.Parameters.path],
            results_dir=path
        )
        return path + '/' + self.Parameters.path + '/docs.tar.gz'

    def create_resource(self, docs_path):
        resource = AppHostDocumentation(self, 'AppHost documentation', docs_path)
        resource_data = ResourceData(resource)
        resource_data.ready()

    def deploy(self, docs_path):
        token = Vault.data('APP_HOST', 'DAAS_TOKEN')
        if self.Parameters.target == 'production':
            target_url = PRODUCTION_URL
            project_id = PRODUCTION_ID
        else:
            target_url = TESTING_URL
            project_id = TESTING_ID
        with open(docs_path, 'rb') as docs:
            response = requests.post(target_url.format(project_id),
                                     files={'archive': docs},
                                     headers={"Authorization": "OAuth {}".format(token)})
            response.raise_for_status()
        pass
