import fileinput
import logging
import sys
import os

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxBoolGroupParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask

from sandbox.projects.common import constants as consts
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.app_host.common import ParentResourcesParameter, get_url_attributes


_RESOURCES = [
    app_host_resources.APP_HOST_EARL_DATA,
    app_host_resources.APP_HOST_EARL_CONFIG_EARL_CONF,
    app_host_resources.APP_HOST_EARL_CONFIG_NGINX_CONF_TEMPLATE,

    app_host_resources.APP_HOST_EARL_CONFIG_LOOP_CONF,
    app_host_resources.APP_HOST_EARL_CONFIG_ISS_HOOK_NOTIFY
]

_VERSION_PLACEHOLDER = "__ver__"


def _make_resource_choices():
    choices = []

    for resource in _RESOURCES:
        dynamic_suffix = ' (dynamic)' if resource.earl_dynamic_resource else ''
        choices.append((resource.earl_path + dynamic_suffix, str(resource)))

    return choices


def _make_default_choice():
    choices = []

    for resource in _RESOURCES:
        if resource.earl_dynamic_resource:
            choices.append(str(resource))

    return ' '.join(choices)


def _get_choosen_resource_names(ctx):
    names = set(ctx.get(EarlDataResourceChoiceParameter.name, '').split(' '))
    names.update(ParentResourcesParameter.resolve(ctx).keys())

    return names


def _set_data_version(resource_type, path, version):
    for name in resource_type.earl_versioned_files:
        fname = os.path.join(path, name)
        if os.path.exists(fname):
            f = fileinput.FileInput(fname, inplace=True, backup='.bak')
            for line in f:
                sys.stdout.write(line.replace(_VERSION_PLACEHOLDER, version))
            f.close()


class EarlDataResourceChoiceParameter(SandboxBoolGroupParameter):
    name = 'choose_resources'
    description = 'Choose resources to build'
    choices = _make_resource_choices()
    default_value = _make_default_choice()
    required = True


class BuildAppHostEarlData(ArcadiaTask):
    """Builds a bundle with earl data"""

    type = "BUILD_APP_HOST_EARL_DATA"

    execution_space = 1000

    input_parameters = [
        SandboxArcadiaUrlParameter,
        EarlDataResourceChoiceParameter
    ]

    def do_execute(self):
        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)
        assert parsed_url.revision

        base_url = url.split("@")[0] + "/apphost/daemons/earl"
        storage = os.path.abspath("earl")

        Arcadia.export(base_url + '@' + parsed_url.revision, storage)

        pkg_version = parsed_url.revision
        pkg_name = "earl_data_%s" % pkg_version

        resource_names = _get_choosen_resource_names(self.ctx)

        parent_resources = ParentResourcesParameter.resolve(self.ctx)

        logging.debug("Building resources: {} ({} requested by parent task)".format(resource_names, parent_resources))

        for resource_type in _RESOURCES:
            if resource_type.name not in resource_names:
                continue

            path = os.path.join(storage, resource_type.earl_path)

            if resource_type.earl_versioned_files:
                _set_data_version(resource_type, path, pkg_version)

            if resource_type.earl_tarball:
                tarball_name = os.path.basename(path) + '.tar.bz2'
                tarball_path = os.path.abspath(tarball_name)
                logging.info("Packing {} into {}".format(resource_type.earl_path, tarball_name))

                base_dir = os.path.dirname(path)
                relative_dir = os.path.basename(path)

                run_process(['tar', '-cpvjf', tarball_path, '-C', base_dir, relative_dir], log_prefix='tar')
                path = tarball_path

            name = os.path.join(pkg_name, resource_type.earl_path)
            parent_resource_id = parent_resources.get(resource_type.name)

            if parent_resource_id is not None:
                self.save_parent_task_resource(path, parent_resource_id)

            self.create_resource(
                description=name,
                resource_path=path,
                resource_type=resource_type,
                attributes=get_url_attributes(parsed_url),
                arch="any"
            )


__Task__ = BuildAppHostEarlData
