# -*- coding: utf-8 -*-

from sandbox import sdk2

import json
import logging

from sandbox.projects import resource_types as rt
from sandbox.projects.resource_types import releasers as rr

from sandbox.projects.app_host import resources as ah_rt
from sandbox.projects.common.nanny import nanny

from sandbox.common.errors import TaskFailure
from sandbox.common.types.resource import State

from sandbox.common.types.task import ReleaseStatus
from sandbox.sandboxsdk.paths import add_write_permissions_for_path, copy_path, make_folder


class ResourceHandler(object):

    def_release_types = [ReleaseStatus.STABLE, ReleaseStatus.UNSTABLE]

    def __init__(self, resource_type, release_types=def_release_types):
        self._type = str(resource_type)
        self._vertical = self._type.split("_")[-1]
        self._resource_type = resource_type
        self._release_types = release_types

    def __repr__(self):
        return repr((self._type, self._release_types))

    def resource_type(self):
        return self._resource_type

    def release_types(self):
        return self._release_types

    def validate(self, resource):
        return resource.state == State.READY and resource.released in self._release_types

    def handle(self, resource, result_path):
        data = sdk2.ResourceData(resource)
        resource_path = data.path

        suffix = "_unstable" if resource.released == ReleaseStatus.UNSTABLE else ""
        result_name = self._get_name(suffix)
        result = result_path.joinpath(result_name)
        copy_path(str(resource_path), str(result))

        return result

    def _get_name(self, suffix):
        pass


class ConfigBundleHandler(ResourceHandler):

    def _get_name(self, suffix):
        return "graphs_{}{}.tar.gz".format(self._vertical, suffix)


class BackendsConfigHandler(ResourceHandler):

    def _get_name(self, suffix):
        return "_backends_{}{}.json".format(self._vertical, suffix)


def _init_handlers():
    handlers = [
        # ConfigBundleHandler(rt.APP_HOST_CONFIG_BUNDLE_ATOM),
        # ConfigBundleHandler(rt.APP_HOST_CONFIG_BUNDLE_COMMON),
        # ConfigBundleHandler(rt.APP_HOST_CONFIG_BUNDLE_WEB),
        ConfigBundleHandler(rt.APP_HOST_CONFIG_BUNDLE_NEWS),
        # ConfigBundleHandler(rt.APP_HOST_CONFIG_BUNDLE_IMGS),
        # ConfigBundleHandler(rt.APP_HOST_CONFIG_BUNDLE_VIDEO),
        # ConfigBundleHandler(ah_rt.APP_HOST_CONFIG_BUNDLE_MAIL),
        # ConfigBundleHandler(ah_rt.APP_HOST_CONFIG_BUNDLE_MAILCORP),
        # ConfigBundleHandler(ah_rt.APP_HOST_CONFIG_BUNDLE_SHARED),

        # BackendsConfigHandler(ah_rt.APP_HOST_BACKENDS_CONFIG_ATOM),
        # BackendsConfigHandler(ah_rt.APP_HOST_BACKENDS_CONFIG_COMMON),
        # BackendsConfigHandler(ah_rt.APP_HOST_BACKENDS_CONFIG_WEB),
        BackendsConfigHandler(ah_rt.APP_HOST_BACKENDS_CONFIG_NEWS),
        # BackendsConfigHandler(ah_rt.APP_HOST_BACKENDS_CONFIG_IMGS),
        # BackendsConfigHandler(ah_rt.APP_HOST_BACKENDS_CONFIG_VIDEO),
        # BackendsConfigHandler(ah_rt.APP_HOST_BACKENDS_CONFIG_MAIL),
        # BackendsConfigHandler(ah_rt.APP_HOST_BACKENDS_CONFIG_MAILCORP),
        # BackendsConfigHandler(ah_rt.APP_HOST_BACKENDS_CONFIG_SHARED),
    ]
    return {h._type: h for h in handlers}


class AppHostEarlDataBundle(sdk2.Resource):
    """AppHost.Earl data bundle including graphs and backends for all verticals"""

    releasers = rr.apphost_releasers
    releasable = True

    contents = sdk2.Attributes.String("Contents", required=True)


class BuildAppHostEarlDataBundle(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Builds Earl data bundle including graphs and backends for all verticals"""

    resource_handlers = _init_handlers()

    class Requirements(sdk2.Task.Requirements):
        disk_space = 4 * 1024

    class Parameters(sdk2.Task.Parameters):

        kill_timeout = 5 * 60

        previous_bundle = sdk2.parameters.Resource(
            'Previous AppHostEarlDataBundle resource',
            resource_type=AppHostEarlDataBundle,
            required=False
        )

        new_releases = sdk2.parameters.List(
            'Ids of new releases to patch the bundle',
            sdk2.parameters.Integer,
            required=True
        )

    def on_enqueue(self):
        for res_id in self.Parameters.new_releases:
            resource = sdk2.Resource[res_id]

            handler = self.resource_handlers.get(str(resource.type))
            if not handler:
                raise TaskFailure("Unsupported resource type: {}".format(resource))

            if not handler.validate(resource):
                raise TaskFailure("Resource is not released: {}".format(resource))

    def on_execute(self):
        prev_resource = self.Parameters.previous_bundle

        bundle_path = self.path("earl_data_bundle")
        logging.info("Created resource dir {}".format(bundle_path))

        versions = {t: {rt: None for rt in ResourceHandler.def_release_types} for t in self.resource_handlers.iterkeys()}

        if prev_resource:
            logging.info("Using previous bundle: {}".format(prev_resource))

            resource_data = sdk2.ResourceData(prev_resource)
            copy_path(str(resource_data.path), str(bundle_path))
            add_write_permissions_for_path(str(bundle_path), recursively=True)

            versions.update(json.loads(prev_resource.contents))
        else:
            make_folder(bundle_path)

        resources = [sdk2.Resource[id] for id in self.Parameters.new_releases]

        # TODO: validation

        for r in resources:
            res_type = str(r.type)
            handler = self.resource_handlers.get(res_type)
            new_path = handler.handle(r, bundle_path)
            versions[res_type][r.released] = r.id
            logging.info("Patched bundle with resource {}, file {}".format(r, new_path))

        descr = "AppHost.Earl data bundle {}".format(self.id)
        result = AppHostEarlDataBundle(self, descr, str(bundle_path))
        result.contents = json.dumps(versions)
        sdk2.ResourceData(result).ready()
