import os.path

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxRadioParameter
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.common.reactor.reactor_api import ReactorApi

from sandbox.projects.app_host.common import UseFuseRepositoryParameter


_APP_HOST_EVENT_LOG_FILTER_ARCADIA_DIR = "apphost/tools/event_log_filter"
_APP_HOST_EVENT_LOG_FILTER_EXECUTABLE_NAME = "apphost_event_log_filter"

_APP_HOST_RELEASE_REACTOR_TOKEN_VAULT_NAME = "reactor-oauth-token"
_APP_HOST_RELEASE_REACTOR_TOKEN_VAULT_OWNER = "APP_HOST"


class AppHostEventLogFilterReactorID(SandboxRadioParameter):
    name = "reactor_id"
    description = "Release to specified Reactor Artifact"
    default_value = "8671"
    choices = [
        ("v1", "6669"),
        ("v2", "8671")
    ]


class BuildAppHostEventLogFilter(ArcadiaTask):
    """Builds apphost_event_log_filter binary for app_host"""

    type = "BUILD_APP_HOST_EVENT_LOG_FILTER"

    execution_space = 50 * 1024

    input_parameters = [AppHostEventLogFilterReactorID, UseFuseRepositoryParameter, SandboxArcadiaUrlParameter] + build_params.get_build_system_params()

    client_tags = ctc.Tag.Group.LINUX

    def do_execute(self):
        if self.ctx.get(UseFuseRepositoryParameter.name):
            with sdk.mount_arc_path(self.ctx[consts.ARCADIA_URL_KEY]) as arcadia_dir:
                self.build(arcadia_dir)
            return
        arcadia_dir = self.get_arcadia_src_dir()
        self.build(arcadia_dir)

    def build(self, arcadia_dir):
        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)
        assert parsed_url.revision

        # prepare build environment
        build_dir = self.abs_path("build")
        make_folder(build_dir)
        os.chdir(build_dir)

        sdk.do_build(
            self.ctx['build_system'],
            arcadia_dir,
            [_APP_HOST_EVENT_LOG_FILTER_ARCADIA_DIR],
            self.ctx['build_type'],
            clear_build=True,
            results_dir=build_dir
        )

        self.create_resource(
            description="APP_HOST_EVENT_LOG_FILTER_EXECUTABLE@{}".format(parsed_url.revision),
            resource_path=os.path.join(
                build_dir,
                _APP_HOST_EVENT_LOG_FILTER_ARCADIA_DIR,
                _APP_HOST_EVENT_LOG_FILTER_EXECUTABLE_NAME
            ),
            resource_type=resource_types.APP_HOST_EVENT_LOG_FILTER_EXECUTABLE,
            arch="any"
        )

    def on_release(self, additional_parameters):
        release_status = additional_parameters['release_status']
        if release_status == "stable":
            token = self.get_vault_data(owner_or_name=_APP_HOST_RELEASE_REACTOR_TOKEN_VAULT_OWNER,
                                        name=_APP_HOST_RELEASE_REACTOR_TOKEN_VAULT_NAME)
            assert token is not None
            ra = ReactorApi(token)

            resources = self.list_resources(resource_types.APP_HOST_EVENT_LOG_FILTER_EXECUTABLE)
            assert isinstance(resources, list) and len(resources) == 1
            ra.instantiateStringArtifact(int(self.ctx["reactor_id"]), resources[0].id)

        ArcadiaTask.on_release(self, additional_parameters)


__Task__ = BuildAppHostEventLogFilter
