import json
import logging
import os.path

from sandbox.projects import resource_types
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.paths import make_folder, copy_path
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxStringParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.app_host.options import ChooseVertical, get_verticals, GraphGenerator
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.app_host import resources as app_host_resources


def _make_url(base_url, subpath, rev):
    return "%s/%s@%s" % (base_url, subpath, rev)


class SpecificGraphs(SandboxStringParameter):
    default_value = ''
    name = 'specific_graphs'
    description = 'Specify graphs versions to load from .mapping files'


class BuildAppHostGraphNameMapping(nanny.ReleaseToNannyTask, ArcadiaTask):

    """Builds APP_HOST_GRAPH_NAME_MAPPING_${VERTICAL} resources"""

    type = "BUILD_APP_HOST_GRAPH_NAME_MAPPING"

    execution_space = 2000

    input_parameters = [ChooseVertical, SpecificGraphs, GraphGenerator, SandboxArcadiaUrlParameter]

    resource_map = {
        'ATOM': resource_types.APP_HOST_GRAPH_NAME_MAPPING_ATOM,
        'NEWS': resource_types.APP_HOST_GRAPH_NAME_MAPPING_NEWS,
        'WEB': resource_types.APP_HOST_GRAPH_NAME_MAPPING_WEB,
        'IMGS': resource_types.APP_HOST_GRAPH_NAME_MAPPING_IMGS,
        'VIDEO': resource_types.APP_HOST_GRAPH_NAME_MAPPING_VIDEO,
        'COMMON': resource_types.APP_HOST_GRAPH_NAME_MAPPING_COMMON,
        'MAIL': app_host_resources.APP_HOST_GRAPH_NAME_MAPPING_MAIL,
        'MAILCORP': app_host_resources.APP_HOST_GRAPH_NAME_MAPPING_MAILCORP,
        'SHARED': app_host_resources.APP_HOST_GRAPH_NAME_MAPPING_SHARED,
    }

    def do_execute(self):
        self.ctx["mapped_versions"] = {}

        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)

        assert parsed_url.revision

        base_url = url.split("@")[0] + "/apphost"

        graph_generator = self.sync_resource(GraphGenerator.get_resource_from_ctx(self.ctx))

        conf_dir = os.path.abspath("conf_dir")
        Arcadia.export(_make_url(base_url, "conf/graph_generator/vertical/", parsed_url.revision), conf_dir)

        generated_dir = os.path.abspath("generated")
        make_folder(generated_dir)

        os.chdir(generated_dir)

        for v in get_verticals(self.ctx):
            retries_key = "{}_retries".format(v)
            self.ctx[retries_key] = 0
            error = False

            args = [
                graph_generator, '-vvv'
            ]
            extended_args = []
            if self.ctx.get(SpecificGraphs.name):
                command = 'selective_mapping'
                extended_args = ['-o', generated_dir, '-r', self.ctx.get(SpecificGraphs.name)]
            else:
                command = 'mapping'

            args.extend([command, v, '-g', conf_dir])
            args.extend(extended_args)
            while self.ctx.get(retries_key) <= 4:

                result = run_process(
                    args,
                    shell=True,
                    check=True,
                    log_prefix="graph_generator_mapping"
                )

                if result.returncode != 0:
                    self.ctx.get[retries_key] += 1
                    error = True
                    result_path = os.path.join(generated_dir, "mapping.err")
                    copy_path(result.stderr_path, result_path)
                    self.create_resource(
                        description="mapping stderr {}".format(self.ctx.get("retries")),
                        resource_path=result_path,
                        resource_type=resource_types.OTHER_RESOURCE,
                        arch="any"
                    )
                    result_path = os.path.join(generated_dir, "mapping.out")
                    copy_path(result.stdout_path, result_path)
                    self.create_resource(
                        description="mapping stdout {}".format(self.ctx.get("retries")),
                        resource_path=result_path,
                        resource_type=resource_types.OTHER_RESOURCE,
                        arch="any"
                    )
                else:
                    error = False
                    break

            if not error:
                if self.ctx.get(SpecificGraphs.name):
                    self.create_resource(
                        description="{} graph_name_mapping@{}".format(v, parsed_url.revision),
                        resource_path=os.path.join(generated_dir, "graph_name_mapping.{}.txt".format(v)),
                        resource_type=BuildAppHostGraphNameMapping.resource_map[v],
                        arch="any"
                    )

                    with open(os.path.join(generated_dir, "result.{}.json".format(v))) as fp:
                        self.ctx["mapped_versions"][v] = json.load(fp)

                else:
                    self.create_resource(
                        description="graph_name_mapping@{}".format(parsed_url.revision),
                        resource_path=os.path.join(conf_dir, v, "graph_name_mapping.txt"),
                        resource_type=BuildAppHostGraphNameMapping.resource_map[v],
                        arch="any"
                    )
            else:
                raise SandboxTaskFailureError("Failed to validate mapping file")

    def on_release(self, parameters):
        logging.debug("on_release {}".format(parameters))

        if self.ctx.get(SpecificGraphs.name) and parameters["release_status"] == "stable":
            url = self.ctx[consts.ARCADIA_URL_KEY]
            graph_generator = self.sync_resource(GraphGenerator.get_resource_from_ctx(self.ctx))
            vertical_dir = self.abs_path("vertical")
            results_dir = self.abs_path("results")

            make_folder(vertical_dir)
            make_folder(results_dir)

            Arcadia.checkout(
                "%s/%s" % (
                    url.split("@")[0], "apphost/conf/graph_generator/vertical",
                ),
                vertical_dir
            )

            for v in get_verticals(self.ctx):
                update_retries_key = "{}_update_retries".format(v)
                self.ctx[update_retries_key] = 0
                vertical_dir_path = os.path.join(vertical_dir, v)

                logging.debug("mapped_versions: ".format(self.ctx["mapped_versions"][v]))
                with open(os.path.join(results_dir, "result.{}.json".format(v)), "w") as fp:
                    json.dump(self.ctx["mapped_versions"][v], fp)

                args = [
                    graph_generator, '-vvv'
                ]

                command = 'update_mapping'
                extended_args = ['-r', results_dir]
                args.extend([command, v, '-g', vertical_dir])
                args.extend(extended_args)

                update_error = None

                while self.ctx.get(update_retries_key) <= 0:
                    result = run_process(
                        args,
                        shell=True,
                        check=True,
                        log_prefix="graph_generator_update_mapping"
                    )

                    if result.returncode != 0:
                        self.ctx[update_retries_key] += 1
                        update_error = "update_mapping failed, code=%s" % result.returncode
                    else:
                        update_error = None
                        break

                if update_error is not None:
                    raise Exception("retry count exceeded, last error: %s" % update_error)

                logging.debug("Diff:")
                logging.debug(Arcadia.diff(vertical_dir_path))

                message = "update_mapping sync, releaser:{releaser} sb:{id} SKIP_REVIEW".format(
                    id=self.id,
                    releaser=parameters["releaser"]
                )
                logging.debug(message)
                Arcadia.commit(
                    vertical_dir_path,
                    message,
                    "robot-ah-releases"
                )
        else:
            nanny.ReleaseToNannyTask.on_release(self, parameters)


__Task__ = BuildAppHostGraphNameMapping
