import glob
import os.path
import os

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.app_host import parse_release_version, gen_symdump, SymbolsDumper
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.search.gdb as gdb
import sandbox.projects.common.constants as consts
import sandbox.projects.common.build.parameters as build_params
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.app_host.options import ChooseVertical, get_verticals
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.app_host.common import UseFuseRepositoryParameter


_HTTP_ADAPTER_PATH = "apphost/daemons/http_adapter/http_adapter"


def _add_http_adapter(build_dir, package_dir):
    copy_path(
        os.path.join(build_dir, _HTTP_ADAPTER_PATH),
        os.path.join(package_dir, "http_adapter")
    )


def _add_tool(build_dir, package_dir, name):
    tool_path = os.path.join(package_dir, name)

    copy_path(
        os.path.join(build_dir, "apphost/tools/%s/%s" % (name, name)),
        tool_path
    )


_TOOLS = [
    "event_log_dump"
]


class BuildAppHostHttpAdapterBundle(nanny.ReleaseToNannyTask, ArcadiaTask):

    """Builds a bundle with app_host and some related tools"""

    type = "BUILD_APP_HOST_HTTP_ADAPTER_BUNDLE"

    execution_space = 100000

    input_parameters = [ChooseVertical] + build_params.get_build_system_params() + \
                       [SandboxArcadiaUrlParameter, SymbolsDumper] + \
                       [UseFuseRepositoryParameter]

    resources_map = {
        "ATOM": resource_types.APP_HOST_HTTP_ADAPTER_BUNDLE_ATOM,
        "WEB": resource_types.APP_HOST_HTTP_ADAPTER_BUNDLE_WEB,
        "NEWS": resource_types.APP_HOST_HTTP_ADAPTER_BUNDLE_NEWS,
        "IMGS": resource_types.APP_HOST_HTTP_ADAPTER_BUNDLE_IMGS,
        "VIDEO": resource_types.APP_HOST_HTTP_ADAPTER_BUNDLE_VIDEO,
        "COMMON": resource_types.APP_HOST_HTTP_ADAPTER_BUNDLE_COMMON,
        "MAIL": app_host_resources.APP_HOST_HTTP_ADAPTER_BUNDLE_MAIL,
        "MAILCORP": app_host_resources.APP_HOST_HTTP_ADAPTER_BUNDLE_MAILCORP,
        "SHARED": app_host_resources.APP_HOST_HTTP_ADAPTER_BUNDLE_SHARED,
    }

    client_tags = ctc.Tag.GENERIC & ctc.Tag.Group.LINUX

    def do_execute(self):
        if self.ctx.get(UseFuseRepositoryParameter.name):
            with sdk.mount_arc_path(self.ctx[consts.ARCADIA_URL_KEY]) as arcadia_dir:
                self.build(arcadia_dir)
            return
        arcadia_dir = self.get_arcadia_src_dir()
        self.build(arcadia_dir)

    def build(self, arcadia_dir):
        gdb.append_to_stable_only(self)
        # prepare build environment
        build_dir = self.abs_path("build")
        make_folder(build_dir)
        os.chdir(build_dir)

        sdk.do_build(
            self.ctx['build_system'],
            arcadia_dir,
            ["apphost"],
            self.ctx['build_type'],
            clear_build=True,
            results_dir=build_dir
        )

        # create package
        url = parse_release_version(self.ctx[consts.ARCADIA_URL_KEY])
        pkg_name = "app_host_http_adapter_bundle_%s" % url

        package_dir = os.path.join(build_dir, pkg_name)
        os.chdir(build_dir)
        make_folder(package_dir)

        _add_http_adapter(build_dir, package_dir)

        for tool in _TOOLS:
            _add_tool(build_dir, package_dir, tool)

        gen_symdump(self, package_dir, build_dir, _HTTP_ADAPTER_PATH)

        run_process("tar czvf %s.tar.gz %s" % (pkg_name, pkg_name), shell=True)

        tar = glob.glob(os.path.join(build_dir, "*.tar.gz"))[-1]

        # save created package as resource
        resource_dir = self.abs_path()
        copy_path(tar, resource_dir)

        tar_name = os.path.basename(tar)

        for vertical in get_verticals(self.ctx):
            new_path = os.path.join(resource_dir, tar_name + vertical + ".tar.gz")
            copy_path(os.path.join(resource_dir, tar_name), new_path)
            self.create_resource(
                description="app_host for {} from {}".format(vertical, str(Arcadia.parse_url(self.ctx[consts.ARCADIA_URL_KEY]))),
                resource_path=new_path,
                resource_type=self.resources_map[vertical],
                arch=self.arch
            )


__Task__ = BuildAppHostHttpAdapterBundle
