import json
import os.path
import logging

from sandbox.projects import resource_types

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.paths import make_folder, copy_path
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.app_host.options import get_verticals, GraphGenerator, ChooseVertical, PVA
from sandbox.projects.app_host.BuildAppHostConfigBundle import BackendsConfigTask
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from copy import deepcopy


class BuildServiceResources(SandboxBoolParameter):
    name = 'build_service_resources'
    description = 'Build cross service resources (loop.conf etc)'
    default_value = True
    required = True


def _make_url(base_url, subpath, rev):
    return "%s/%s@%s" % (base_url, subpath, rev)


class BuildAppHostHttpAdapterConfigBundle(nanny.ReleaseToNannyTask, ArcadiaTask):

    """Builds a bundle with app_host http_adapter configs"""

    type = "BUILD_APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE"

    execution_space = 1000

    input_parameters = [
        ChooseVertical,
        BackendsConfigTask,
        BuildServiceResources,
        SandboxArcadiaUrlParameter,
        GraphGenerator,
        PVA,
    ]

    global_configs = {
        "iss_hook_notify.py": resource_types.APP_HOST_HTTP_ADAPTER_ISS_HOOK_NOTIFY,
    }

    global_pva_configs = {
        "production_http_adapter_loop.conf": resource_types.APP_HOST_HTTP_ADAPTER_LOOP_CONF,
        "production_http_adapter_loop_isolated.conf": app_host_resources.APP_HOST_HTTP_ADAPTER_LOOP_ISOLATED_CONF,
        "testing_http_adapter_loop.conf": app_host_resources.APP_HOST_HTTP_ADAPTER_TESTING_LOOP_CONF,
        "testing_http_adapter_loop_isolated.conf": app_host_resources.APP_HOST_HTTP_ADAPTER_TESTING_LOOP_ISOLATED_CONF,
    }

    pva_configs = {
        'production_http_adapter_loop.conf': {
            'SHARED': app_host_resources.APP_HOST_HTTP_ADAPTER_LOOP_CONF_SHARED,
        },
        'production_http_adapter_loop_isolated.conf': {
            'SHARED': app_host_resources.APP_HOST_HTTP_ADAPTER_LOOP_ISOLATED_CONF_SHARED,
            'COMMON': app_host_resources.APP_HOST_HTTP_ADAPTER_LOOP_ISOLATED_CONF_COMMON,
            'MAIL': app_host_resources.APP_HOST_HTTP_ADAPTER_LOOP_ISOLATED_CONF_MAIL,
            'MAILCORP': app_host_resources.APP_HOST_HTTP_ADAPTER_LOOP_ISOLATED_CONF_MAILCORP,
        },
        'testing_http_adapter_loop_isolated.conf': {
            'SHARED': app_host_resources.APP_HOST_HTTP_ADAPTER_TESTING_LOOP_ISOLATED_CONF_SHARED,
            'COMMON': app_host_resources.APP_HOST_HTTP_ADAPTER_TESTING_LOOP_ISOLATED_CONF_COMMON,
            'MAIL': app_host_resources.APP_HOST_HTTP_ADAPTER_TESTING_LOOP_ISOLATED_CONF_MAIL,
            'MAILCORP': app_host_resources.APP_HOST_HTTP_ADAPTER_TESTING_LOOP_ISOLATED_CONF_MAILCORP,
        }
    }

    resource_map = {
        "ATOM": resource_types.APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE_ATOM,
        "WEB": resource_types.APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE_WEB,
        "NEWS": resource_types.APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE_NEWS,
        "IMGS": resource_types.APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE_IMGS,
        "VIDEO": resource_types.APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE_VIDEO,
        "COMMON": resource_types.APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE_COMMON,
        "MAIL": app_host_resources.APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE_MAIL,
        "MAILCORP": app_host_resources.APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE_MAILCORP,
        "SHARED": app_host_resources.APP_HOST_HTTP_ADAPTER_CONFIG_BUNDLE_SHARED,
    }

    push_client_fork_set = {
        'SHARED',
    }

    nginx_set = {
        'COMMON', 'MAIL', 'MAILCORP'
    }

    def __setup_paths(self, base_url, url):
        self.__conf_dir = os.path.abspath("conf_dir")
        Arcadia.export(_make_url(base_url, "conf/http_adapter", url.revision), self.__conf_dir)

        self.__pva_conf_dir = os.path.abspath("pva_conf_dir")
        Arcadia.export(_make_url(base_url, "conf/pva", url.revision), self.__pva_conf_dir)

        self.__nginx_conf_dir = os.path.abspath("nginx_conf_dir")
        Arcadia.export(_make_url(base_url, "conf/nginx", url.revision), self.__nginx_conf_dir)

    def __run_pva(self, config_name, vertical, output):
        pva = self.sync_resource(self.__get_pva())
        pva_config = os.path.join(self.__pva_conf_dir, "config.json")

        logging.info("Building config {} for vertical {} into {} with PVA"
                     .format(config_name, vertical, output))

        logging.debug("PVA Config: {}".format(json.load(open(pva_config))))

        try:
            run_process([pva, pva_config, vertical, config_name, output],
                        check=True,
                        outputs_to_one_file=False,
                        log_prefix="pva")
        except Exception as e:
            logging.error("Unexpected PVA error: ", e)
            raise

    def init_or_decrement_subtask_retry_count(self):
        r = self.ctx.get("backends_config_task_retries")

        if r == 0:
            raise SandboxTaskFailureError("child task failed, retry limit exceeded")

        self.ctx["backends_config_task_retries"] = 2 if r is None else r - 1

    def __get_pva(self):
        if not hasattr(self, '__pva'):
            self.__pva = PVA.get_resource_from_ctx(self.ctx)
        return self.__pva

    def do_execute(self):
        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)
        assert parsed_url.revision

        base_url = url.split("@")[0] + "/apphost"
        self.__setup_paths(base_url, parsed_url)

        self.ctx[GraphGenerator.name] = GraphGenerator.get_resource_from_ctx(self.ctx)
        graph_generator = self.sync_resource(self.ctx[GraphGenerator.name])

        if not self.ctx.get(BackendsConfigTask.name):
            context = deepcopy(self.ctx)
            context["choose_app_to_build"] = "http_adapter"
            subtask = self.create_subtask(
                task_type="BUILD_APP_HOST_BACKENDS_CONFIG",
                description=self.descr + " (build for http_adapter configs)",
                input_parameters=context,
                priority=self.priority,
                inherit_notifications=True
            )
            self.ctx[BackendsConfigTask.name] = subtask.id
            self.init_or_decrement_subtask_retry_count()
            logging.info(
                "Starting subtask BUILD_APP_HOST_BACKENDS_CONFIG {}".format(
                    subtask
                )
            )
            self.wait_tasks(
                [subtask],
                (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED,
                 self.Status.EXCEPTION),
                True
            )
        task_id = self.ctx.get(BackendsConfigTask.name)

        assert self.ctx.get(BackendsConfigTask.name)

        resources = channel.sandbox.list_resources(resource_type="APP_HOST_BACKENDS_CONFIG",
                                                   task_id=task_id,
                                                   status="READY")

        resource_ids = [(res.id, res.attributes["vertical"]) for res in resources]
        msgs = []

        self.ctx["verticals"] = [res.attributes["vertical"] for res in resources]

        for resource_id, vertical in resource_ids:
            # create package
            pkg_version = parsed_url.revision
            pkg_name = "app_host_http_adapter_config_bundle_{}_{}".format(vertical, pkg_version)
            pkg_path = os.path.abspath(pkg_name)

            make_folder(pkg_name)
            os.chdir(pkg_name)

            backends_list = self.sync_resource(resource_id)

            result = run_process(
                [
                    graph_generator,
                    'http_adapter', vertical,
                    '-b', backends_list,
                    '-c', self.__conf_dir,
                    '-o', self.abs_path(pkg_name)
                ],
                shell=True,
                check=True,
                log_prefix="graph_generator"
            )

            stderr = open(result.stderr_path or result.stdout_path).readlines()
            if stderr:
                msgs.append(vertical + '\n' + '\n'.join(stderr))

            push_client_dstpath = os.path.join(pkg_path, 'push_client.yml')
            self.__run_pva("production_http_adapter_push_client.yml", vertical, push_client_dstpath)

            if vertical in self.push_client_fork_set:
                push_client_fork_dstpath = os.path.join(pkg_path, 'push_client_fork.yml')
                self.__run_pva("production_app_host_push_client_fork.yml", vertical, push_client_fork_dstpath)

            if vertical in self.nginx_set:
                for f in ['dkjson', 'init', 'worker', 'metrics']:
                    from_path = os.path.join(self.__nginx_conf_dir, "{}.lua".format(f))
                    to_path = os.path.join(pkg_path, "{}.lua".format(f))
                    copy_path(from_path, to_path)

            os.chdir("..")

            tar = pkg_path + ".tar.gz"
            run_process("tar czvf %s %s" % (tar, pkg_name), shell=True)

            res_id = self.create_resource(
                description=pkg_name,
                resource_path=tar,
                resource_type=self.resource_map[vertical],
                arch="any"
            )

            self.mark_resource_ready(res_id.id)

        # TODO: get rid of copy-paste
        if self.ctx.get(BuildServiceResources.name):
            generated_dir = os.path.abspath("generated")
            make_folder(generated_dir)
            os.chdir(generated_dir)

            for config_name, config_type in self.global_pva_configs.items():
                output_path = os.path.join(generated_dir, config_name)

                self.__run_pva(config_name, "", output_path)

                self.create_resource(
                    description="{}@{}".format(config_name, parsed_url.revision),
                    resource_path=output_path,
                    resource_type=config_type,
                    arch="any")

            for config_name, vertical_settings in self.pva_configs.items():  # no config_map here
                for vertical in get_verticals(self.ctx):
                    config_type = vertical_settings.get(vertical)
                    if config_type is not None:
                        pos = config_name.rfind('.')
                        if pos != -1:
                            config_file_name = config_name[:pos]
                            config_file_ext = config_name[pos:]
                        else:
                            config_file_name = config_name
                            config_file_ext = ''

                        output_path = os.path.join(generated_dir, "{}.{}{}".format(config_file_name, vertical, config_file_ext))

                        self.__run_pva(config_name, vertical, output_path)

                        self.create_resource(
                            description="{}@{}".format(config_name, parsed_url.revision),
                            resource_path=output_path,
                            resource_type=config_type,
                            arch="any")

            for config_name, config_type in self.global_configs.iteritems():
                self.create_resource(
                    description="{}@{}".format(config_name, parsed_url.revision),
                    resource_path=os.path.join(self.__conf_dir, config_name),
                    resource_type=config_type,
                    arch="any")

            res_id = self.create_resource(
                description="nginx.conf template",
                resource_path=os.path.join(self.__nginx_conf_dir, "nginx.conf"),
                resource_type=app_host_resources.APP_HOST_HTTP_ADAPTER_NGINX_CONF_TMPL,
                arch="any"
            )

            self.mark_resource_ready(res_id.id)


__Task__ = BuildAppHostHttpAdapterConfigBundle
