import logging
import os

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.app_host.BuildAppHostBundle import _APP_HOST_RELEASE_VAULT_OWNER, _APP_HOST_RELEASE_VAULT_NAME
from sandbox.projects.common import constants as consts
from sandbox.projects.common.app_host.options import ChooseVertical, get_verticals, GraphGenerator
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxSvnTemporaryError
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxBoolParameter
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia


class CommitResults(SandboxBoolParameter):
    name = 'commit'
    description = 'Commit results to repository?'
    default_value = True


class NoQuentao(SandboxBoolParameter):
    name = 'no_quentao'
    description = 'Build without quentao'
    default_value = False


def _make_url(base_url, subpath, rev):
    return "%s/%s@%s" % (base_url, subpath, rev)


class BuildAppHostMonitoring(ArcadiaTask):
    """Builds a bundle with app_host config and graphs"""

    type = "BUILD_APP_HOST_MONITORING"

    execution_space = 4000

    input_parameters = [
        ChooseVertical,
        GraphGenerator,
        SandboxArcadiaUrlParameter,
        CommitResults,
        NoQuentao
    ]

    environment = (environments.PipEnvironment('funcy', '1.10.2', use_wheel=True),)

    client_tags = ctc.Tag.Group.LINUX

    def __setup_paths(self, base_url, rev):
        self.__verticals_dir = os.path.abspath("verticals_dir")
        self.__quentao_dir = os.path.abspath("golovan_panel")
        self.__temp_dir = os.path.abspath("temp_dir")
        os.makedirs(self.__temp_dir)

        Arcadia.export(_make_url(base_url, "web/app_host/conf/graph_generator/vertical", rev), self.__verticals_dir)
        for root, dirs, files in os.walk(self.__verticals_dir):
            for name in files:
                logging.debug("file: {}".format(os.path.join(root, name)))
        if not self.ctx.get(NoQuentao):
            Arcadia.checkout(_make_url(base_url, "junk/sereglond/golovan/src/golovan_panel", rev), self.__quentao_dir)

    def do_execute(self):
        # prepare build environment
        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)

        assert parsed_url.revision

        base_url = url.split("@")[0]

        self.__setup_paths(base_url, parsed_url.revision)
        graph_generator = self.sync_resource(GraphGenerator.get_resource_from_ctx(self.ctx))

        self.ctx["retries"] = 0
        monitoring_failed = False
        logging.debug("NoQuentao {}".format(self.ctx.get(NoQuentao.name)))
        while self.ctx.get("retries") <= 4:
            logging.debug("Launch retry %s" % self.ctx.get('retries'))
            if self.ctx.get(NoQuentao.name):
                result = run_process(
                    [
                        graph_generator,
                        '-vvv',
                        'monitoring',
                        '-g', self.__verticals_dir,
                        '-t', self.__temp_dir,
                        '-d', '_graph_cleanup.txt',
                        '--upload-to-ambry',
                    ] + get_verticals(self.ctx),
                    shell=True,
                    check=False,
                    log_prefix="graph_generator_monitoring"
                )
            else:
                result = run_process(
                    [
                        graph_generator,
                        '-vvv',
                        'monitoringng',
                        '-g', self.__verticals_dir,
                        '-t', self.__temp_dir,
                        '-q', self.__quentao_dir,
                        '--upload-to-ambry',
                    ] + get_verticals(self.ctx),
                    shell=True,
                    check=False,
                    log_prefix="graph_generator_monitoringng"
                )

            if self.ctx.get(NoQuentao.name):
                for f in ["alerts.json", "menu.json"]:
                    p = os.path.join(self.__temp_dir, f)
                    logging.debug("p: {}".format(p))
                    if os.path.isfile(p):
                        self.create_resource(
                            description=f,
                            resource_path=p,
                            resource_type=resource_types.OTHER_RESOURCE,
                            arch="any"
                        )

            if result.returncode != 0:
                self.ctx["retries"] += 1
                if result.stdout_path:
                    stdout_res = os.path.join(self.abs_path('monitoring.stdout.{}.txt'.format(self.ctx["retries"])))
                    copy_path(result.stdout_path, stdout_res)

                    self.create_resource(
                        description=stdout_res,
                        resource_path=stdout_res,
                        resource_type=resource_types.OTHER_RESOURCE,
                        arch="any"
                    )
                if result.stderr_path:
                    stderr_res = os.path.join(self.abs_path('monitoring.stderr.{}.txt'.format(self.ctx["retries"])))
                    copy_path(result.stderr_path, stderr_res)
                    self.create_resource(
                        description=stderr_res,
                        resource_path=stderr_res,
                        resource_type=resource_types.OTHER_RESOURCE,
                        arch="any"
                    )
                monitoring_failed = True
            else:
                monitoring_failed = False
                break

        if monitoring_failed:
            tar = self.abs_path("temp_dir.tgz")
            run_process("tar czvf {} {}".format(tar, self.__temp_dir), shell=True, check=True, log_prefix="tar")

            self.create_resource(
                description=tar,
                resource_path=tar,
                resource_type=resource_types.OTHER_RESOURCE,
                arch="any"
            )
            raise SandboxTaskFailureError("Failed to build monitoring")
        else:
            if not self.ctx.get(NoQuentao.name):
                logging.debug("Launch quentao test")
                self.ctx["quentao_retries"] = 0
                quentao_failed = False
                while self.ctx.get("quentao_retries") <= 4:
                    os.chdir(self.__quentao_dir)
                    result = run_process(
                        [
                            'python',
                            os.path.join(self.__quentao_dir, 'base.py'),
                            '-s', '-d'
                        ],
                        shell=True,
                        check=False,
                        log_prefix="quentao"
                    )

                    if result.returncode != 0:
                        if result.stdout_path:
                            stdout_res = os.path.join(self.abs_path('quentao.stdout.{}.txt'.format(self.ctx["retries"])))
                            copy_path(result.stdout_path, stdout_res)

                            self.create_resource(
                                description=stdout_res,
                                resource_path=stdout_res,
                                resource_type=resource_types.OTHER_RESOURCE,
                                arch="any"
                            )
                        if result.stderr_path:
                            stderr_res = os.path.join(self.abs_path('quentao.stderr.{}.txt'.format(self.ctx["retries"])))
                            copy_path(result.stderr_path, stderr_res)
                            self.create_resource(
                                description=stderr_res,
                                resource_path=stderr_res,
                                resource_type=resource_types.OTHER_RESOURCE,
                                arch="any"
                            )
                        quentao_failed = True
                        self.ctx["quentao_retries"] += 1
                    else:
                        quentao_failed = False
                        break

                logging.debug("Diff:")
                logging.debug(Arcadia.diff(self.__quentao_dir))

                if quentao_failed:
                    raise SandboxTaskFailureError("Quentao build failed")
                else:
                    logging.debug(
                        "Commiting using {}@{}".format(_APP_HOST_RELEASE_VAULT_OWNER, _APP_HOST_RELEASE_VAULT_NAME))

                    if Arcadia.diff(self.__quentao_dir) and self.ctx.get(CommitResults.name):
                        self.ctx["svn_retries"] = 0
                        while self.ctx.get("svn_retries") <= 4:
                            try:
                                with ssh.Key(self, _APP_HOST_RELEASE_VAULT_OWNER, _APP_HOST_RELEASE_VAULT_NAME):
                                    Arcadia.update(self.__quentao_dir)
                                    Arcadia.commit(self.__quentao_dir, "apphost gen (auto)", _APP_HOST_RELEASE_VAULT_OWNER)
                            except SandboxSvnTemporaryError as e:
                                logging.debug(e)
                                self.ctx["svn_retries"] += 1
                                if self.ctx["svn_retries"] > 4:
                                    raise SandboxTaskFailureError(
                                        "Failed to commit to quentao: {}, {}".format(self.ctx.get("svn_retries"), e))
                            else:
                                logging.debug("Succesfully committed on {} try".format(self.ctx.get("svn_retries")))
                                break
                    else:
                        logging.debug("Nothing to commit")


__Task__ = BuildAppHostMonitoring
