# -*- coding: utf-8 -*-
import glob
import logging
import os.path

import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.parameters as sb_params
import sandbox.sdk2.parameters as parameters
import sandbox.projects.common.constants as consts
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.common.search.gdb as gdb
import sandbox.projects.common.utils as utils

from sandbox.projects import resource_types
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.common.app_host import parse_release_version, gen_symdump, SymbolsDumper
from sandbox.projects.common.app_host.options import ChooseVertical, get_verticals
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.build import dependency_banner
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox import sdk2


_SRC_SETUP_PATH = "web/src_setup/daemons/src_setup/src_setup"


class BuildDefaultSrcSetup(SandboxBoolParameter):
    name = 'build_default_src_setup'
    description = 'Build default src_setup?'
    default_value = False


class Checkout(SandboxBoolParameter):
    name = "checkout"
    description = "Set checkout option for build"
    default = False


class ClearBuild(sb_params.SandboxBoolParameter):
    name = consts.CLEAR_BUILD_KEY
    description = 'Clear build'
    default_value = False
    group = 'Base build params'


class BuildSystem(parameters.String):
    name = consts.BUILD_SYSTEM_KEY
    description = 'Build system'
    required = True
    default_value = consts.SEMI_DISTBUILD_BUILD_SYSTEM
    group = 'Build system params'
    choices = [
        ('Ya', consts.YMAKE_BUILD_SYSTEM),
        # TODO: remove after experiments
        ('Ya force', consts.YA_MAKE_FORCE_BUILD_SYSTEM),
        ('semi-distbuild', consts.SEMI_DISTBUILD_BUILD_SYSTEM),
        ('distbuild', consts.DISTBUILD_BUILD_SYSTEM),
    ]
    sub_fields = {
        consts.YMAKE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
        consts.YA_MAKE_FORCE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
    }


class ArcTokenOwner(parameters.String):
    name = 'arc_token_owner'
    description = 'ARC_TOKEN owner'
    required = True
    group = 'Build system params'


@rm_notify.notify2()
class BuildAppHostSrcSetup(nanny.ReleaseToNannyTask, ArcadiaTask):

    """
    Builds a bundle with src_setup binary (app_host servant that hosts src setups and proxies) and some related stuff
    """

    type = "BUILD_APP_HOST_SRC_SETUP"

    # FIXME(mvel): do exec space selection for trunk/branches
    execution_space = 100000

    input_parameters = [
        BuildDefaultSrcSetup,
        ChooseVertical,
        Checkout,
        SandboxArcadiaUrlParameter,
        SymbolsDumper,
        ClearBuild,
        BuildSystem,
        build_params.ThinLTO,
        ArcTokenOwner,
    ] + (
        [k for k in build_params.get_build_system_params() if k.name != "clear_build" and k.name != "build_system"] +
        build_params.get_check_deps_params()
    )

    environment = [task_env.TaskRequirements.startrek_client]
    client_tags = task_env.TaskTags.startrek_client & ctc.Tag.GENERIC & ctc.Tag.Group.LINUX

    resources_map = {
        "ATOM": resource_types.APP_HOST_SRC_SETUP_BUNDLE_ATOM,
        "WEB": resource_types.APP_HOST_SRC_SETUP_BUNDLE_WEB,
        "NEWS": resource_types.APP_HOST_SRC_SETUP_BUNDLE_NEWS,
        "IMGS": resource_types.APP_HOST_SRC_SETUP_BUNDLE_IMGS,
        "VIDEO": resource_types.APP_HOST_SRC_SETUP_BUNDLE_VIDEO,
        "VIDEO_HOSTING": app_host_resources.APP_HOST_SRC_SETUP_BUNDLE_VIDEO_HOSTING,
        "COMMON": app_host_resources.APP_HOST_SRC_SETUP_BUNDLE_COMMON,
        "MAIL": app_host_resources.APP_HOST_SRC_SETUP_BUNDLE_MAIL,
        "MAILCORP": app_host_resources.APP_HOST_SRC_SETUP_BUNDLE_MAILCORP,
        "SHARED": app_host_resources.APP_HOST_SRC_SETUP_BUNDLE_SHARED,
        "YDO": app_host_resources.APP_HOST_SRC_SETUP_BUNDLE_YDO,
        "DEFAULT": resource_types.APP_HOST_SRC_SETUP_BUNDLE,
        "MUSIC": app_host_resources.APP_HOST_SRC_SETUP_BUNDLE_MUSIC,
        "GOODS": app_host_resources.APP_HOST_SRC_SETUP_BUNDLE_GOODS,
    }

    def on_enqueue(self):
        self.ctx[rm_const.COMPONENT_CTX_KEY] = "src_setup"
        ArcadiaTask.on_enqueue(self)

    @staticmethod
    def _get_src_setup_path(build_dir, vertical):
        if vertical != "DEFAULT":
            return os.path.join(build_dir, "web/src_setup/daemons/vertical/{}/src_setup/src_setup".format(vertical))
        else:
            return os.path.join(build_dir, _SRC_SETUP_PATH)

    def _build_vertical(self, vertical="DEFAULT"):
        build_dir = self.abs_path(os.path.join("build", vertical))

        make_folder(build_dir)
        os.chdir(build_dir)

        build_type = consts.RELEASE_BUILD_TYPE

        targets = {"web/src_setup/tools/event_log_dump"}

        if vertical != "DEFAULT":
            targets.add("web/src_setup/daemons/vertical/{}/src_setup".format(vertical))
        else:
            targets.add("web/src_setup/daemons/src_setup")

        checkout = utils.get_or_default(self.ctx, Checkout)
        token = sdk2.Vault.data(utils.get_or_default(self.ctx, ArcTokenOwner), name='ARC_TOKEN')
        with sdk.mount_arc_path(
            self.ctx[consts.ARCADIA_URL_KEY],
            use_arc_instead_of_aapi=True,
            fallback=True,
            arc_oauth_token=token
        ) as arcadia_dir:
            logging.info("Arcadia repo path: %s", arcadia_dir)

            if not checkout:
                # we have arcadia and can calculate dependencies
                self._ban_dependencies(arcadia_dir, targets, vertical)

            sdk.do_build(
                utils.get_or_default(self.ctx, BuildSystem),
                arcadia_dir,
                list(targets),
                build_type,
                clear_build=utils.get_or_default(self.ctx, ClearBuild),
                results_dir=build_dir,
                checkout=checkout,
                thinlto=utils.get_or_default(self.ctx, build_params.ThinLTO),
            )

        return build_dir

    def _ban_dependencies(self, arcadia_dir, targets, vertical):
        if not targets:
            return

        deps_resource = dependency_banner.create_resources(
            self,
            output_file_name="deps_{}.txt".format(vertical),
            description=vertical,
        )
        dependency_banner.check_dependencies(self, arcadia_dir, targets, deps_resource.path)

    def do_execute(self):
        pkg_name = "src_setup_bundle_%s" % parse_release_version(self.ctx[consts.ARCADIA_URL_KEY])

        gdb.append_to_stable_only(self)

        verticals = get_verticals(self.ctx)
        if self.ctx.get(BuildDefaultSrcSetup.name):
            verticals.append("DEFAULT")

        for vertical in verticals:

            build_dir = self._build_vertical(vertical)

            package_dir = os.path.join(build_dir, pkg_name)

            os.chdir(build_dir)
            make_folder(package_dir)

            logging.debug(build_dir)
            logging.debug(package_dir)
            logging.debug(pkg_name)
            logging.debug(self._get_src_setup_path(build_dir, vertical))
            logging.debug(os.path.join(package_dir, "src_setup"))

            copy_path(
                os.path.join(self._get_src_setup_path(build_dir, vertical)),
                os.path.join(package_dir, "src_setup")
            )
            copy_path(
                os.path.join(build_dir, "web/src_setup/tools/event_log_dump/event_log_dump"),
                os.path.join(package_dir, "event_log_dump")
            )

            gen_symdump(self, package_dir, build_dir, self._get_src_setup_path(build_dir, vertical))

            run_process("tar czvf %s.tar.gz %s" % (pkg_name, pkg_name), shell=True)

            tar = glob.glob(os.path.join(build_dir, "*.tar.gz"))[-1]

            # save created package as resource
            resource_dir = self.abs_path()
            copy_path(tar, resource_dir)

            tar_name = os.path.basename(tar)

            new_path = os.path.join(resource_dir, tar_name + vertical + ".tar.gz")
            copy_path(os.path.join(resource_dir, tar_name), new_path)

            self.create_resource(
                description="src_setup for {} from {}".format(
                    vertical, str(Arcadia.parse_url(self.ctx[consts.ARCADIA_URL_KEY]))
                ),
                resource_path=new_path,
                resource_type=self.resources_map[vertical],
                arch=self.arch
            )

    def on_release(self, rest):
        nanny.ReleaseToNannyTask.on_release(self, rest)
        ArcadiaTask.on_release(self, rest)


__Task__ = BuildAppHostSrcSetup
