import json
import logging
import os.path

from sandbox.projects import resource_types
from sandbox.projects.app_host.BuildAppHostConfigBundle import BackendsConfigTask, BuildOnlyServiceResources
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.common import constants as consts
from sandbox.projects.common.app_host.options import GraphGenerator, ChooseVertical, PVA
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import make_folder, copy_path
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.release_machine import rm_notify
import sandbox.projects.release_machine.core.const as rm_const


# XXX


class BuildServiceResources(SandboxBoolParameter):
    name = 'build_service_resources'
    description = 'Build cross service resources (loop.conf etc)'
    default_value = False
    required = True


class BuildResourcesForChoosenVerticalsOnly(SandboxBoolParameter):
    name = 'build_resources_for_choosen_verticals_only'
    description = 'Build all resources only for choosen verticals'
    default_value = False
    required = True


class BuildTemplatesDir(SandboxBoolParameter):
    name = 'build_templates_dir'
    description = 'Save templates dir'
    default_value = True


def _make_url(base_url, subpath, rev):
    return "%s/%s@%s" % (base_url, subpath, rev)


@rm_notify.notify2()
class BuildAppHostSrcSetupConfigBundle(nanny.ReleaseToNannyTask, ArcadiaTask):

    """Builds a bundle with src_setup configs"""

    type = "BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE"

    execution_space = 1000

    input_parameters = [
        ChooseVertical,
        BackendsConfigTask,
        BuildServiceResources,
        BuildResourcesForChoosenVerticalsOnly,
        BuildOnlyServiceResources,
        SandboxArcadiaUrlParameter,
        GraphGenerator,
        PVA,
        BuildTemplatesDir
    ]

    pva_configs = {
        "production_src_setup_loop.conf": resource_types.APP_HOST_SRC_SETUP_PRODUCTION_LOOP_CONF,
        "production_src_setup_loop_isolated.conf": app_host_resources.APP_HOST_SRC_SETUP_PRODUCTION_LOOP_ISOLATED_CONF,
        "testing_src_setup_loop.conf": resource_types.APP_HOST_SRC_SETUP_TESTING_LOOP_CONF,
        "testing_src_setup_loop_isolated.conf": app_host_resources.APP_HOST_SRC_SETUP_TESTING_LOOP_ISOLATED_CONF,

    }

    resources_map = {
        "ATOM": {
            'bundle': resource_types.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_ATOM,
        },
        "WEB": {
            'bundle': resource_types.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_WEB,
            "pva": {
                "production_src_setup_loop.conf": resource_types.APP_HOST_SRC_SETUP_PRODUCTION_LOOP_CONF_WEB,
                "testing_src_setup_loop.conf": resource_types.APP_HOST_SRC_SETUP_TESTING_LOOP_CONF_WEB
            }
        },
        "NEWS": {
            'bundle': resource_types.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_NEWS,
        },
        "IMGS": {
            'bundle': resource_types.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_IMGS,
            "pva": {
                "production_src_setup_loop.conf": app_host_resources.APP_HOST_SRC_SETUP_PRODUCTION_LOOP_CONF_IMGS,
                "testing_src_setup_loop.conf": app_host_resources.APP_HOST_SRC_SETUP_TESTING_LOOP_CONF_IMGS,
                "production_src_setup_loop_isolated.conf": app_host_resources.APP_HOST_SRC_SETUP_PRODUCTION_LOOP_ISOLATED_CONF_IMGS,
                "testing_src_setup_loop_isolated.conf": app_host_resources.APP_HOST_SRC_SETUP_TESTING_LOOP_ISOLATED_CONF_IMGS,
            }
        },
        "VIDEO": {
            "bundle": resource_types.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_VIDEO,
            "pva": {
                "production_src_setup_loop.conf": resource_types.APP_HOST_SRC_SETUP_PRODUCTION_LOOP_CONF_VIDEO,
                "testing_src_setup_loop.conf": resource_types.APP_HOST_SRC_SETUP_TESTING_LOOP_CONF_VIDEO,
                "production_src_setup_loop_isolated.conf": app_host_resources.APP_HOST_SRC_SETUP_PRODUCTION_LOOP_ISOLATED_CONF_VIDEO,
                "testing_src_setup_loop_isolated.conf": app_host_resources.APP_HOST_SRC_SETUP_TESTING_LOOP_ISOLATED_CONF_VIDEO
            }
        },
        "VIDEO_HOSTING": {
            "bundle": app_host_resources.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_VIDEO_HOSTING,
        },
        "COMMON": {
            "bundle": app_host_resources.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_COMMON,
        },
        "MAIL": {
            "bundle": app_host_resources.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_MAIL,
        },
        "MAILCORP": {
            "bundle": app_host_resources.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_MAILCORP,
        },
        "SHARED": {
            "bundle": app_host_resources.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_SHARED,
        },
        "MUSIC": {
            "bundle": app_host_resources.APP_HOST_SRC_SETUP_CONFIG_BUNDLE_MUSIC,
        },
    }

    def __setup_paths(self, base_url, ah_base_url, url):
        self.__conf_dir = os.path.abspath("conf_dir")
        self.__ah_vertical_dir = os.path.abspath("ah_vertical_dir")

        Arcadia.export(_make_url(base_url, "conf", url.revision), self.__conf_dir)
        Arcadia.export(_make_url(ah_base_url, "conf/graph_generator/vertical", url.revision), self.__ah_vertical_dir)

    def __run_pva(self, config_name, vertical, output):
        pva = self.sync_resource(self.__get_pva())
        pva_config = os.path.join(self.__conf_dir, "pva", "config.json")

        logging.info("Building config {} for vertical {} into {} with PVA"
                     .format(config_name, vertical, output))

        logging.debug("PVA Config: {}".format(json.load(open(pva_config))))

        try:
            run_process([pva, pva_config, vertical or "''", config_name, output],
                        shell=True,
                        check=True,
                        outputs_to_one_file=False,
                        log_prefix="pva")
        except Exception as e:
            logging.error("Unexpected PVA error: ", e)
            raise

    def init_or_decrement_subtask_retry_count(self):
        r = self.ctx.get("backends_config_task_retries")

        if r == 0:
            raise SandboxTaskFailureError("child task failed, retry limit exceeded")

        self.ctx["backends_config_task_retries"] = 2 if r is None else r - 1

    def __get_graph_generator(self):
        if not hasattr(self, '__graph_generator'):
            self.__graph_generator = GraphGenerator.get_resource_from_ctx(self.ctx)
        return self.__graph_generator

    def __get_pva(self):
        if not hasattr(self, '__pva'):
            self.__pva = PVA.get_resource_from_ctx(self.ctx)
        return self.__pva

    def on_enqueue(self):
        self.ctx[rm_const.COMPONENT_CTX_KEY] = "src_setup"
        ArcadiaTask.on_enqueue(self)

    def do_execute(self):
        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)
        assert parsed_url.revision
        base_url = url.split("@")[0] + "/web/src_setup"
        ah_base_url = url.split("@")[0] + "/web/app_host"

        self.__setup_paths(base_url, ah_base_url, parsed_url)

        choosen_verticals = set()

        if not self.ctx.get(BuildOnlyServiceResources.name):
            if not self.ctx.get(BackendsConfigTask.name) and not self.ctx.get("backends_config_task_id"):
                if not self.ctx.get(GraphGenerator.name):
                    context = self.ctx.copy()
                    context[GraphGenerator.name] = self.__get_graph_generator()
                else:
                    context = self.ctx

                context["choose_app_to_build"] = "src_setup"

                subtask = self.create_subtask(
                    task_type="BUILD_APP_HOST_BACKENDS_CONFIG",
                    description=self.descr,
                    input_parameters=context,
                    priority=self.priority,
                    inherit_notifications=True
                )
                self.ctx["backends_config_task_id"] = subtask.id
                self.init_or_decrement_subtask_retry_count()
                logging.info(
                    "Starting subtask BUILD_APP_HOST_BACKENDS_CONFIG {}".format(
                        subtask
                    )
                )
                self.wait_tasks(
                    [subtask],
                    (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED,
                     self.Status.EXCEPTION),
                    True
                )

            graph_generator = self.sync_resource(self.__get_graph_generator())

            # get backends config resource
            task_id = self.ctx.get("backends_config_task_id") or self.ctx.get(BackendsConfigTask.name)
            if task_id is not None:
                resources = channel.sandbox.list_resources(resource_type="APP_HOST_BACKENDS_CONFIG",
                                                           task_id=task_id,
                                                           status="READY")
                if resources:
                    resource_ids = [(res.id, res.attributes) for res in resources]
                else:
                    del self.ctx["backends_config_task_id"]
                    self.do_execute()
            else:
                raise SandboxTaskFailureError("No task_id in backend task or selected config task")

            msgs = []

            for resource_id, attributes in resource_ids:
                vertical = attributes.get("vertical")
                choosen_verticals.add(vertical)
                # create package
                pkg_version = parsed_url.revision
                pkg_name = "src_setup_config_bundle_{}_{}".format(vertical, pkg_version)
                pkg_path = os.path.abspath(pkg_name)

                make_folder(pkg_name)
                os.chdir(pkg_name)

                backends_list = self.sync_resource(resource_id)

                result = run_process(
                    [
                        graph_generator,
                        'setup', vertical,
                        '-b', backends_list,
                        '-c', self.__conf_dir,
                        '-g', self.__ah_vertical_dir,
                        '-o', self.abs_path(pkg_name)
                    ],
                    shell=True,
                    check=True,
                    log_prefix="graph_generator"

                )

                stderr = open(result.stderr_path or result.stdout_path).readlines()
                if stderr:
                    msgs.append(vertical + '\n' + '\n'.join(stderr))

                # copy push-client.yml
                push_client_srcpath = os.path.join(self.__conf_dir, 'push-client-{}.yaml'.format(vertical))
                push_client_dstpath = os.path.join(pkg_path, 'push_client.yml')
                if os.path.exists(push_client_srcpath):
                    copy_path(push_client_srcpath,  push_client_dstpath)
                    logging.debug("Copied {} to {}".format(push_client_srcpath, push_client_dstpath))
                else:
                    logging.debug("Copy {} to {} FAILED, not such file in src".format(push_client_srcpath, push_client_dstpath))

                os.chdir("..")

                tar = pkg_path + ".tar.gz"
                run_process("tar czvf %s %s" % (tar, pkg_name), shell=True)

                res_id = self.create_resource(
                    description=pkg_name,
                    resource_path=tar,
                    resource_type=self.resources_map[vertical].get('bundle'),
                    arch="any"
                )

                self.mark_resource_ready(res_id.id)

        if self.ctx.get(BuildServiceResources.name):
            generated_dir = os.path.abspath("generated")

            if not self.ctx.get(BuildResourcesForChoosenVerticalsOnly.name):
                for config_name, config_type in self.pva_configs.items():
                    output_path = os.path.join(generated_dir, config_name)

                    self.__run_pva(config_name, '', output_path)

                    self.create_resource(
                        description="{}@{}".format(config_name, parsed_url.revision),
                        resource_path=output_path,
                        resource_type=config_type,
                        arch="any")

            for vertical in self.resources_map.iterkeys():
                if vertical not in choosen_verticals:
                    continue
                vertical_pva_configs = self.resources_map.get(vertical).get('pva')
                if vertical_pva_configs:
                    for config_name, config_type in vertical_pva_configs.iteritems():
                        output_path = os.path.join(generated_dir, config_name + vertical)

                        self.__run_pva(config_name, vertical, output_path)

                        self.create_resource(
                            description="{}@{}".format(config_name, parsed_url.revision),
                            resource_path=output_path,
                            resource_type=config_type,
                            arch="any")

        if self.ctx.get(BuildTemplatesDir.name):
            self.create_resource(
                description="{}@{}".format("templates", parsed_url.revision),
                resource_path=os.path.join(self.__conf_dir, "vertical"),
                resource_type=resource_types.APP_HOST_SRC_SETUP_TEMPLATES,
                arch="any"
            )


__Task__ = BuildAppHostSrcSetupConfigBundle
