import logging

from sandbox.sandboxsdk.parameters import TaskSelector
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.app_host.options import GraphGenerator

from sandbox.common.types.task import Status
from sandbox.common.utils import chain


class ConfigBundleTaskOld(TaskSelector):
    name = 'BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE_OLD'
    task_type = 'BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE'
    description = "OLD config bundle task"
    required = True
    default_value = None

    @classmethod
    def get_custom_parameters(cls):
        return {
            'type': ','.join(chain(cls.task_type or ())),
            'status': str(Status.RELEASED),
            'limit': 20,
        }


class ConfigBundleTaskNew(TaskSelector):
    name = 'BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE_NEW'
    task_type = 'BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE'
    description = "NEW config bundle task"
    required = True
    default_value = None


class BuildAppHostConfigBundleDiff(SandboxTask):

    """Shows diff for app_host config bundle"""

    type = "BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE_DIFF"

    execution_space = 4000

    input_parameters = [ConfigBundleTaskOld, ConfigBundleTaskNew, GraphGenerator]

    def on_execute(self):

        old_id = self.ctx.get(ConfigBundleTaskOld.name)
        new_id = self.ctx.get(ConfigBundleTaskNew.name)

        graph_generator = self.sync_resource(GraphGenerator.get_resource_from_ctx(self.ctx))

        assert old_id
        assert new_id

        task_ids = {'old': old_id, 'new': new_id}
        self.ctx['resource_ids'] = {'old': {}, 'new': {}}

        for key, task_id in task_ids.items():
            resources = channel.sandbox.list_resources(task_id=task_id, status="READY")
            for res in resources:
                if res.type.startswith('APP_HOST_SRC_SETUP_CONFIG_BUNDLE_'):
                    vertical = res.type.replace('APP_HOST_SRC_SETUP_CONFIG_BUNDLE_', '')
                    path = self.sync_resource(res.id)

                    folder = "{}_{}".format(vertical, key)
                    folder = self.abs_path(folder)
                    make_folder(folder)

                    self.ctx.get('resource_ids')[key][vertical] = folder

                    run_process("tar xzvf {} -C {} --strip-components=1".format(path, folder),
                                shell=True, check=True, log_prefix="tar_{}_{}".format(vertical, key))
        msgs = []
        logging.debug(self.ctx.get('resource_ids'))
        for vertical, folder in self.ctx.get('resource_ids').get('new').items():
            result = run_process(
                [
                    graph_generator,
                    '-d',
                    'setup_diff',
                    '-o', self.ctx.get('resource_ids').get('old').get(vertical),
                    '-n', self.ctx.get('resource_ids').get('new').get(vertical)
                ],
                shell=True,
                check=True,
                log_prefix="graph_generator_{}".format(vertical)
            )

            stderr = open(result.stderr_path or result.stdout_path).readlines()
            if stderr:
                msgs.append(vertical + '\n' + '\n'.join(stderr))

        if len(msgs):
            self.set_info("\n".join(msgs))


__Task__ = BuildAppHostConfigBundleDiff
