# -*- coding: utf-8 -*-

import os
from sandbox import sdk2
import jinja2
import datetime
import requests
import logging

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common import requests_wrapper
from sandbox.projects.common import task_env
from sandbox.projects.common.nanny.client import NannyClient


class CalculateAppHostMetricsOnBeta(sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        disk_space = 1000
        environments = (environments.PipEnvironment('yasmapi', use_wheel=True),)

    class Parameters(sdk2.Task.Parameters):
        metrics_task = sdk2.parameters.Task("Metrics task id", task_type="LAUNCH_METRICS_FOR_PRIEMKA", required=True)
        signal_names = ["unistat-SELF-SS-Failures_dmmm",
                        "unistat-SELF-CS-SourceFailures_dmmm",
                        "perc(unistat-SELF-CS-SourceFailures_dmmm, unistat-SELF-SS-Requests_dmmm)",
                        "instancectl-exit_signal_9_mmmm",
                        "instancectl-app_host.exit_signal_9_mmmm",
                        "instancectl-resolver.exit_signal_9_mmmm",
                        "instancectl-exit_signal_11_mmmm",
                        "instancectl-exit_signal_6_mmmm",
                        "modules-unistat-fetch_errors_mmmm",
                        "modules-unistat-filtered_signals_mmmm",
                        ]

    @staticmethod
    def _get_app_host_instances(service_id):
        token = sdk2.Vault.data("APP_HOST", "nanny_oauth_token")
        nanny = NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=token)
        hostnames = []
        for host in nanny.get_service_instances(service_id).get("result"):
            if host.get("network_settings") == "MTN_ENABLED":
                hostnames.append((host.get("container_hostname"), host.get("port")))
            else:
                hostnames.append((host.get("hostname"), host.get("port")))

        return hostnames

    @staticmethod
    def _get_app_host_graphs(hostnames):
        graphs = []
        for hostname, port in hostnames:
            result = requests_wrapper.get_r("{}:{}/admin?action=viewconfig:graphs:list".format(hostname, port))
            if result.ok:
                for graph in result.json():
                    if "@" not in graph:
                        graphs.append(graph)
                logging.debug(graphs)
                return graphs

        raise SandboxTaskFailureError("Could not download graphs list from {}".format(hostnames))

    def _get_yasm_signals_for_graphs(self, prj, start, end):
        from collections import OrderedDict
        from yasmapi import GolovanRequest

        hosts = "ASEARCH"
        period = 300

        tags = OrderedDict()
        tags["itype"] = "apphost"
        tags["prj"] = prj
        tags["geo"] = "sas,man,vla"
        tags["ctype"] = "priemka-apphost"

        tags_str = ";".join(["%s=%s" % (x, y) for x, y in tags.iteritems()])

        signals = {signal: tags_str + ":" + signal for signal in self.Parameters.signal_names}
        results = {signal: 0 for signal in self.Parameters.signal_names}

        for signal, signal_request in signals.iteritems():
            logging.debug(signal_request)
            points = GolovanRequest(hosts, period, start, end, [signal_request], max_retry=3, retry_delay=5)
            for ts, value in points:
                value_result = value.get(signal_request, 0)
                if value_result is None:
                    value_result = 0

                logging.debug("{} {} {} {}".format(signal, ts, value, value_result))
                results[signal] += value_result

                logging.debug(results[signal])

        logging.debug(results)

        return results

    @staticmethod
    def _get_service_id_from_beta(beta_name):
        r = requests_wrapper.post_r(
            "https://yappy.z.yandex-team.ru/api/yappy.services.Model/retrieveBeta",
            json={"name": beta_name, "withHistory": False}
        )

        if r.ok:
            return r.json().get('components')[0].get('slot').get('id')
        else:
            raise SandboxTaskFailureError("Failed to get service id for {}".format(beta_name))

    def _get_metrics_launch_id(self):
        return self.Parameters.metrics_task.Context.launch_info.get('launch_id')

    @staticmethod
    def _get_scraper_id(link):
        return link.split("/")[-1]

    @staticmethod
    def _get_scraper_download_interval(scraper_id, session):
        r = requests_wrapper.sget_r(
            session,
            "https://scraper.yandex-team.ru/api/scraper/batch/{}/status".format(scraper_id)
        )
        if r.json().get("status") == "COMPLETE":
            return r.json().get("created-date"), r.json().get("updated-date")

    @staticmethod
    def _get_link_and_beta_name(answer):
        return answer.get("scraperLink"), answer.get("server").get("name")

    def _get_mlm_info(self, launch_id, session):
        result = []

        r = requests_wrapper.sget_r(session, "https://mlm.yandex-team.ru/api/launch-set/{}".format(launch_id))

        for launch in r.json().get('launches'):
            evaluation = launch.get("evaluationType")
            region = launch.get("regionalType")
            groups = launch.get('queryGroups')

            assert not len(groups) % 2
            results = len(groups) / 2

            for i in range(results):
                before = groups[i]
                after = groups[results + i]
                result.append(
                    (evaluation, region, self._get_link_and_beta_name(before), self._get_link_and_beta_name(after))
                )

        return result

    @staticmethod
    def _convert_service_name_to_tag(service_name):
        return service_name.replace("_", "-")

    @staticmethod
    def _sort_dates(dates):
        r = []
        for date in dates:
            d = date.split(".")[0]
            epoch = int(datetime.datetime.strptime(d, "%Y-%m-%dT%H:%M:%S").strftime('%s'))
            r.append(epoch)

        return sorted(r)

    # removed 3 hours msk time, i'm lazy
    def _get_earliest(self, dates):
        return self._sort_dates(dates)[0]

    def _get_latest(self, dates):
        return self._sort_dates(dates)[-1]

    def on_execute(self):
        launch_id = self._get_metrics_launch_id()

        token = sdk2.Vault.data("APP_HOST", "mlm_token")
        session = requests.Session()
        session.headers.update({
            'Content-Type': 'application/json',
            'Authorization': 'OAuth {}'.format(token),
        })

        # beta_name -> [created_times, updated_times]
        start_stop_times = {
        }

        # beta_name -> [download_start, download_end]
        start_stop_time = {
        }

        for evaluation, region, before, after in self._get_mlm_info(launch_id, session):
            before_link, before_name = before
            after_link, after_name = after

            before_dates = self._get_scraper_download_interval(self._get_scraper_id(before_link), session)
            after_dates = self._get_scraper_download_interval(self._get_scraper_id(after_link), session)

            logging.debug("{}, {}, ({}, {}), ({}, {}), {}, {}".format(evaluation, region, before_link, before_name,
                                                                      after_link, after_name, before_dates, after_dates
                                                                      )
                          )

            start_stop_times.setdefault(before_name, {"created": [], "updated": []})
            start_stop_times.setdefault(after_name, {"created": [], "updated": []})

            start_stop_times[before_name]["created"].append(before_dates[0])
            start_stop_times[before_name]["updated"].append(before_dates[1])

            start_stop_times[after_name]["created"].append(after_dates[0])
            start_stop_times[after_name]["updated"].append(after_dates[1])

        logging.debug(start_stop_times)

        for beta_name, dates in start_stop_times.iteritems():
            start_stop_time[beta_name] = [
                self._get_earliest(dates.get("created")),
                self._get_latest(dates.get("updated"))
            ]

        logging.debug(start_stop_time)

        result = {beta_name: {} for beta_name in start_stop_time.iterkeys()}

        for beta_name, times in start_stop_time.iteritems():
            start_time, end_time = times
            service_id = self._get_service_id_from_beta(beta_name)
            service_prj = self._convert_service_name_to_tag(service_id)
            result[beta_name] = self._get_yasm_signals_for_graphs(service_prj, start_time, end_time)

        logging.debug(result)
        self.Context.result = result
        # self.set_info(json.dumps(result, indent=4))

    @sdk2.footer()
    def display_footer(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=['jinja2.ext.do'])
        return env.get_template("footer.html").render(
            {
                "result": self.Context.result,
                "signals": self.Parameters.signal_names
            }
        )
