# -*- coding: utf-8 -*-

from sandbox import sdk2
import requests
import logging
import json

class CheckGolovanSignalsQuota(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        beta_name = sdk2.parameters.String("Beta name", default="", required=True)
        signals_quota = sdk2.parameters.Integer("Quota of signals", default=0, required=True)

    def get_beta_hosts(self):
        r = requests.post(
            "https://yappy.z.yandex-team.ru/api/yappy.services.Model/retrieveBetasSlots",
            headers={
                "Content-Type": "application/json",
                "Authorization": "OAuth {}".format(sdk2.Vault.data("APP_HOST", "yappy_token"))
            },
            json={
                "betas_regexps": self.Parameters.beta_name,
                "allocated": True
            }
        )

        if not r.raise_for_status():
            logging.info("[INFO] Data from Yappy received")
            logging.info("[INFO] Yappy answer: {}".format(json.dumps(r.json(), indent=4)))

            response = r.json()

            if "betaSlots" in response:
                for beta in response["betaSlots"]:
                    if beta["betaName"] == self.Parameters.beta_name:
                        component_name = '-'.join(part for part in self.Parameters.beta_name.split('-')[0:-2])
                        hosts = beta["typeToSlots"][component_name]["hosts"]
                        logging.info("[INFO] Found hosts: {}".format(", ".join(hosts)))
                        return hosts
            else:
                logging.info("[INFO] No one betaSlots was found by betas_regexps '{}'".format(self.Parameters.beta_name))

            return []

    def on_execute(self):
        logging.info("[INFO] Starting Task")

        hosts = self.get_beta_hosts()

        logging.info("[INFO] {} has such hosts: ".format(self.Parameters.beta_name) + ', '.join(host for host in hosts))

        max_signals = 0
        for host in hosts:
            req = requests.get('http://{}/_golovan'.format(host))
            size_of_signals = len(req.json())

            logging.info("[INFO] Host {} : Signals {}".format(host, size_of_signals))

            max_signals = size_of_signals if size_of_signals > max_signals else max_signals

        if max_signals > self.Parameters.signals_quota:
            logging.error("[ERROR] Max size of signals ({}) is greater than quota ({})".format(max_signals, self.Parameters.signals_quota))
        else:
            logging.info("[INFO] Max size of signals ({}) is less than quota ({})".format(max_signals, self.Parameters.signals_quota))

        logging.info("[INFO] Ending Task")
