import logging
import os

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxBoolParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import paths

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask

from sandbox.projects.common import constants as consts
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny

from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.app_host.common import ParentResourcesParameter, get_url_attributes


_RESOURCES = [
    app_host_resources.APP_HOST_DAEMON_EARL_EXECUTABLE,

    app_host_resources.APP_HOST_EARL_DATA,
    app_host_resources.APP_HOST_EARL_CONFIG_EARL_CONF,
    app_host_resources.APP_HOST_EARL_CONFIG_NGINX_CONF_TEMPLATE,

    app_host_resources.APP_HOST_EARL_CONFIG_LOOP_CONF,
    app_host_resources.APP_HOST_EARL_CONFIG_ISS_HOOK_NOTIFY
]


class EarlOnlyDynamicResourcesParameter(SandboxBoolParameter):
    name = 'only_dynamic'
    description = 'Build and release only dynamic resources'
    required = True
    default = False


class ReleaseAppHostEarl(nanny.ReleaseToNannyTask, ArcadiaTask):
    """Gathers required earl components for a release"""

    type = "RELEASE_APP_HOST_EARL"

    execution_space = 1000

    input_parameters = [
        SandboxArcadiaUrlParameter,
        EarlOnlyDynamicResourcesParameter
    ]

    def do_execute(self):
        url = self.ctx[consts.ARCADIA_URL_KEY]

        parsed_url = Arcadia.parse_url(url)
        assert parsed_url.revision

        if 'required_attributes' not in self.ctx:
            self.ctx['required_attributes'] = get_url_attributes(parsed_url)

        if 'existing_resources' not in self.ctx:
            self.ctx['existing_resources'] = self.__find_existing_resources()

        if not self.ctx.get('step_executed_subtasks'):
            self.ctx['step_executed_subtasks'] = True
            self.__exec_subtasks()
            # May or may not break execution here

        if not self.ctx.get('step_executed_import'):
            self.ctx['step_executed_import'] = True
            self.__import_existing_resources()

    def __find_existing_resources(self):
        attributes = self.ctx['required_attributes']

        existing_resources = {}

        for resource_type in self.__list_requested_resource_types():
            logging.info("Searching for {} with attributes: {}".format(resource_type, attributes))

            found_resources = channel.sandbox.list_resources(resource_type=resource_type,
                                                             all_attrs=attributes,
                                                             limit=1,
                                                             status='READY',
                                                             arch=self.arch,
                                                             hidden=True)

            logging.debug("Found resources: {}".format(found_resources))

            if found_resources is not None:
                resource = found_resources[0]
                existing_resources[resource_type.name] = resource.id

        return existing_resources

    def __exec_subtasks(self):
        url = self.ctx[consts.ARCADIA_URL_KEY]

        existing_resources = self.ctx['existing_resources']

        resources_to_build = {}

        for resource_type in self.__list_requested_resource_types():
            if resource_type.name in existing_resources:
                continue

            fill_resource_map = resources_to_build.setdefault(resource_type.earl_build_task, {})

            fill_resource_map[resource_type.name] = self.__allocate_local_resource(url, resource_type).id

        if not resources_to_build:
            return

        for task_name, fill_resource_map in resources_to_build.iteritems():
            name_list = ', '.join(fill_resource_map.iterkeys())

            self.create_subtask(
                task_type=task_name,
                description="{} // building {} from {}".format(self.descr, name_list, url),
                input_parameters={
                    consts.ARCADIA_URL_KEY: url,
                    ParentResourcesParameter.name: fill_resource_map
                },
                arch=self.arch
            )

        utils.wait_all_subtasks_stop()

    def __import_existing_resources(self):
        url = self.ctx[consts.ARCADIA_URL_KEY]
        existing_resources = self.ctx['existing_resources']

        for resource_type in self.__list_requested_resource_types():
            remote_id = existing_resources.get(resource_type)

            if remote_id is None:
                continue

            remote_path = self.sync_resource(remote_id)

            resource = self.__allocate_local_resource(url, resource_type)

            try:
                paths.copy_path(remote_path, resource.path, copy_function=os.link)
            except OSError:
                paths.copy_path(remote_path, resource.path)

            self.mark_resource_ready(resource.id)

    def __allocate_local_resource(self, url, resource_type):
        attributes = self.ctx['required_attributes']

        path = os.path.basename(resource_type.earl_path)

        if resource_type.earl_tarball:
            path = path + '.tar.bz2'

        resource = self.create_resource(
            description=resource_type.name + ' // ' + url + ' // ' + self.descr,
            resource_type=resource_type,
            resource_path=path,
            attributes=attributes
        )

        return resource

    def __list_requested_resource_types(self):
        only_dynamic = self.ctx['only_dynamic']

        for resource_type in _RESOURCES:
            if not only_dynamic or resource_type.earl_dynamic_resource:
                yield resource_type


__Task__ = ReleaseAppHostEarl
