import json
import logging
import os
import shutil
import six
import time
from tempfile import mkdtemp

import sandbox.projects.abc.client as abc_client
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.helpers.staff_helper as staff_helper

import sandbox.sdk2 as sdk2
from sandbox.projects.app_host import resources as apphost_resources
from sandbox.projects.app_host.vertical_by_button.mixins import apphost_vertical
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import constants as sdk_consts
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.nanny import client as nanny_client
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.common import resource_selectors
from sandbox.projects.release_machine import client
from sandbox.projects.release_machine import security as rm_sec
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.svn import Arcadia


DEFAULT_PARENT_MACRO_NAME = "_APPHOST_NETS_"
DEFAULT_BALANCER_BACKEND = "rtc_balancer_apphost-verticals_in_yandex-team_ru_man_sas_vla"
DEFAULT_RECIPE_NAME = "auto"
MORTY_RECIPE_NAME = "morty_apphost_auto"
PARENT_MACRO_NAME_TEMPLATE = "_APPHOST_{vertical}_NETS_"
CHILD_MACRO_NAME_TEMPLATE = "_APPHOST_{vertical}_{ctype}_NETS_"
POD_SETS_NANNY_URL = "https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/"
ENDPOINT_SETS_NANNY_URL = "https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/endpoint-sets/"
AWACS_API_URL = "https://awacs.yandex-team.ru/api/"
SERVICE_NANNY_NAME = "{ctype}_app_host_{location}_{vertical_name}"
DEFAULT_APPHOST_NANNY_SERVICE = "app_host_vertical_default_service"
DEFAULT_NAMESPACE = "apphost-verticals.in.yandex-team.ru"
DNS_TEMPLATE = "{ctype}.{hostname}.apphost.in.yandex-team.ru"
SERVICES_GROUP_NANNY_URL = "https://nanny.yandex-team.ru/api/ya_vault/"
UPSTREAM_TEMPLATE_FILE = "upstream_template.md"
RM_COMPONENTS_DIR = "sandbox/projects/release_machine/components"
APPHOST_COMBINATOR = "apphost/conf/combinator"
APPHOST_JSON_CONFIG = "configs_source/config_specs/app_host_json.json"
INSTALLATION_JSON_FILE = "app_host/installation/{vertical}.json"
INSTALLATION_CONFIG = "configs_source/pieces/app_host/installation/{vertical}.json"
GEN_CONFIG_PATH = "configs_source/gen_config.json"
CREATE_NANNY_SERVICES_TASK_DIR = "sandbox/projects/app_host/vertical_by_button/CreateNannyServicesForApphostVertical"
ARC_COMMON_PATH = "arcadia"
ROBOT_MORTY_LOGIN = "robot-morty"

SECRET_VOLUMES = {
    "horizon_agent_secrets": {
        "name": "horizon_agent",
        "id": "sec-01dm5877q5cnv5nwbg8z3wxve4"
    },
    "pushclient_tvm_secret": {
        "name": "tvm.apphost.pushclient.DEFAULT",
        "id": "sec-01ey4dkeg030wnwdjxfhrcg362"
    }
}


class TimeoutError(Exception):
    """
    For use in wait-cycles.
    """
    pass


class CreateNannyServicesForApphostVertical(apphost_vertical.ApphostVerticalBaseTask):
    class Parameters(apphost_vertical.ApphostVerticalBaseTask.Parameters):
        kill_timeout = 2 * len(rm_const.ALL_LOCATIONS) * 60 * 60  # 10 hours
        use_temporary_quota = sdk2.parameters.Bool("Use temporary quota in YP", default=False)
        abc_group_id = sdk2.parameters.String("ABC group id for use in service", required=True)

        with sdk2.parameters.Output():
            configs = sdk2.parameters.Resource("Configs", resource_type=apphost_resources.AppHostConfigBundleUnified)

    class Context(apphost_vertical.ApphostVerticalBaseTask.Context):
        dashboard_name = None
        abc_slug = ""

    def on_enqueue(self):
        apphost_vertical.ApphostVerticalBaseTask.on_enqueue(self)
        with self.memoize_stage.configs_dir:
            logging.debug("Create resource for configs")
            self.Parameters.configs = apphost_resources.AppHostConfigBundleUnified(
                task=self,
                description="Configs for {}".format(self.Parameters.vertical_name),
                path=apphost_resources.AppHostConfigBundleUnified.arcadia_build_path,
                ttl="inf"
            )

    def create_macro(self, macro_name, parent):
        """
        Create new network macro in racktables.

        :param macro_name: new macro name
        :param parent: parent macro name
        """
        logging.debug("Create macro %s", macro_name)
        try:
            self.prj_networks_api.create_macro(
                name=macro_name,
                owner_service="svc_apphost",
                owners=",".join(["svc_apphost", "svc_{abc_slug}".format(abc_slug=self.Context.abc_slug)]),
                parent=parent,
                description="Macro for apphost vertical {}".format(self.Parameters.vertical_name),
            )
        except Exception as exc:
            eh.log_exception("Got exception", exc.inner_exception)
        else:
            self.prj_networks_api.create_network(macro_name)

    def create_all_macros_for_yp_pods(self):
        """
        Create parent macro for this service and child macros for prod and probably hamster.
        """
        parent_macro_name = PARENT_MACRO_NAME_TEMPLATE.format(
            vertical=self.Parameters.vertical_name.upper(),
        )
        if not self.prj_networks_api.macro_exists(parent_macro_name):
            self.set_info("Create macro {}".format(parent_macro_name))
            self.create_macro(
                macro_name=parent_macro_name,
                parent=DEFAULT_PARENT_MACRO_NAME,
            )
        for ctype in self.Context.ctypes:
            macro_name = CHILD_MACRO_NAME_TEMPLATE.format(
                vertical=self.Parameters.vertical_name.upper(),
                ctype=apphost_vertical.Ctypes(ctype).name.upper(),
            )
            if not self.prj_networks_api.macro_exists(macro_name):
                self.set_info("Create macro {}".format(macro_name))
                self.create_macro(
                    macro_name=macro_name,
                    parent=parent_macro_name,
                )
        self.set_info("All macros created")

    def delete_all_macros_for_yp_pods(self):
        """
        Remove all created macros for this service.
        """
        logging.debug("Delete all created macros for this apphost vertical")
        for ctype in self.Context.ctypes:
            macro_name = CHILD_MACRO_NAME_TEMPLATE.format(
                vertical=self.Parameters.vertical_name.upper(),
                ctype=apphost_vertical.Ctypes(ctype).name.upper(),
            )
            if self.prj_networks_api.macro_exists(macro_name):
                logging.debug("Delete %s", macro_name)
                self.prj_networks_api.delete_macro(
                    name=macro_name,
                )
        parent_macro_name = PARENT_MACRO_NAME_TEMPLATE.format(
            vertical=self.Parameters.vertical_name.upper(),
        )
        if self.prj_networks_api.macro_exists(parent_macro_name):
            logging.debug("Delete %s", parent_macro_name)
            self.prj_networks_api.delete_macro(
                name=parent_macro_name,
            )

    @decorators.retries(3, delay=10)
    def get_yp_pods(self, location, ctype):
        """
        Get yp pods from location and ctype.

        :param location: one of POSSIBLE_LOCATIONS
        :param ctype: 'production' or 'hamster'
        :return: ListPodsResponse instance
        """
        from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
        logging.info("Checking yp pods existence from location %s and ctype %s", location, ctype)
        req_pb = pod_sets_api_pb2.ListPodsRequest()
        req_pb.service_id = SERVICE_NANNY_NAME.format(
            ctype=ctype,
            location=location,
            vertical_name=self.Parameters.vertical_name,
        )
        req_pb.cluster = location.upper()
        try:
            pods = self.pod_sets_rpc_client.list_pods(req_pb)
            logging.debug("Got pods %s, total %s", pods, pods.total)
            return pods
        except Exception as exc:
            eh.log_exception("Got exception while getting yp pods", exc)

    @decorators.retries(3, delay=10, default_instead_of_raise=True)
    def get_pod_set(self, location, ctype):
        """
        Get yp pod set from location and ctype.

        :param location: one of POSSIBLE_LOCATIONS
        :param ctype: 'production' or 'hamster'
        :return: GetPodSetResponse instance
        """
        from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
        logging.info("Checking pod set existence from location %s and ctype %s", location, ctype)
        req_pb = pod_sets_api_pb2.GetPodSetRequest()
        req_pb.service_id = SERVICE_NANNY_NAME.format(
            ctype=ctype,
            location=location,
            vertical_name=self.Parameters.vertical_name,
        )
        req_pb.cluster = location.upper()
        try:
            pods = self.pod_sets_rpc_client.get_pod_set(req_pb)
            logging.debug("Got pod set %s", pods)
            return pods
        except Exception as exc:
            eh.log_exception("Got exception while getting pod set", exc)

    def remove_yp_pods_from_service(self, location, ctype):
        """
        Remove created yp pods in location and ctype.

        :param location: one of POSSIBLE_LOCATIONS
        :param ctype: 'production' or 'hamster'
        """
        from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
        logging.info("Removing pod set from location %s and ctype %s", location, ctype)
        req_pb = pod_sets_api_pb2.RemovePodSetRequest()
        req_pb.service_id = SERVICE_NANNY_NAME.format(
            ctype=ctype,
            location=location,
            vertical_name=self.Parameters.vertical_name,
        )
        req_pb.cluster = location.upper()
        try:
            self.pod_sets_rpc_client.remove_pod_set(req_pb)
        except Exception as exc:
            eh.log_exception("Got exception while removing pod set", exc)

    def create_yp_pods_in_location(self, location, ctype, service_id, network_macro):
        """
        Create yp pods for location, ctype with network macro.

        :param location: one of POSSIBLE_LOCATIONS
        :param service_id: nanny service name
        :param network_macro: network macro
        :return: created yp pod id
        """
        from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
        logging.info('Creating pod set...')

        req = pod_sets_api_pb2.CreatePodSetRequest()
        req.service_id = service_id
        req.cluster = location.upper()
        antiaffinity = req.antiaffinity_constraints
        antiaffinity.node_max_pods = 1
        antiaffinity.rack_max_pods = 0
        req.allocation_request.replicas = 1
        req.allocation_request.vcpu_guarantee = 500
        req.allocation_request.vcpu_limit = 500
        req.allocation_request.memory_guarantee_megabytes = 4096
        req.allocation_request.anonymous_memory_limit_megabytes = 0
        req.allocation_request.network_macro = network_macro
        req.allocation_request.snapshots_count = 5
        req.allocation_request.root_fs_quota_megabytes = 1024
        req.allocation_request.work_dir_quota_megabytes = 1024
        req.allocation_request.root_volume_storage_class = "hdd"
        req.allocation_request.root_bandwidth_guarantee_megabytes_per_sec = 8
        req.allocation_request.root_bandwidth_limit_megabytes_per_sec = 16
        volumes = req.allocation_request.persistent_volumes.add()
        volumes.disk_quota_megabytes = 5120
        volumes.mount_point = "/logs"
        volumes.storage_class = "hdd"
        volumes.bandwidth_guarantee_megabytes_per_sec = 5
        volumes.bandwidth_limit_megabytes_per_sec = 10
        cores_volume = req.allocation_request.persistent_volumes.add()
        cores_volume.disk_quota_megabytes = 5120
        cores_volume.mount_point = "/cores"
        cores_volume.storage_class = "hdd"
        cores_volume.bandwidth_guarantee_megabytes_per_sec = 5
        cores_volume.bandwidth_limit_megabytes_per_sec = 10

        quota_settings = req.quota_settings

        if self.Parameters.use_temporary_quota:
            quota_settings.mode = pod_sets_api_pb2.ResourceQuotaSettings.Mode.TMP_ACCOUNT
        else:
            quota_settings.mode = pod_sets_api_pb2.ResourceQuotaSettings.Mode.ABC_SERVICE
            quota_settings.abc_service_id = int(self.Parameters.abc_group_id)
        return self.wait_yp_pod_to_create(request=req, location=location, ctype=ctype)

    def wait_yp_pod_to_create(self, request, location, ctype):
        """
        Wait for network macro to appear in yp.

        :param request: CreatePodSetRequest instance
        """

        from nanny_rpc_client import exceptions as nanny_rpc_client_exceptions

        logging.debug("Wait for macro in YP")

        @apphost_vertical.waiter(wait_time_cycle_sec=60, total_wait_time_cycles_sec=15 * 60)
        def wait_func():
            try:

                # We should wait for new macro enable to use in YP.
                # When it is not possible to use it command below will fail without creating anything.
                yp_pod = self.pod_sets_rpc_client.create_pod_set(request)

            except nanny_rpc_client_exceptions.BadRequestError as bre:

                eh.log_exception("Got BadRequestError while trying to create pod set", bre)
                self.set_info("BAD REQUEST ERROR received from Nanny: {}".format(bre.message))

            except Exception as exc:
                eh.log_exception("Got exception while calling get_object method", exc)
                yp_pods = self.get_yp_pods(location=location, ctype=ctype)
                if yp_pods.total:
                    logging.debug("Got yp pod object %s", yp_pods.pods[0].meta.id)
                    return rm_core.Ok(value=yp_pods.pods[0].meta.id)

            else:
                logging.debug("Got yp pod object %s", yp_pod)
                return rm_core.Ok(yp_pod.pod_ids[0])

            return rm_core.Error()

        return wait_func()

    @decorators.change_exception_to_none
    def check_nanny_service_existence(self, service_id):
        """
        Check nanny service existence.

        :param service_id: nanny service name
        :return: boolean, whether nanny service exists or not
        """
        logging.debug("Check nanny service %s existence", service_id)
        return self.nanny_client.get_service(service_id)

    def copy_nanny_service(self, location, ctype):
        """
        Copy default nanny service as new service in location with ctype.

        :param location: one of POSSIBLE_LOCATIONS
        :param ctype: 'production' or 'hamster'
        """
        service_id = SERVICE_NANNY_NAME.format(
            ctype=ctype,
            location=location,
            vertical_name=self.Parameters.vertical_name,
        )
        logging.debug("Copy %s into %s", DEFAULT_APPHOST_NANNY_SERVICE, service_id)
        try:
            self.nanny_client.copy_service(
                src_service_id=DEFAULT_APPHOST_NANNY_SERVICE,
                new_service_id=service_id,
                category="apphost/{}/{}".format(self.Parameters.vertical_name, ctype),
                description="Generate {ctype} service for {vertical} in {location}".format(
                    ctype=ctype,
                    vertical=self.Parameters.vertical_name,
                    location=location,
                ),
                abc_group_id=int(self.Parameters.abc_group_id),
            )
        except Exception as exc:
            eh.log_exception("Got exception while copying nanny service", exc)
        self.check_nanny_service_existence(service_id)

    def delete_nanny_service(self, location, ctype):
        """
        Delete nanny service for current vertical in location with ctype.
        :param location: one of POSSIBLE_LOCATIONS
        :param ctype: 'production' or 'hamster'
        """
        service_id = SERVICE_NANNY_NAME.format(
            ctype=ctype,
            location=location,
            vertical_name=self.Parameters.vertical_name,
        )
        pods = self.get_yp_pods(location=location, ctype=ctype)
        pod_set = self.get_pod_set(location=location, ctype=ctype)
        if pod_set or (pods and pods.total):
            self.delete_all_service_snapshots(service_id=service_id)
            self.remove_yp_pods_from_service(location=location, ctype=ctype)

        if self.check_nanny_service_existence(service_id=service_id) is not None:
            logging.debug("Delete nanny service %s", service_id)
            try:
                self.nanny_client.delete_service(service_id=service_id)
            except Exception as exc:
                eh.log_exception("Got exception while copying nanny service", exc)

    def delete_all_service_snapshots(self, service_id):
        """
        Delete all snapshots from service.

        :param service_id: nanny service name
        """
        logging.debug("Delete all snapshots in %s", service_id)
        self.nanny_client.delete_all_snapshots(service_id=service_id)
        logging.debug("Wait for all snapshots to delete")
        current_state = self.nanny_client.get_service_current_state(service_id=service_id)
        logging.debug("Got current service state %s", current_state)
        start_time = int(time.time())
        while current_state["content"]["active_snapshots"]:
            if int(time.time()) - start_time > 15 * 60:
                logging.error("Waiting too long for snapshots to delete")
                raise TimeoutError
            time.sleep(60)
            current_state = self.nanny_client.get_service_current_state(service_id=service_id)
            logging.debug("Got current service state %s", current_state)

    def update_service_info_attrs(self, service_id, ctype, location):
        """
        Update info attributes in nanny service, change tickets integration resource.

        :param service_id: nanny service name
        :param ctype: 'production' or 'hamster'
        :param location: one of POSSIBLE_LOCATIONS
        """
        info_attrs = self.nanny_client.get_service_info_attrs(service_id=service_id)
        logging.debug("Got info attrs for service %s: %s", service_id, info_attrs)
        info_attrs["content"]["tickets_integration"]["service_release_rules"][2]["sandbox_resource_type"] = (
            "APP_HOST_STABLE_BRANCH_{vertical}".format(vertical=self.Parameters.vertical_name.upper())
        )
        labels_for_service = {
            "ctype": "prod" if ctype == "production" else ctype,
            "prj": self.Parameters.vertical_name,
            "geo": location,
            "dc": location,
        }
        for key, value in labels_for_service.items():
            found_label = False
            for label in info_attrs["content"]["labels"]:
                if label["key"] == key:
                    label["value"] = value
                    found_label = True
                    break
            if not found_label:
                info_attrs["content"]["labels"].append({"key": key, "value": value})
        info_attrs["snapshot_id"] = info_attrs["_id"]
        info_attrs.pop("_id")
        info_attrs["comment"] = "Update stable_branch resource in tickets integration"
        logging.debug("Update info attrs: %s", info_attrs)
        self.nanny_client.set_service_info_attrs(service_id=service_id, info_attrs=info_attrs)

    def update_service_auth_attrs(self, service_id):
        """
        Update auth attributes in nanny service, add service owners.

        :param service_id: nanny service name
        """
        auth_attrs = self.nanny_client.get_service_auth_attrs(service_id=service_id)
        logging.debug("Got auth attrs for service %s: %s", service_id, auth_attrs)
        service_owners = list({rm_const.ROBOT_RELEASER_USER_NAME, self.author, self.c_info.responsible})
        auth_attrs["content"]["owners"]["logins"] = service_owners
        staff_group_id = self.get_staff_group_id_by_abc_group_id(self.Parameters.abc_group_id)
        auth_attrs["content"]["owners"]["groups"] = list({staff_group_id, "64819"})

        if ROBOT_MORTY_LOGIN not in auth_attrs["content"]["conf_managers"]["logins"]:
            auth_attrs["content"]["conf_managers"]["logins"].append(ROBOT_MORTY_LOGIN)
        if ROBOT_MORTY_LOGIN not in auth_attrs["content"]["ops_managers"]["logins"]:
            auth_attrs["content"]["ops_managers"]["logins"].append(ROBOT_MORTY_LOGIN)

        logging.debug("Update auth attrs with users: %s", service_owners)
        self.nanny_client.set_service_auth_attrs(
            service_id=service_id,
            snapshot_id=auth_attrs["_id"],
            content=auth_attrs["content"],
        )

    def update_apphost_combinator(self):
        Arcadia.update(os.path.join(ARC_COMMON_PATH, APPHOST_COMBINATOR), set_depth="infinity", parents=True)
        short_ctypes = ["prod"]
        if apphost_vertical.Ctypes.hamster.value in self.Context.ctypes:
            short_ctypes.append("hamster")
        with open(os.path.join(self.ya_root, ARC_COMMON_PATH, APPHOST_COMBINATOR, GEN_CONFIG_PATH), "r") as gen_config:
            gen_config_file = json.load(gen_config)
            gen_config_file["ctypes_mapping"][self.Parameters.vertical_name.upper()] = {
                "@{ctype}".format(ctype=ctype): [ctype] for ctype in short_ctypes
            }

        with open(os.path.join(self.ya_root, ARC_COMMON_PATH, APPHOST_COMBINATOR, GEN_CONFIG_PATH), "w") as gen_config:
            json.dump(gen_config_file, gen_config, indent=4, sort_keys=True, separators=(',', ':'))

        installation_config_path = os.path.join(
            self.ya_root,
            ARC_COMMON_PATH,
            APPHOST_COMBINATOR,
            INSTALLATION_CONFIG.format(vertical=self.vertical_name_snake_case),
        )
        with open(installation_config_path, "w") as installation_config:
            json.dump(
                {"installation": "{vertical}".format(vertical=self.vertical_name_upper_case)},
                installation_config,
                indent=4,
                separators=(',', ':'),
            )

        Arcadia.add(
            installation_config_path,
            parents=True,
            force=True,
        )

        with open(
            os.path.join(self.ya_root, ARC_COMMON_PATH, APPHOST_COMBINATOR, APPHOST_JSON_CONFIG), "r"
        ) as json_config:
            json_config_file = json.load(json_config)
            for config_element in json_config_file:
                if isinstance(config_element, dict) and isinstance(config_element.items()[0][1], six.string_types):
                    if config_element.items()[0][1].startswith("app_host/installation/"):
                        logging.debug("Installation configs found")
                        config_element["+{}".format(self.vertical_name_upper_case)] = INSTALLATION_JSON_FILE.format(
                            vertical=self.vertical_name_snake_case,
                        )
                        break

        with open(
            os.path.join(self.ya_root, ARC_COMMON_PATH, APPHOST_COMBINATOR, APPHOST_JSON_CONFIG), "w"
        ) as json_config:
            json.dump(json_config_file, json_config, indent=4, sort_keys=True, separators=(',', ':'))

        build_path = mkdtemp()
        self.set_info("Start building apphost combinator")
        arcadiasdk.do_build(
            build_system=sdk_consts.YA_MAKE_FORCE_BUILD_SYSTEM,
            source_root=os.path.join(self.ya_root, ARC_COMMON_PATH),
            targets=[APPHOST_COMBINATOR],
            build_type=sdk_consts.RELEASE_BUILD_TYPE,
            clear_build=False,
            checkout=True,
            results_dir=build_path,
        )
        with sdk2.helpers.ProcessLog(self, logger="combinator_run") as pl1:
            sp.Popen(
                [os.path.join(build_path, "bin", "combinator")],
                cwd=os.path.join(self.ya_root, ARC_COMMON_PATH, APPHOST_COMBINATOR),
                stdout=pl1.stdout,
                stderr=pl1.stderr,
            )
        generated_path = mkdtemp()
        arcadiasdk.do_build(
            build_system=sdk_consts.YMAKE_BUILD_SYSTEM,
            source_root=os.path.join(self.ya_root, ARC_COMMON_PATH),
            targets=[os.path.join(APPHOST_COMBINATOR, "generated")],
            build_type=sdk_consts.RELEASE_BUILD_TYPE,
            clear_build=False,
            checkout=True,
            results_dir=generated_path,
        )
        os.makedirs(str(self.Parameters.configs.path.parent))
        shutil.copyfile(
            os.path.join(generated_path, APPHOST_COMBINATOR, "generated/configs.tar.gz"),
            str(self.Parameters.configs.path)
        )
        sdk2.ResourceData(self.Parameters.configs).ready()
        self.set_info("Apphost combinator build finished")

        for ctype in short_ctypes:
            try:
                dir_to_add = os.path.join(
                    self.ya_root,
                    ARC_COMMON_PATH,
                    APPHOST_COMBINATOR,
                    "generated",
                    self.Parameters.vertical_name.upper(),
                    ctype,
                )
                Arcadia.add(
                    dir_to_add,
                    parents=True,
                    force=True,
                )
            except Exception as exc:
                eh.log_exception(
                    "Could not add dir {dir}, got exception".format(dir=dir_to_add),
                    exc,
                )
        if Arcadia.diff(os.path.join(self.ya_root, ARC_COMMON_PATH)):
            commit_message = "Add new vertical `{vertical}`".format(
                vertical=self.Parameters.vertical_name,
            )
            review_id = self._create_review(commit_message=commit_message)
            self.skip_check_in_review_and_commit(review_id=review_id)
            Arcadia.update(os.path.join(ARC_COMMON_PATH, APPHOST_COMBINATOR))

    def _find_last_released(self, resource_type):
        last_resource_id, _ = resource_selectors.by_last_released_task(resource_type=sdk2.Resource[resource_type])
        last_released_resource = sdk2.Resource[last_resource_id]
        logging.info("Last released task id %s", last_released_resource.task_id)
        # sdk2.Resource.find() returns object `sandbox.sdk2.internal.common.Query` object
        # and `query.limit(0)` returns min(total_query_objects, 3000).
        # This means that it's not neccessary to call `limit(total_query_objects)` to get all objects.
        resources = list(sdk2.Resource.find(task_id=last_released_resource.task_id).limit(0))
        logging.info("Last released resources for %s: %s", last_resource_id, resources)
        return resources

    def update_runtime_attr_spec(self, service_id, ):
        runtime_attrs = self.nanny_client.get_service_runtime_attrs(service_id=service_id)
        snapshot_id = runtime_attrs["_id"]
        logging.debug("Got runtime attrs for service %s: %s", service_id, runtime_attrs)
        last_released_resources = self._find_last_released(self.c_info.releases_cfg__resources_info[0].resource_type)
        last_released_resources.extend(self._find_last_released(sdk2.Resource["APP_HOST_DAEMON_EXECUTABLE"]))

        for resource in last_released_resources:
            if resource.type in self.RESOURCE_TYPE_TO_FILE_NAME:
                for runtime_file in runtime_attrs["content"]["resources"]["sandbox_files"]:
                    if runtime_file["local_path"] == self.RESOURCE_TYPE_TO_FILE_NAME[resource.type]["name"]:
                        runtime_file["task_type"] = self.RESOURCE_TYPE_TO_FILE_NAME[resource.type]["task"]
                        runtime_file["task_id"] = str(resource.task_id)
                        runtime_file["resource_id"] = str(resource.id)
                        runtime_file["resource_type"] = str(resource.type)
                        logging.debug("Resource %s updated: %s", resource.type, runtime_file)
        for runtime_file in runtime_attrs["content"]["resources"]["sandbox_files"]:
            if runtime_file["local_path"] == "configs.tar.gz":
                runtime_file["task_type"] = str(self.type)
                runtime_file["task_id"] = str(self.id)
                runtime_file["resource_id"] = str(self.Parameters.configs.id)
                runtime_file["resource_type"] = "APP_HOST_CONFIG_BUNDLE_UNIFIED"
                logging.debug("Resource APP_HOST_CONFIG_BUNDLE_UNIFIED updated: %s", runtime_file)
        update_runtime_attrs_response = self.nanny_client.update_service_runtime_attrs(
            service_id=service_id,
            content=runtime_attrs["content"],
            snapshot_id=snapshot_id,
        )
        logging.debug("Update runtime_attrs response %s", update_runtime_attrs_response)

    def update_service_instance_spec(self, service_id):
        """
        Update instance spec in nanny service, add needed vault secret volumes.

        :param service_id: nanny service name
        """
        instance_spec = self.nanny_client.get_service_instance_spec(service_id=service_id)
        logging.debug("Got instance spec for service %s: %s", service_id, instance_spec)

        current_volumes = [volume["name"] for volume in instance_spec["content"]["volume"]]
        changed = False

        for volume_name, secret in SECRET_VOLUMES.items():
            if volume_name in current_volumes:
                logging.debug("%s is already in services volumes", volume_name)
                continue
            logging.debug("Add secret volume %s to service instance spec", volume_name)

            secret_versions_response = self.vault_client.get_secret(secret["id"])
            secret_version = secret_versions_response["secret_versions"][0]["version"]
            delegation_token, _ = self.vault_client.create_token(secret["id"])
            instance_spec["content"]["volume"].append(
                {
                    "name": volume_name,
                    "type": "VAULT_SECRET",
                    "vaultSecretVolume": {
                        "vaultSecret": {
                            "secretName": secret["name"],
                            "secretId": secret["id"],
                            "secretVer": secret_version,
                            "delegationToken": delegation_token,
                        },
                    },
                }
            )
            changed = True

        containers = instance_spec["content"]["containers"]
        containers[0]["coredumpPolicy"]["coredumpProcessor"]["aggregator"]["saas"]["serviceName"] = "apphost"
        containers[1]["coredumpPolicy"]["coredumpProcessor"]["aggregator"]["saas"]["serviceName"] = "horizon"

        if not changed:
            return

        instance_spec["comment"] = "Add secret volumes"
        self.nanny_client.update_service_instance_spec(service_id=service_id, instance_spec_data=instance_spec)

    def update_service_instances(self, service_id, ctype, pod_id, location):
        """
        Enable YP pods in nanny service.

        :param service_id: nanny service name
        :param pod_id: YP pod id to enable
        :param location: one of POSSIBLE_LOCATIONS
        """
        instances_data = self.nanny_client.get_service_instances(service_id=service_id)
        logging.debug("Got instances data for service %s: %s", service_id, instances_data)
        if len(instances_data["content"]["yp_pod_ids"]["pods"]) > 0:
            existing_pods = [pod["pod_id"] for pod in instances_data["content"]["yp_pod_ids"]["pods"]]
            if pod_id in existing_pods:
                logging.debug("Pod is already enabled")
                return
        instances_data["content"]["yp_pod_ids"]["pods"].append(
            {
                "pod_id": pod_id,
                "cluster": location.upper(),
            }
        )
        instances_data["content"]["yp_pod_ids"]["orthogonal_tags"]["prj"] = self.Parameters.vertical_name
        instances_data["content"]["yp_pod_ids"]["orthogonal_tags"]["metaprj"] = self.Parameters.vertical_name
        instances_data["content"]["yp_pod_ids"]["orthogonal_tags"]["ctype"] = "prod" if ctype == "production" else ctype
        instances_data["content"]["iss_settings"]["hooks_resource_limits"]["iss_hook_install"]["cpu_limit"] = 100
        instances_data["comment"] = "Enable pod {pod} in location {location}".format(pod=pod_id, location=location)
        self.nanny_client.update_service_instances(service_id=service_id, instances_data=instances_data)

    def prepare_nanny_service(self, location, ctype):
        """
        Prepare nanny service in location and ctype.
        :param location: one of POSSIBLE_LOCATIONS
        :param ctype: 'production' or 'hamster'
        """
        self.copy_nanny_service(location=location, ctype=ctype)
        network_macro = CHILD_MACRO_NAME_TEMPLATE.format(
            vertical=self.Parameters.vertical_name.upper(),
            ctype=apphost_vertical.Ctypes(ctype).name.upper(),
        )
        service_id = SERVICE_NANNY_NAME.format(
            ctype=ctype,
            location=location,
            vertical_name=self.Parameters.vertical_name,
        )
        self.set_info("Prepare nanny service {}".format(service_id))
        pods = self.get_yp_pods(location=location, ctype=ctype)
        if not pods.total:
            pod_id = self.create_yp_pods_in_location(
                location=location,
                ctype=ctype,
                service_id=service_id,
                network_macro=network_macro,
            )
        else:
            pod_id = pods.pods[0].meta.id
        if self.check_endpoint_set_existence(location=location, ctype=ctype) is None:
            self.create_endpoint_set(service_id=service_id, ctype=ctype, location=location)
        self.update_service_instance_spec(service_id=service_id)
        self.update_service_instances(service_id=service_id, ctype=ctype, pod_id=pod_id, location=location)
        self.update_runtime_attr_spec(service_id=service_id)
        self.update_service_info_attrs(service_id=service_id, ctype=ctype, location=location)
        self.update_service_auth_attrs(service_id=service_id)
        self.set_info("Nanny service {} prepared".format(service_id))

    @decorators.change_exception_to_none
    def check_endpoint_set_existence(self, location, ctype):
        """
        Check whether endpoint set exists or not.

        :param location: one of POSSIBLE_LOCATIONS
        :param ctype: 'production' or 'hamster'
        :return: boolean, whether endpoint set exists or not.
        """
        from infra.nanny.yp_lite_api.proto import endpoint_sets_api_pb2

        req_pb = endpoint_sets_api_pb2.GetEndpointSetRequest()
        req_pb.id = SERVICE_NANNY_NAME.format(
            ctype=ctype,
            location=location,
            vertical_name=self.Parameters.vertical_name,
        )
        req_pb.cluster = location.upper()
        return self.endpoint_sets_rpc_client.get_endpoint_set(req_pb)

    def create_endpoint_set(self, service_id, ctype, location):
        """
        Create endpoint set in nanny service.

        :param service_id: nanny service name
        :param ctype: endpoint set exists or not.
        :param location: one of POSSIBLE_LOCATIONS
        """
        from infra.nanny.yp_lite_api.proto import endpoint_sets_api_pb2
        logging.info('Creating endpoint set...')
        req = endpoint_sets_api_pb2.CreateEndpointSetRequest()
        req.cluster = location.upper()
        req.meta.id = SERVICE_NANNY_NAME.format(
            ctype=ctype,
            location=location,
            vertical_name=self.Parameters.vertical_name,
        )
        req.meta.service_id = service_id
        req.spec.protocol = "TCP"
        req.spec.port = 84
        endpoint_set_response = self.endpoint_sets_rpc_client.create_endpoint_set(req)
        logging.debug("Got %s endpoint set created", endpoint_set_response.endpoint_set.meta.id)

    @decorators.change_exception_to_none
    def check_domain_existence(self):
        """
        Check whether domain for nanny service exists or not.

        :return: GetDomainResponse if domain for nanny service exists else None
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2

        domain_stub = api_stub.DomainServiceStub(self.awacs_client)
        req_pb = api_pb2.GetDomainRequest()
        req_pb.id = self.Parameters.vertical_name
        req_pb.namespace_id = DEFAULT_NAMESPACE
        return domain_stub.get_domain(req_pb)

    def create_domain(self, upstreams):
        """
        Create domain for nanny service in common balancer.
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2
        from infra.awacs.proto import model_pb2
        from infra.awacs.proto import modules_pb2

        if self.check_domain_existence() is not None:
            logging.debug("Domain %s already exists", self.Parameters.vertical_name)
            return

        domain_stub = api_stub.DomainServiceStub(self.awacs_client)
        fqdn = "*.{}.apphost.in.yandex-team.ru".format(self.Parameters.vertical_name)
        req_pb = api_pb2.CreateDomainRequest()
        req_pb.meta.id = self.Parameters.vertical_name
        req_pb.meta.namespace_id = DEFAULT_NAMESPACE
        req_pb.meta.comment = "Add upstream for {}".format(self.Parameters.vertical_name)
        req_pb.order.protocol = model_pb2.DomainSpec.Config.Protocol.HTTP_AND_HTTPS
        req_pb.order.fqdns.extend([fqdn])
        include_upstreams = req_pb.order.include_upstreams
        include_upstreams.type = modules_pb2.IncludeUpstreamsType.BY_ID
        include_upstreams.ids.extend(upstreams)
        cert_order_content = req_pb.order.cert_order.content
        cert_order_content.abc_service_id = int(self.Parameters.abc_group_id)
        cert_order_content.ca_name = "InternalCA"
        cert_order_content.common_name = fqdn
        cert_order_content.public_key_algorithm_id = "rsa"

        try:
            logging.debug("Start create_domain method request")
            resp_pb = domain_stub.create_domain(req_pb)
        except Exception as exc:
            logging.exception("Got exception %s", exc)
            return
        logging.debug("Got create_domain method response %s", resp_pb)

    def remove_domain(self):
        """
        Remove domain for nanny service from common balancer.
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2

        check_domain_resp = self.check_domain_existence()

        if check_domain_resp is None:
            logging.debug("Domain %s already removed", self.Parameters.vertical_name)
            return

        domain_stub = api_stub.DomainServiceStub(self.awacs_client)
        req_pb = api_pb2.RemoveDomainRequest()
        req_pb.id = self.Parameters.vertical_name
        req_pb.namespace_id = DEFAULT_NAMESPACE
        req_pb.version = check_domain_resp.domain.meta.version
        try:
            resp_pb = domain_stub.remove_domain(req_pb)
        except Exception as exc:
            logging.debug(
                "Could not remove domain %s for namespace %s, got exception: %s",
                self.Parameters.vertical_name,
                DEFAULT_NAMESPACE,
                exc,
            )
        else:
            logging.debug("Got remove_domain method response: %s", resp_pb)

    @decorators.change_exception_to_none
    def check_certificate_existence(self):
        """
        Check whether certificate for nanny service exists or not.

        :return: GetDomainResponse if domain for nanny service exists else None
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2

        cert_stub = api_stub.CertificateServiceStub(self.awacs_client)
        req_pb = api_pb2.GetCertificateRequest()
        req_pb.id = "*.{}.apphost.in.yandex-team.ru".format(self.Parameters.vertical_name)
        req_pb.namespace_id = DEFAULT_NAMESPACE
        return cert_stub.get_certificate(req_pb)

    def remove_certificate(self):
        """
        Remove certificate for nanny service from common balancer.
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2
        from infra.awacs.proto import model_pb2

        check_cert_resp = self.check_certificate_existence()

        if check_cert_resp is None:
            logging.debug("Certificate %s already removed", self.Parameters.vertical_name)
            return

        fqdn = "*.{}.apphost.in.yandex-team.ru".format(self.Parameters.vertical_name)
        cert_stub = api_stub.CertificateServiceStub(self.awacs_client)
        req_pb = api_pb2.RemoveCertificateRequest()
        req_pb.id = fqdn
        req_pb.namespace_id = DEFAULT_NAMESPACE
        req_pb.version = check_cert_resp.certificate.meta.version
        req_pb.state = model_pb2.CertificateSpec.State.REMOVED_FROM_AWACS
        try:
            resp_pb = cert_stub.remove_certificate(req_pb)
        except Exception as exc:
            logging.debug(
                "Could not remove certificate %s for namespace %s, got exception: %s", fqdn, DEFAULT_NAMESPACE, exc,
            )
        else:
            logging.debug("Got remove_domain method response: %s", resp_pb)

    def prepare_balancer_upstream(self):
        """
        Prepare backend, DNS and upstream for service in common balancer.
        """
        upstreams = []
        self.set_info("Prepare balancer")
        for ctype in self.Context.ctypes:
            ctyped_vertical_name = "{ctype}_{vertical}".format(
                ctype=ctype, vertical=self.Parameters.vertical_name,
            )
            upstreams.append(ctyped_vertical_name)
            self.create_backend(ctyped_vertical_name, ctype=ctype)
            self.create_dns(ctyped_vertical_name, ctype=ctype)
            self.create_upstream(ctyped_vertical_name, ctype=ctype)
        self.create_domain(upstreams)
        self.set_info("Balancer prepared")

    def remove_balancer_upstream(self):
        """
        Remove backend, DNS and upstream for service from common balancer.
        """
        self.remove_certificate()
        self.remove_domain()
        for ctype in self.Context.ctypes:
            ctyped_vertical_name = "{ctype}_{vertical}".format(
                ctype=ctype, vertical=self.Parameters.vertical_name,
            )
            self.remove_upstream(ctyped_vertical_name)
            self.remove_backend(ctyped_vertical_name)
            self.remove_dns(ctyped_vertical_name, ctype=ctype)

    def remove_upstream(self, ctyped_vertical_name):
        """
        Remove upstream for nanny service from common balancer.
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2

        upstream_resp = self.check_upstream_existence(ctyped_vertical_name)
        if upstream_resp is None:
            logging.debug("Upstream %s already removed", ctyped_vertical_name)
            return

        dns_stub = api_stub.UpstreamServiceStub(self.awacs_client)
        req_pb = api_pb2.RemoveUpstreamRequest()
        req_pb.id = ctyped_vertical_name
        req_pb.namespace_id = DEFAULT_NAMESPACE
        req_pb.version = upstream_resp.upstream.meta.version
        try:
            resp_pb = dns_stub.remove_upstream(req_pb)
        except Exception as exc:
            logging.debug("Could not remove upstream %s, got exception: %s", ctyped_vertical_name, exc)
        else:
            logging.debug("Got remove_upstream method response: %s", resp_pb)

    @decorators.change_exception_to_none
    def check_upstream_existence(self, ctyped_vertical_name):
        """
        Check whether upstream for nanny service exists or not.

        :param ctyped_vertical_name: upstream name.
        :return: GetDnsRecordResponse if upstream for nanny service exists else None
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2

        dns_stub = api_stub.UpstreamServiceStub(self.awacs_client)
        req_pb = api_pb2.GetDnsRecordRequest()
        req_pb.id = ctyped_vertical_name
        req_pb.namespace_id = DEFAULT_NAMESPACE
        return dns_stub.get_upstream(req_pb)

    def create_upstream(self, ctyped_vertical_name, ctype):
        """
        Create upstream for nanny service in common balancer.

        :param ctyped_vertical_name: upstream name.
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2
        from infra.awacs.proto import model_pb2

        if self.check_upstream_existence(ctyped_vertical_name) is not None:
            logging.debug("Upstream %s already exists", ctyped_vertical_name)
            return

        with open(
            os.path.join(self.ya_root, "arcadia", CREATE_NANNY_SERVICES_TASK_DIR, UPSTREAM_TEMPLATE_FILE), "r"
        ) as upstream_template_file:
            upstream_template = upstream_template_file.read()

        upstream_stub = api_stub.UpstreamServiceStub(self.awacs_client)
        req_pb = api_pb2.CreateUpstreamRequest(
            meta=model_pb2.UpstreamMeta(
                id=ctyped_vertical_name,
                namespace_id=DEFAULT_NAMESPACE,
                comment="Add upstream for {}".format(ctyped_vertical_name),
                auth=model_pb2.Auth(type=model_pb2.Auth.AuthType.STAFF),
            ),
            spec=model_pb2.UpstreamSpec(
                type=model_pb2.BalancerType.YANDEX_BALANCER,
                labels={"order": "10000000"},
                yandex_balancer=model_pb2.YandexBalancerUpstreamSpec(
                    mode=model_pb2.YandexBalancerUpstreamSpec.ConfigMode.EASY_MODE2,
                    yaml=upstream_template.format(
                        upstream_id=ctyped_vertical_name,
                        host_name="{}.{}".format(ctype, self.Parameters.vertical_name),
                        backend_id=ctyped_vertical_name,
                    ),
                )
            )
        )

        try:
            logging.debug("Start create_upstream method response")
            resp_pb = upstream_stub.create_upstream(req_pb)
        except Exception as exc:
            logging.exception("Got exception %s", exc)
            return
        logging.debug("Got create_upstream method response %s", resp_pb)

    def remove_dns(self, ctyped_vertical_name, ctype):
        """
        Remove DNS for nanny service from common balancer.
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2

        if self.check_dns_existence(ctype) is None:
            logging.debug("DNS %s already removed", ctyped_vertical_name)
            return

        dns_stub = api_stub.DnsRecordServiceStub(self.awacs_client)
        req_pb = api_pb2.RemoveDnsRecordRequest()
        req_pb.id = DNS_TEMPLATE.format(ctype=ctype, hostname=self.Parameters.vertical_name)
        req_pb.namespace_id = DEFAULT_NAMESPACE
        try:
            resp_pb = dns_stub.remove_dns_record(req_pb)
        except Exception as exc:
            logging.debug(
                "Could not remove DNS %s for namespace %s, got exception: %s",
                ctyped_vertical_name,
                DEFAULT_NAMESPACE,
                exc,
            )
        else:
            logging.debug("Got remove_dns_record method response: %s", resp_pb)

    @decorators.change_exception_to_none
    def check_dns_existence(self, ctype):
        """
        Check whether DNS for nanny service exists in common balancer or not.

        :return: boolean, whether DNS for nanny service exists in common balancer or not
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2

        dns_stub = api_stub.DnsRecordServiceStub(self.awacs_client)
        req_pb = api_pb2.GetDnsRecordRequest()
        req_pb.id = DNS_TEMPLATE.format(ctype=ctype, hostname=self.Parameters.vertical_name)
        req_pb.namespace_id = DEFAULT_NAMESPACE
        return dns_stub.get_dns_record(req_pb)

    def create_dns(self, ctyped_vertical_name, ctype):
        """
        Create DNS for nanny service in common balancer.
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2
        from infra.awacs.proto import model_pb2

        if self.check_dns_existence(ctype) is not None:
            logging.debug("DNS %s already exists", ctyped_vertical_name)
            return

        dns_stub = api_stub.DnsRecordServiceStub(self.awacs_client)
        req_pb = api_pb2.CreateDnsRecordRequest()
        req_pb.meta.id = DNS_TEMPLATE.format(ctype=ctype, hostname=self.Parameters.vertical_name)
        req_pb.meta.namespace_id = DEFAULT_NAMESPACE
        req_pb.meta.comment = "Add DNS {}".format(ctyped_vertical_name)
        req_pb.meta.auth.type = model_pb2.Auth.AuthType.STAFF

        req_pb.spec.name_server.id = "in.yandex-team.ru"
        req_pb.spec.name_server.namespace_id = "infra"

        req_pb.spec.address.zone = "{ctype}.{vertical}.apphost".format(
            ctype=ctype, vertical=self.Parameters.vertical_name,
        )

        balancer_stub = api_stub.BalancerServiceStub(self.awacs_client)
        req_balancer_pb = api_pb2.ListBalancersRequest()
        req_balancer_pb.namespace_id = DEFAULT_NAMESPACE
        balancer_ids = [balancer.meta.id for balancer in balancer_stub.list_balancers(req_balancer_pb).balancers]

        req_pb.spec.address.backends.type = req_pb.spec.address.backends.BALANCERS
        req_pb.spec.address.backends.balancers.extend([
            model_pb2.DnsBackendsSelector.Balancer(id=balancer_id)
            for balancer_id in balancer_ids
        ])

        try:
            logging.debug("Start create_dns_record method response")
            resp_pb = dns_stub.create_dns_record(req_pb)
        except Exception as exc:
            logging.exception("Got exception %s", exc)
            return
        logging.debug("Got create_dns_record method response %s", resp_pb)

    def remove_backend(self, ctyped_vertical_name):
        """
        Remove backend for nanny service in common balancer,
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2

        backend_resp = self.check_backend_existence(ctyped_vertical_name)

        if backend_resp is None:
            logging.debug("Backend %s already removed", ctyped_vertical_name)
            return

        dns_stub = api_stub.BackendServiceStub(self.awacs_client)
        req_pb = api_pb2.RemoveBackendRequest()
        req_pb.id = ctyped_vertical_name
        req_pb.version = backend_resp.backend.meta.version
        req_pb.namespace_id = DEFAULT_NAMESPACE
        try:
            resp_pb = dns_stub.remove_backend(req_pb)
        except Exception as exc:
            logging.debug(
                "Could not remove backend %s for namespace %s, got exception: %s",
                ctyped_vertical_name,
                DEFAULT_NAMESPACE,
                exc,
            )
        else:
            logging.debug("Got remove_backend method response: %s", resp_pb)

    @decorators.change_exception_to_none
    def check_backend_existence(self, ctyped_vertical_name):
        """
        Check whether backend for nanny service in common balancer exists or not.

        :return: GetBackendResponse if backend for nanny service in common balancer exists else None.
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2

        backend_stub = api_stub.BackendServiceStub(self.awacs_client)
        req_pb = api_pb2.GetBackendRequest()
        req_pb.id = ctyped_vertical_name
        req_pb.namespace_id = DEFAULT_NAMESPACE
        return backend_stub.get_backend(req_pb)

    def create_backend(self, ctyped_vertical_name, ctype):
        """
        Create backend for nanny service in common balancer.
        """
        from infra.awacs.proto import api_stub
        from infra.awacs.proto import api_pb2
        from infra.awacs.proto import model_pb2

        if self.check_backend_existence(ctyped_vertical_name) is not None:
            logging.debug("Backend %s already exists", ctyped_vertical_name)
            return

        backend_stub = api_stub.BackendServiceStub(self.awacs_client)
        req_pb = api_pb2.CreateBackendRequest()
        req_pb.meta.id = ctyped_vertical_name
        req_pb.meta.namespace_id = DEFAULT_NAMESPACE
        req_pb.meta.comment = "Add backend {}".format(ctyped_vertical_name)
        req_pb.meta.auth.type = model_pb2.Auth.AuthType.STAFF
        selector = req_pb.spec.selector
        selector.type = selector.Type.YP_ENDPOINT_SETS_SD
        for location in self.Context.locations:
            selector.yp_endpoint_sets.extend([
                selector.YpEndpointSet(
                    cluster=location,
                    endpoint_set_id=SERVICE_NANNY_NAME.format(
                        ctype=ctype,
                        location=location,
                        vertical_name=self.Parameters.vertical_name,
                    ),
                    port=model_pb2.Port(policy=model_pb2.Port.Policy.KEEP),
                    weight=model_pb2.Weight(policy=model_pb2.Weight.Policy.KEEP)
                )
            ])
        try:
            logging.debug("Start create_backend method response")
            resp_pb = backend_stub.create_backend(req_pb)
        except Exception as exc:
            logging.exception("Got exception %s", exc)
            return
        logging.debug("Got create_backend method response %s", resp_pb)

    def prepare_all_nanny_services(self):
        """
        Prepare all nanny services for this apphost vertical using Context.ctypes and Context.locations.
        """
        self.set_info("Prepare all nanny services for vertical")
        for ctype in self.Context.ctypes:
            for location in self.Context.locations:
                self.prepare_nanny_service(location=location, ctype=ctype)

    def remove_all_nanny_services(self):
        """
        Remove all nanny services and yp pods for this apphost vertical using Context.ctypes and Context.locations.
        """
        logging.debug("Remove all nanny services for vertical")
        for ctype in self.Context.ctypes:
            for location in self.Context.locations:
                self.delete_nanny_service(location=location, ctype=ctype)

    def get_staff_group_id_by_abc_group_id(self, abc_group_id):
        staff_api_response = self.staff_client.get_persons(
            url="groups", params={"service.id": abc_group_id, "_fields": "id"},
        )
        staff_group_id = str(staff_api_response["result"][0]["id"])
        logging.debug("Got staff id %s for abc group id %s", staff_group_id, abc_group_id)
        return staff_group_id

    @decorators.change_exception_to_none
    def check_dashboard_existence(self):
        """
        Check dashboard existence.

        :return: boolean, whether nanny service exists or not
        """
        logging.debug("Check dashboard %s existence", self.Context.dashboard_name)
        return self.nanny_client.get_dashboard_content(dashboard_id=self.Context.dashboard_name)

    def create_dashboard(self):
        self.set_info("Create dashboard {}".format(self.Context.dashboard_name))
        services = []
        for ctype in self.Context.ctypes:
            for location in self.Context.locations:
                services.append({
                    "service_id": SERVICE_NANNY_NAME.format(
                        ctype=ctype,
                        location=location,
                        vertical_name=self.Parameters.vertical_name,
                    )}
                )
        c_info = rmc.get_component("{}_graphs".format(self.Parameters.vertical_name))
        staff_group_id = self.get_staff_group_id_by_abc_group_id(self.Parameters.abc_group_id)

        content = {
            "groups": [{"id": "stable", "services": services}],
            "labels": [],
            "managers": {
                "groups": [],
                "logins": [ROBOT_MORTY_LOGIN]
            },
            "name": self.Parameters.vertical_name,
            "owners": {
                "groups": [staff_group_id, "64819"],  # 64819 is Apphost Development administration group
                "logins": list({rm_const.ROBOT_RELEASER_USER_NAME, self.author, c_info.responsible}),
            }
        }
        logging.debug("Got content for new dashboard %s", content)
        if self.check_dashboard_existence():
            logging.debug("Dashboard %s already exists, update it.", self.Context.dashboard_name)
            self.nanny_client.update_dashboard(_id=self.Context.dashboard_name, content=content)
        else:
            logging.debug("Create dashboard %s for new services", self.Context.dashboard_name)
            self.nanny_client.create_dashboard(_id=self.Context.dashboard_name, content=content)
        self.set_info("Dashboard {} created".format(self.Context.dashboard_name))

    def delete_dashboard(self):
        """
        Delete dashboard for apphost vertical in Nanny.
        """
        if self.check_dashboard_existence() is None:
            logging.debug("There is no dashboard %s, we should not delete it", self.Context.dashboard_name)
            return
        logging.debug("Delete dashboard %s", self.Context.dashboard_name)
        self.nanny_client.delete_dashboard(_id=self.Context.dashboard_name)

    def delete_recipe(self, recipe_name):
        """
        Delete recipe for apphost vertical deployment in Nanny.

        :param recipe_name: apphost vertical recipe name
        """
        if self.nanny_client.safe_get_dashboard_recipe(
            dashboard=self.Context.dashboard_name,
            recipe_name=recipe_name,
        ) is None:
            logging.debug("There is no recipe %s, we should not delete it", recipe_name)
            return
        logging.debug("Delete recipe %s", recipe_name)
        self.nanny_client.delete_recipe_from_dashboard(
            dashboard_id=self.Context.dashboard_name,
            recipe_id=recipe_name,
        )

    def create_recipe(self, recipe_name, manual_confirms=False):
        """
        Create recipe for apphost vertical deployment in Nanny.

        :param recipe_name: apphost vertical recipe name
        :param manual_confirms: True for setting manual confirm flag in recipe, False otherwise
        """
        from app_host.conf.recipes.lib import RecipeBuilder

        self.set_info("Create recipe {}".format(recipe_name))
        schema = {
            "name": recipe_name,
            "groups": []
        }

        for ctype in self.Context.ctypes:
            services = [
                SERVICE_NANNY_NAME.format(
                    ctype=ctype,
                    location=location,
                    vertical_name=self.Parameters.vertical_name,
                ) for location in self.Context.locations
            ]
            if ctype == apphost_vertical.Ctypes.prod.value:
                schema["groups"].append({
                    "alias": ctype,
                    "sequence": [
                        {
                            "service": service,
                            "confirm": manual_confirms
                        } for service in services
                    ]
                })
            else:
                schema["groups"].append({
                    "alias": ctype,
                    "wait_for": [apphost_vertical.Ctypes.prod.value],
                    "sequence": [
                        {
                            "services": services,
                            "confirm": manual_confirms
                        }
                    ]
                })

        content = RecipeBuilder(schema).build()

        logging.debug("Got content for new recipe %s", content)
        if self.nanny_client.safe_get_dashboard_recipe(
            dashboard=self.Context.dashboard_name,
            recipe_name=recipe_name,
        ) is not None:
            logging.debug("Recipe %s already exists, update recipe", recipe_name)
            self.nanny_client.update_recipe_in_dashboard(
                dashboard=self.Context.dashboard_name,
                id=recipe_name,
                content=content,
            )
        else:
            logging.debug("Create recipe %s", recipe_name)
            logging.debug(self.nanny_client.create_recipe_in_dashboard(
                dashboard=self.Context.dashboard_name,
                id=recipe_name,
                content=content,
            ))
        self.set_info("Recipe {} created".format(recipe_name))

    def deploy_recipe(self, recipe_name):
        """
        Deploy recipe in Nanny by its name.

        :param recipe_name: recipe name to deploy
        """
        INIT_WAIT_TIME_MINUTES = 5
        SLEEP_INTERVAL_MINUTES = 10
        TOTAL_WAITING_TIME_MINUTES = len(rm_const.ALL_LOCATIONS) * 60 * 2  # Because HBF deploys every 3 hours

        self.set_info(
            "Deploy recipe {recipe_name} in dashboard {dashboard_name}".format(
                recipe_name=recipe_name,
                dashboard_name=self.Context.dashboard_name,
            )
        )
        deploy_response = self.nanny_client.deploy_dashboard(
            dashboard=self.Context.dashboard_name,
            recipe=recipe_name,
            use_latest_snapshot=True,
        )
        logging.debug("Got deploy dashboard response: %s", deploy_response)
        task_group_id = deploy_response["taskgroup_id"]
        self.set_info("Wait dashboard to deploy for {} minutes. Could last for 3 hours".format(INIT_WAIT_TIME_MINUTES))

        @apphost_vertical.waiter(
            init_wait_time_sec=INIT_WAIT_TIME_MINUTES * 60,
            wait_time_cycle_sec=SLEEP_INTERVAL_MINUTES * 60,
            total_wait_time_cycles_sec=TOTAL_WAITING_TIME_MINUTES * 60,
        )
        def wait_func():
            task_groups = self.nanny_client.get_dashboard_taskgroups(
                dashboard=self.Context.dashboard_name,
                statuses=None,
            )
            deployment_status = None
            for task_group in task_groups:
                if task_group["id"] == task_group_id:
                    deployment_status = task_group["status"]
                    break
            if deployment_status is None:
                eh.check_failed("Could not find task_group for id {}, failing".format(task_group_id))
            logging.debug("Got deployment status %s", deployment_status)
            if deployment_status == "DONE":
                self.set_info("Deployment finished")
                return rm_core.Ok()
            self.set_info("Sleep for {} minutes".format(SLEEP_INTERVAL_MINUTES))
            return rm_core.Error()
        wait_func()

    def prepare_dashboard(self):
        """
        Create dashboard for apphost vertical services deployment, add default and morty deployment recipes and deploy
        the default one.
        """
        self.create_dashboard()
        self.create_recipe(DEFAULT_RECIPE_NAME, manual_confirms=False)
        self.create_recipe(MORTY_RECIPE_NAME, manual_confirms=True)
        self.deploy_recipe(DEFAULT_RECIPE_NAME)

    def remove_dashboard(self):
        """
        Remove default and morty deployment recipes and the whole dashboard for apphost vertical.
        """
        self.delete_recipe(DEFAULT_RECIPE_NAME)
        self.delete_recipe(MORTY_RECIPE_NAME)
        self.delete_dashboard()

    def add_prerelease_jobs(self, trunk_revision):
        """
        Before run prerelease process we should wait for all prerelease jobs appear in Testenv trunk database.
        """
        logging.debug("Add all prerelease jobs")
        prerelease_jobs_names = [
            rm_const.JobTypes.rm_job_name(
                job.job_params["job_type"],
                self.c_info.name,
                job.job_params.get("job_name_parameter", ""),
            ) for job in self.c_info.testenv_cfg__job_graph._prerelease
        ]
        existing_jobs = set(self.te_client.get_tests(self.c_info.testenv_cfg__trunk_db))
        jobs_to_add = []
        for job in prerelease_jobs_names:
            if job not in existing_jobs:
                jobs_to_add.append(job)

        logging.debug("Got prerelease jobs %s to add", jobs_to_add)

        self.te_client.add_tests(
            db_name=self.c_info.testenv_cfg__trunk_db,
            tests_names=jobs_to_add,
            start_revision=trunk_revision,
        )
        self.te_client.start_db(db_name=self.c_info.testenv_cfg__trunk_db)

    def wait_aux_process_success(self, database):
        """
        Wait for auxiliary process successful ending.
        """
        INIT_WAIT_MINUTES = 5
        SLEEP_INTERVAL_MINUTES = 2
        MAX_WAITING_TIME_MINUTES = 300
        self.set_info("Wait {} minutes for aux process to succeed".format(INIT_WAIT_MINUTES))

        @apphost_vertical.waiter(
            init_wait_time_sec=INIT_WAIT_MINUTES * 60,
            wait_time_cycle_sec=SLEEP_INTERVAL_MINUTES * 60,
            total_wait_time_cycles_sec=MAX_WAITING_TIME_MINUTES * 60,
        )
        def wait_func():
            aux_runs = self.te_client.get_aux_runs(db=database)
            logging.debug("Got aux runs %s", aux_runs)
            if aux_runs:
                auxiliary_check = aux_runs[0]["auxiliary_check"]
                logging.debug("Got auxiliary check %s", auxiliary_check)
                if "FINISHED" not in auxiliary_check:
                    self.set_info("Sleep for {} minutes".format(SLEEP_INTERVAL_MINUTES))
                    return rm_core.Error()
                else:
                    self.set_info("Got aux process finished")
                    return rm_core.Ok()
                self.set_info("Sleep for {} minutes".format(SLEEP_INTERVAL_MINUTES))
            return rm_core.Error()
        wait_func()

    def wait_tag_to_appear_in_branch(self):
        """
        In order to run release process in branch we should wait for tag appears in branch.
        """
        INIT_WAIT_TIME_MINUTES = 5
        SLEEP_INTERVAL_MINUTES = 2
        MAX_WAITING_TIME_MINUTES = 300

        self.set_info("Wait for tag to appear in branch, init wait time 2 minutes")

        @apphost_vertical.waiter(
            init_wait_time_sec=INIT_WAIT_TIME_MINUTES * 60,
            wait_time_cycle_sec=SLEEP_INTERVAL_MINUTES * 60,
            total_wait_time_cycles_sec=MAX_WAITING_TIME_MINUTES * 60,
        )
        def wait_func():
            current_branch_num = self.c_info.last_branch_num + 1
            scopes = self.rm_client.get_scopes(component_name=self.c_info.name, start_scope_number=current_branch_num, limit=1)
            logging.debug("Got scopes %s", scopes)
            branch_scopes = scopes.get("branchScopes", [])
            if not branch_scopes:
                self.set_info("Sleep for {} minutes".format(SLEEP_INTERVAL_MINUTES))
                return rm_core.Error("No branch scopes found in Release Machine for {vertical}".format(
                    vertical=self.Parameters.vertical_name,
                ))
            tags = branch_scopes[0].get("tags", "")
            logging.debug("Got tags %s in branch %s", tags, current_branch_num)
            if not tags:
                self.set_info("Sleep for {} minutes".format(SLEEP_INTERVAL_MINUTES))
                return rm_core.Error("No tags for {branch_num} found in Release Machine for {vertical}".format(
                    branch_num=current_branch_num,
                    vertical=self.Parameters.vertical_name,
                ))
            else:
                self.set_info("Got first tag appear in branch %s", current_branch_num)
                return rm_core.Ok()
        wait_func()

    def release_vertical(self):
        """
        Run release vertical process.
        """
        component_name = "{}_graphs".format(self.Parameters.vertical_name)
        c_info = rmc.get_component(component_name)
        self.c_info = c_info
        last_trunk_revision = Arcadia.info(
            os.path.join(rm_svn.TRUNK_PATH, ARC_COMMON_PATH)
        )["commit_revision"]
        self.add_prerelease_jobs(trunk_revision=last_trunk_revision)
        self.set_info("Start prerelease process")
        response = self.rm_client.post_pre_release(
            last_trunk_revision,
            self.c_info.testenv_cfg__trunk_db,
            component_name,
            int(self.c_info.last_branch_num),
        )
        logging.debug("Got response for prerelease %s", response)
        self.set_info("Waiting for prerelease process to finish, max 15 minutes")
        self.wait_aux_process_success(database=self.c_info.testenv_cfg__trunk_db)
        self.wait_tag_to_appear_in_branch()
        first_revision_in_branch = Arcadia.log(
            self.c_info.full_branch_path(branch_num=self.c_info.last_branch_num + 1),
            revision_from="r0", revision_to="HEAD",
            stop_on_copy=True, track_copy=True, limit=1
        )[0]["revision"]
        first_branch_te_db = self.c_info.testenv_cfg__db_template.format(testenv_db_num=self.c_info.last_branch_num + 1)
        self.set_info("Start release process")
        response = self.rm_client.post_release(
            revision=first_revision_in_branch,
            testenv_db=first_branch_te_db,
            component_name=component_name,
            release_type=rm_const.ReleaseStatus.stable,
            release_item="{}__{}".format(self.c_info.name, rm_const.ReleaseStatus.stable).upper(),
            release_notes="First release",
            scope_number=int(self.c_info.last_scope_num),
            tag_number=1,
        )
        logging.debug("Got response for release %s", response)
        self.wait_aux_process_success(database=first_branch_te_db)
        self.set_info("Last branch released")

    def prepare_abc_service_slug(self):
        service_info = self.abc_client.get_service_info(service_id=self.Parameters.abc_group_id)
        self.Context.abc_slug = service_info["slug"]

    def on_execute(self):
        super(CreateNannyServicesForApphostVertical, self).on_execute()

        import nanny_rpc_client
        from infra.nanny.yp_lite_api.py_stubs import pod_sets_api_stub
        from infra.nanny.yp_lite_api.py_stubs import endpoint_sets_api_stub
        from library.python.vault_client.client import VaultClient
        from library.python.vault_client.instances import VAULT_PRODUCTION_API
        import saas.library.python.racktables.project_networks as prj_networks

        self._token = rm_sec.get_rm_token(self)
        self.abc_client = abc_client.AbcClient(self._token)
        self.prepare_abc_service_slug()
        self.nanny_client = nanny_client.NannyClient(rm_const.Urls.NANNY_BASE_URL, self._token)
        self.prj_networks_api = prj_networks.ProjectNetworksApi(self._token)
        self.pod_sets_rpc_client = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(
            nanny_rpc_client.RetryingRpcClient(rpc_url=POD_SETS_NANNY_URL, oauth_token=self._token)
        )
        self.endpoint_sets_rpc_client = endpoint_sets_api_stub.YpLiteUIEndpointSetsServiceStub(
            nanny_rpc_client.RetryingRpcClient(rpc_url=ENDPOINT_SETS_NANNY_URL, oauth_token=self._token)
        )

        self.awacs_client = nanny_rpc_client.RetryingRpcClient(rpc_url=AWACS_API_URL, oauth_token=self._token)
        self.staff_client = staff_helper.StaffApi(token=self._token)
        self.vault_client = VaultClient(
            host=VAULT_PRODUCTION_API,
            decode_files=True,
            authorization="OAuth {}".format(self._token),
        )
        self.te_client = TEClient(oauth_token=self._token)
        self.rm_client = client.RMClient()

        Arcadia.checkout(os.path.join(rm_svn.TRUNK_PATH, "arcadia"), ARC_COMMON_PATH, depth="immediates")
        Arcadia.update(os.path.join(ARC_COMMON_PATH, CREATE_NANNY_SERVICES_TASK_DIR), set_depth="files", parents=True)
        vert_name_upper = self.Parameters.vertical_name.upper()
        self.Context.dashboard_name = "apphost_{vertical}".format(vertical=self.Parameters.vertical_name)

        self.RESOURCE_TYPE_TO_FILE_NAME = {
            "STATBOX_PUSHCLIENT": {"name": "push-client", "task": "BUILD_STATBOX_PUSHCLIENT"},
            "APP_HOST_DAEMON_EXECUTABLE": {"name": "app_host", "task": "BUILD_APP_HOST_EXECUTABLES"},
            "APP_HOST_INSTANCECTL_CONF": {"name": "instancectl.conf", "task": "BUILD_APP_HOST_EXECUTABLES"},
            "APP_HOST_EVLOGDUMP_EXECUTABLE": {"name": "evlogdump", "task": "BUILD_APP_HOST_EXECUTABLES"},
            "APP_HOST_GRPC_CLIENT_EXECUTABLE": {"name": "grpc_client", "task": "BUILD_APP_HOST_EXECUTABLES"},
            "APP_HOST_SERVANT_CLIENT_EXECUTABLE": {"name": "servant_client", "task": "BUILD_APP_HOST_EXECUTABLES"},
            "APP_HOST_STABLE_BRANCH_{vertical}".format(vertical=vert_name_upper): {
                "name": "stable_branch", "task": "BUILD_HORIZON_AGENT_CONFIG"
            },
            "APP_HOST_CONFIG_BUNDLE_UNIFIED": {"name": "configs.tar.gz", "task": "BUILD_APP_HOST_EXECUTABLES"},
            "APP_HOST_HTTP_ROUTING_CONFIG_{vertical}".format(vertical=vert_name_upper): {
                "name": "routing_config.json", "task": "BUILD_HORIZON_AGENT_CONFIG"
            },
            "APP_HOST_REQUIRED_GRAPHS_{vertical}".format(vertical=vert_name_upper): {
                "name": "required_graphs.txt", "task": "BUILD_HORIZON_AGENT_CONFIG"
            },
            "HORIZON_AGENT_BINARY": {"name": "horizon_agent", "task": "BUILD_APP_HOST_EXECUTABLES"},
        }

        self.release_vertical()
        self.update_apphost_combinator()
        self.create_all_macros_for_yp_pods()
        self.prepare_all_nanny_services()
        self.prepare_balancer_upstream()
        self.prepare_dashboard()

        self.set_info("All services created")
        self.set_info(
            "Now you can try `http://production.{}.apphost.in.yandex-team.ru/ping` in incognito tab".format(
                self.Parameters.vertical_name
            )
        )

    def on_failure(self, prev_status):
        if not self.Parameters.cleanup_on_error:
            super(CreateNannyServicesForApphostVertical, self).on_failure(prev_status)
            return
        self.remove_dashboard()
        self.remove_balancer_upstream()
        self.remove_all_nanny_services()
        self.delete_all_macros_for_yp_pods()
        super(CreateNannyServicesForApphostVertical, self).on_failure(prev_status)
