import logging
import time
import json

import sandbox.common.rest as sb_rest
import sandbox.common.types.task as ctt
import sandbox.projects.abc.client as abc_client
import sandbox.projects.app_host.vertical_by_button.CreateNewRmForApphostVertical as create_new_rm
import sandbox.projects.app_host.vertical_by_button.CreateNannyServicesForApphostVertical as create_nanny_services
import sandbox.projects.common.link_builder as lb
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.tasks.ReleaseMachineConfigCrawler as release_machine_config_crawler
import sandbox.sdk2 as sdk2
from sandbox.projects.app_host.vertical_by_button.mixins import apphost_vertical
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.testenv_client import api_client as te_api_client
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine import security as rm_sec
from sandbox.sandboxsdk import sandboxapi


ARC_COMMON_PATH = "arcadia"
ROBOT_TESTENV = "robot-testenv"


class CreateNewApphostVertical(apphost_vertical.ApphostVerticalBaseTask):
    class Parameters(apphost_vertical.ApphostVerticalBaseTask.Parameters):
        kill_timeout = 11 * 60 * 60  # 11 hours
        use_temporary_quota = sdk2.parameters.Bool("Use temporary quota in YP", default=False)
        quota_abc_group_id = sdk2.parameters.String("ABC group id or slug for use in service", required=True)

    class Context(apphost_vertical.ApphostVerticalBaseTask.Context):
        locations = []
        ctypes = []
        quota_abc_group_id = ""

    def wait_task(self, task, total_wait_time_minutes=7):
        FIRST_WAITING_TIME_MINUTES = 2
        self.set_info(
            "Wait task {task_link} to finish for {wait_time} minutes".format(
                task_link=lb.task_link(task.id, task.type),
                wait_time=FIRST_WAITING_TIME_MINUTES,
            ),
            do_escape=False,
        )

        @apphost_vertical.waiter(
            init_wait_time_sec=FIRST_WAITING_TIME_MINUTES * 60,
            wait_time_cycle_sec=20,
            total_wait_time_cycles_sec=total_wait_time_minutes * 60,
        )
        def wait_func():
            task_status = rm_utils.get_task_field(task.id, "status", self.server)
            logging.debug("Got status %s", task_status)
            if task_status in ctt.Status.Group.SUCCEED:
                logging.debug("Task finished")
                return rm_core.Ok()
            elif task_status in ctt.Status.Group.BREAK or task_status in ctt.Status.Group.SCHEDULER_FAILURE:
                logging.debug("Got status %s, failing", task_status)
                eh.check_failed(
                    "Got unexpected task status {status} in task {id}".format(status=task_status, id=task.id)
                )
            return rm_core.Error()
        wait_func()

    def wait_config_to_deploy(self, committed_revision):
        """
        Wait while task RELEASE_MACHINE_CONFIG_CRAWLER become released on revision after new config commit revision.

        :param committed_revision: new RM config commit svn revision
        """

        FIRST_WAITING_TIME_MINUTES = 5
        SECOND_WAITING_TIME_MINUTES = 0
        SLEEP_INTERVAL_MINUTES = 2

        # 40 minutes to wait RELEASE_MACHINE_CONFIG_CRAWLER become released on new revision
        TOTAL_WAITING_TIME_MINUTES = 40

        self.set_info(
            "Wait config to deploy in Sandbox for {} minutes. "
            "Probably it could be longer (i.e. 15 minutes) because of SB slowdown".format(
                FIRST_WAITING_TIME_MINUTES,
            )
        )

        @apphost_vertical.waiter(
            init_wait_time_sec=FIRST_WAITING_TIME_MINUTES * 60,
            wait_time_cycle_sec=SLEEP_INTERVAL_MINUTES * 60,
            total_wait_time_cycles_sec=TOTAL_WAITING_TIME_MINUTES * 60,
        )
        def wait_func(task_type):
            attrs = {
                "released": "stable",
                "task_type": task_type,
            }
            last_released_res = sdk2.Resource.find(
                type=sdk2.service_resources.SandboxTasksBinary,
                arch=sandboxapi.ARCH_LINUX,
                attrs=attrs
            ).order(-sdk2.Resource.id).first()
            build_arc_url = rm_utils.get_input_or_ctx_field(last_released_res.task_id, "arcadia_url")
            build_revision = None
            if "@" in build_arc_url:
                build_revision = build_arc_url.split("@")[1]
            else:
                ci_context = json.loads(rm_utils.get_ctx_field(last_released_res.task_id, rm_const.CI_CONTEXT_KEY))
                build_revision = ci_context.get("revision").get("number")
                if build_revision is None:
                    logging.error(
                        "Cannot find revision from arcadia_url input and from CI context for task %s",
                        task_type,
                    )
                    return rm_core.Error()
            logging.debug("Got build revision for task %s: %s", task_type, build_revision)
            if int(build_revision) >= int(committed_revision):
                logging.debug("Got released revision %s, which is greater than %s", build_revision, committed_revision)
                return rm_core.Ok()
            return rm_core.Error()

        wait_func(task_type="RELEASE_MACHINE_CONFIG_CRAWLER")
        wait_func(task_type="CREATE_NANNY_SERVICES_FOR_APPHOST_VERTICAL")

        self.set_info(
            "Wait config to deploy in Testenv for {} minutes. "
            "Probably it could be longer (i.e. 45 minutes) because of Testenv slowdown".format(
                FIRST_WAITING_TIME_MINUTES,
            )
        )
        oauth_token = rm_sec.get_rm_token(self)
        testenv_api_client = te_api_client.TestenvApiClient(oauth_token)

        @apphost_vertical.waiter(
            init_wait_time_sec=SECOND_WAITING_TIME_MINUTES * 60,
            wait_time_cycle_sec=SLEEP_INTERVAL_MINUTES * 60,
            total_wait_time_cycles_sec=TOTAL_WAITING_TIME_MINUTES * 60,
        )
        def wait_testenv_func():
            engine_revision = testenv_api_client.system_info[""].GET()["version"]
            logging.debug("Got engine revision for Testenv: %s", engine_revision)
            if int(engine_revision) >= int(committed_revision):
                logging.debug("Got released revision %s, which is greater than %s", engine_revision, committed_revision)
                return rm_core.Ok()
            return rm_core.Error()

        wait_testenv_func()
        time.sleep(10 * 60)  # Sleep 10 minutes, https://st.yandex-team.ru/DEVTOOLSSUPPORT-5829#600a9b66bf7c9b3bc835e7f2

    def create_nanny_services_for_apphost_vertical(self):
        create_nanny_services_task = create_nanny_services.CreateNannyServicesForApphostVertical(
            self,
            priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH),
            vertical_name=self.Parameters.vertical_name.lower(),
            responsible=self.Parameters.responsible,
            startrek_queue=self.Parameters.startrek_queue,
            trunk_task_owner=self.Parameters.trunk_task_owner,
            use_temporary_quota=self.Parameters.use_temporary_quota,
            abc_group_id=self.Context.quota_abc_group_id,
            locations=self.Context.locations,
            cleanup_on_error=self.Parameters.cleanup_on_error,
            owner=self.owner,
            binary_executor_release_type=rm_core.rm_const.ReleaseStatus.stable,
        )
        if apphost_vertical.Ctypes.hamster.value in self.Context.ctypes:
            create_nanny_services_task.Parameters.hamster = True
        create_nanny_services_task.save().enqueue()

        self.set_info(
            "Run {task_link}. "
            "This task creates nanny services, section for service in common apphost balancer and deploys new service. "
            "This is the most long-running task (about 4 hours), in order to be aware of its progress please "
            "check it {task_link}".format(
                task_link=lb.task_link(create_nanny_services_task.id, create_nanny_services_task.type)
            ),
            do_escape=False,
        )
        self.wait_task(
            task=create_nanny_services_task,
            total_wait_time_minutes=2 * len(rm_const.ALL_LOCATIONS) * 60,
        )

    def run_new_rm_component_creator(self):
        create_new_rm_task = create_new_rm.CreateNewRmForApphostVertical(
            self,
            priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH),
            vertical_name=self.Parameters.vertical_name.lower(),
            responsible=self.Parameters.responsible,
            startrek_queue=self.Parameters.startrek_queue,
            trunk_task_owner=self.Parameters.trunk_task_owner,
            locations=self.Context.locations,
            owner=self.owner,
            responsible_abc_duty_slugs=self.Parameters.responsible_abc_duty_slugs,
            responsible_abc_group_id=self.Parameters.responsible_abc_group_id,
            responsible_abc_role_scopes=self.Parameters.responsible_abc_role_scopes,
        )
        if apphost_vertical.Ctypes.hamster.value in self.Context.ctypes:
            create_new_rm_task.Parameters.hamster = True
        create_new_rm_task.save().enqueue()

        self.set_info(
            "Run {task_link}. "
            "This task creates review with all neccessary configs - RM, Woland and commits it".format(
                task_link=lb.task_link(create_new_rm_task.id, create_new_rm_task.type)
            ),
            do_escape=False,
        )
        self.wait_task(task=create_new_rm_task)
        create_new_rm_task.reload()
        return create_new_rm_task

    def run_rm_crawler(self):
        rm_crawler_task = release_machine_config_crawler.ReleaseMachineConfigCrawler(
            self,
            priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH),
            component_name_filter="{}$".format(self.component_name),
            owner=self.owner,
        )
        rm_crawler_task.save().enqueue()
        self.set_info(
            "Run {task_link}. "
            "This task adds info about new component in RM UI, "
            "launches task PREPARE_RM_COMPONENT_ENVIRONMENT (creates component trunk database in Testenv, etc.)".format(
                task_link=lb.task_link(rm_crawler_task.id, rm_crawler_task.type)
            ),
            do_escape=False,
        )
        self.wait_task(task=rm_crawler_task)

    def create_new_rm_for_apphost_vertical(self):
        create_new_rm_task = self.run_new_rm_component_creator()
        review_id = create_new_rm_task.Context.review_id

        if review_id:
            self.set_info(
                "Got review {review_link} for new RM config".format(review_link=lb.review_link(review=review_id)),
                do_escape=False,
            )
            committed_revision = self.skip_check_in_review_and_commit(review_id=review_id)
            self.wait_config_to_deploy(committed_revision=committed_revision)
        self.run_rm_crawler()

    def check_robot_in_abc_group(self):
        abc_commiters = self.abc_helper.get_people_from_service(self.Context.quota_abc_group_id)
        if rm_core.rm_const.ROBOT_RELEASER_USER_NAME not in abc_commiters:
            return ["Robot {robot} should be added into abc_group {abc_group_id}".format(
                robot=rm_core.rm_const.ROBOT_RELEASER_USER_NAME,
                abc_group_id=self.Context.quota_abc_group_id,
            )]
        return []

    def check_robots_in_sandbox_group(self):
        group_info = self.sb_client.group[self.Parameters.trunk_task_owner].read()
        errors = []
        if group_info is None:
            errors.append("Could not find sandbox group {group}".format(group=self.Parameters.trunk_task_owner))
        for robot in [rm_core.rm_const.ROBOT_RELEASER_USER_NAME, ROBOT_TESTENV]:
            if robot not in group_info["members"]:
                errors.append("Robot {robot} does not belong to group {group}. Please add it in this group".format(
                    robot=robot,
                    group=self.Parameters.trunk_task_owner,
                ))
        return errors

    def check_vault_token_shared(self):
        vault_info = self.sb_client.vault.read(
            name=rm_core.rm_const.COMMON_TOKEN_NAME, owner=rm_core.rm_const.COMMON_TOKEN_OWNER, limit=1
        )["items"][0]
        if self.Parameters.trunk_task_owner not in vault_info["shared"]:
            vault_update = {
                "data": sdk2.Vault.data(
                    owner_or_name=rm_core.rm_const.COMMON_TOKEN_OWNER,
                    name=rm_core.rm_const.COMMON_TOKEN_NAME,
                ),
                "description": vault_info["description"],
                "name": vault_info["name"],
                "owner": vault_info["owner"],
                "shared": vault_info["shared"]
            }
            vault_update["shared"].append(self.Parameters.trunk_task_owner)
            try:
                self.sb_client.vault[vault_info["id"]] = vault_update
            except Exception as exc:
                eh.log_exception("Got exception while updating SB Vault", exc)
                return [
                    "Sandbox group {group} should be added into vault {token_name}. "
                    "Ask Release Machine admins to share this token with this group.".format(
                        group=self.Parameters.trunk_task_owner,
                        token_name=rm_core.rm_const.COMMON_TOKEN_NAME,
                    )
                ]
        return []

    def check_access(self):
        errors = self.prepare_abc_service_id()
        errors += self.check_robot_in_abc_group()
        errors += self.check_robots_in_sandbox_group()
        errors += self.check_vault_token_shared()
        if errors:
            eh.check_failed("\n".join(errors))

    def prepare_abc_service_id(self):
        if self.Parameters.quota_abc_group_id.isdigit():
            service_info = self.abc_helper.get_service_info(service_id=self.Parameters.quota_abc_group_id)
        else:
            service_info = self.abc_helper.get_service_info_by_slug(service_slug=self.Parameters.quota_abc_group_id)
        if service_info is None:
            return [
                "Couldn't get access to abc with slug {service_slug}, please fix it".format(
                    service_slug=self.Parameters.quota_abc_group_id,
                ),
            ]
        self.Context.quota_abc_group_id = str(service_info["id"])
        return []

    def on_execute(self):
        super(CreateNewApphostVertical, self).on_execute()
        self.abc_helper = abc_client.AbcClient(rm_sec.get_rm_token(self))
        self.sb_client = sb_rest.Client()
        self.check_access()

        self.create_new_rm_for_apphost_vertical()
        self.create_nanny_services_for_apphost_vertical()
