from collections import OrderedDict
import logging
import jinja2
import os
import shutil
import yaml

import sandbox.projects.common.error_handlers as eh
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.common.decorators as decorators
import sandbox.sdk2 as sdk2

from sandbox.projects.app_host.vertical_by_button.mixins import apphost_vertical
from sandbox.sdk2.vcs.svn import Arcadia


RESOURCES_DIR_PATH = "sandbox/projects/horizon/resources"
RESOURCES_FILE_TEMPLATE_NAME = "resources_file_template.jinja2"
VERTICAL_SETTINGS_FILE_TEMPLATE_NAME = "_vertical_settings.jinja2"
CONFIG_FILE_TEMPLATE_NAME = "config_template.jinja2"
BUILD_HORIZON_AGENT_CONFIG_PATH = "sandbox/projects/horizon/BuildHorizonAgentConfig/__init__.py"
ARC_COMMON_PATH = "arcadia"
TEST_RM_COMPONENTS_FILE_PATH = "sandbox/projects/release_machine/tests/test_release_machine_components.py"
APPHOST_CONFIGS_DIR = "sandbox/projects/release_machine/components/configs/apphost_verticals"
CREATE_NEW_RM_TASK_DIR = "sandbox/projects/app_host/vertical_by_button/CreateNewRmForApphostVertical"
WOLAND_PANELS_DIR = "search/tools/woland/panels/apphost/"
VERTICAL_SETTINGS_FILE_NAME = "_vertical_settings.json"
REQUIRED_GRAPHS_FILE_NAME = "_required_graphs.txt"
ROUTING_CONFIG_FILE_NAME = "_routing_config.json"
WOLAND_SERVER_FILE_NAME = "server.yaml"
PING_GRAPH_FILE_NAME = "ping.json"


def ordered_load(stream, Loader=yaml.Loader, object_pairs_hook=OrderedDict):
    class OrderedLoader(Loader):
        pass

    def construct_mapping(loader, node):
        loader.flatten_mapping(node)
        return object_pairs_hook(loader.construct_pairs(node))

    OrderedLoader.add_constructor(
        yaml.resolver.BaseResolver.DEFAULT_MAPPING_TAG,
        construct_mapping,
    )
    return yaml.load(stream, OrderedLoader)


def ordered_dump(data, stream=None, Dumper=yaml.Dumper, **kwds):
    class OrderedDumper(Dumper):
        pass

    def _dict_representer(dumper, data):
        return dumper.represent_mapping(
            yaml.resolver.BaseResolver.DEFAULT_MAPPING_TAG,
            data.items(),
        )

    OrderedDumper.add_representer(OrderedDict, _dict_representer)
    return yaml.dump(data, stream, OrderedDumper, **kwds)


class CreateNewRmForApphostVertical(apphost_vertical.ApphostVerticalBaseTask):
    class Parameters(apphost_vertical.ApphostVerticalBaseTask.Parameters):
        with sdk2.parameters.Output():
            review_id = sdk2.parameters.String("Created review")

    class Context(apphost_vertical.ApphostVerticalBaseTask.Context):
        review_id = None

    @decorators.memoized_property
    def ya_root(self):
        return os.path.realpath(".")

    @decorators.memoized_property
    def deploy_service(self):
        return "production_app_host_{location}_{vertical_name}".format(
            location=self.Context.locations[0], vertical_name=self.Parameters.vertical_name
        )

    @decorators.memoized_property
    def deploy_dashboard(self):
        return "apphost_{vertical_name}".format(vertical_name=self.Parameters.vertical_name)

    def on_execute(self):
        super(CreateNewRmForApphostVertical, self).on_execute()

        self._check_vertical_name_correctness()
        self._checkout_and_update_arcadia()
        self._add_new_resources()
        self._update_woland_config()
        self._add_new_config()

        commit_message = "New config for component `{comp_name_camel_case}`".format(
            comp_name_camel_case=self.comp_name_camel_case,
        )
        self.set_info("Create review with all necessary files for RM, Apphost and Woland")
        review_id = self._create_review(commit_message=commit_message)

        self.Parameters.review_id = review_id
        self.Context.review_id = review_id
        self.Context.save()

    def _generate_woland_panel(self):
        panel_config = {
            "prj": [self.vertical_name_snake_case],
            "itype": ["apphost"],
            "geo": list(set([
                location if location not in rm_const.AUX_LOCATIONS else "msk" for location in self.Context.locations
            ])),
            "ctype": ["prod", "prestable"],
            "include_monitors": ["apphost/verticals/_snippets/generic"]
        }

        panel_path = "apphost/verticals/" + self.vertical_name_snake_case
        panel_full_path = self._arc_path(
            WOLAND_PANELS_DIR, "verticals", "{}.yaml".format(self.vertical_name_snake_case),
        )

        with open(panel_full_path, "w") as woland_panel_file:
            woland_panel_file.write(ordered_dump(panel_config, Dumper=yaml.SafeDumper, default_flow_style=False))

        Arcadia.add(panel_full_path, force=True)

        return panel_path

    def _update_woland_config(self):
        Arcadia.update(
            self._arc_path(WOLAND_PANELS_DIR),
            set_depth="infinity",
            parents=True,
        )

        panel_path = self._generate_woland_panel()

        with open(
            self._arc_path(WOLAND_PANELS_DIR, WOLAND_SERVER_FILE_NAME), "r",
        ) as server_file:
            server_config = ordered_load(server_file.read())
        if not any([nanny_service["prj"] == self.vertical_name_snake_case for nanny_service in server_config["nanny"]]):
            server_config["nanny"].append(
                {
                    "prj": str(self.vertical_name_snake_case),
                    "service": "production_app_host_{{geo}}_{vertical}".format(vertical=self.vertical_name_snake_case),
                }
            )

        if panel_path not in server_config["services"]:
            server_config["services"].append(panel_path)

        with open(
            self._arc_path(WOLAND_PANELS_DIR, WOLAND_SERVER_FILE_NAME), "w",
        ) as server_file:
            server_file.write(ordered_dump(server_config, Dumper=yaml.SafeDumper, default_flow_style=False))

    def _check_vertical_name_correctness(self):
        for symbol in self.Parameters.vertical_name:
            if symbol not in rm_const.APPHOST_VERTICAL_NAME_VALID_SYMBOLS:
                eh.check_failed(
                    "Got forbidden symbol in component_name [{}]: '{}'. "
                    "Allowed only lower english letters, numbers".format(self.Parameters.vertical_name, symbol)
                )

    @staticmethod
    def _add_file_in_ya_make(dir_path_local, file_name):
        """
        Add new file in PY_SRCS in ya.make file in directory
        :param dir_path_local: directory where new file is added
        :param file_name: new file to add in ya.make
        """
        ya_make_file_path = os.path.join(dir_path_local, "ya.make")
        logging.debug("Updating %s", ya_make_file_path)
        with open(ya_make_file_path, "r") as ya_make_file:
            ya_make_lines = ya_make_file.readlines()
        inside_py_srcs = False
        should_add_file_name = True
        ya_make_lines_updated = []
        py_srcs_file_line = "    {}\n".format(file_name)
        if py_srcs_file_line in ya_make_lines:
            logging.debug("We have already added import to %s", ya_make_file_path)
            return
        for line in ya_make_lines:
            if line == "PY_SRCS(\n":
                inside_py_srcs = True
            if inside_py_srcs and line == ")\n":
                if should_add_file_name:
                    ya_make_lines_updated.append("    {}\n".format(file_name))
                    should_add_file_name = False
                inside_py_srcs = False
            if inside_py_srcs and should_add_file_name:
                curr_file_name = line.strip()
                if curr_file_name > file_name:
                    ya_make_lines_updated.append("    {}\n".format(file_name))
                    should_add_file_name = False
            ya_make_lines_updated.append(line)
        with open(ya_make_file_path, "w") as ya_make_file:
            ya_make_file.writelines(ya_make_lines_updated)

    def _update_test_rm_components_file(self):
        """
        It's neccessary to add resources for new component in test_release_machine_components.py
        """
        logging.info("Updating test_release_machine_components.py file")
        test_rm_components_file_path = os.path.join(ARC_COMMON_PATH, TEST_RM_COMPONENTS_FILE_PATH)
        Arcadia.update(test_rm_components_file_path, parents=True)
        with open(os.path.join(self.ya_root, test_rm_components_file_path), "r") as test_rm_file:
            test_rm_file_lines = test_rm_file.readlines()
        test_rm_file_lines_updated = []
        resources_file_name = "{comp_name_snake_case}_resources".format(
            comp_name_snake_case=self.comp_name_snake_case,
        )
        line_to_add = "from sandbox.projects.horizon.resources import {}  # noqa: UnusedImport\n".format(
            resources_file_name,
        )
        if line_to_add in test_rm_file_lines:
            logging.debug("We have already added import to %s", TEST_RM_COMPONENTS_FILE_PATH)
            return

        for line in test_rm_file_lines:
            if line == "from sandbox.projects.horizon import resources as horizon_res  # noqa: UnusedImport\n":
                test_rm_file_lines_updated.append(line_to_add)
            test_rm_file_lines_updated.append(line)
        with open(os.path.join(self.ya_root, test_rm_components_file_path), "w") as test_rm_file:
            test_rm_file.writelines(test_rm_file_lines_updated)

    def _update_build_horizon_agent_config_task(self):
        """
        BuildHorizonAgentConfig task should know about new resources for apphost vertical,
        so we should add corresponding imports in task file.
        """
        logging.info("Updating BuildHorizonAgentConfig task parameters")
        build_horizon_agent_config_file_path = os.path.join(ARC_COMMON_PATH, BUILD_HORIZON_AGENT_CONFIG_PATH)
        Arcadia.update(build_horizon_agent_config_file_path, set_depth="infinity", parents=True)
        with open(os.path.join(self.ya_root, build_horizon_agent_config_file_path), "r") as build_horizon_file:
            build_horizon_lines = build_horizon_file.readlines()
        build_horizon_lines_updated = []
        comp_resources_file = "{}_resources".format(self.comp_name_snake_case)
        if "from sandbox.projects.horizon.resources import {}\n".format(comp_resources_file) in build_horizon_lines:
            logging.debug("We have already added new component to %s", BUILD_HORIZON_AGENT_CONFIG_PATH)
            return
        for line in build_horizon_lines:
            if line == "from sandbox.projects.horizon import resources\n":
                build_horizon_lines_updated.append(
                    "from sandbox.projects.horizon.resources import {}\n".format(comp_resources_file)
                )
            if line == "vertical_resource_map = {\n":
                build_horizon_lines_updated.append(line)
                build_horizon_lines_updated.append(
                    "    '{vertical_name_upper_case}': ("
                    "{comp_resources_file}.HorizonAgentConfig{vertical_name_camel_case}, "
                    "{comp_resources_file}.AppHostStableBranch{vertical_name_camel_case}"
                    "),\n".format(
                        comp_resources_file=comp_resources_file,
                        vertical_name_camel_case=self.vertical_name_camel_case,
                        vertical_name_upper_case=self.vertical_name_upper_case,
                    )
                )
                continue
            build_horizon_lines_updated.append(line)
        with open(os.path.join(self.ya_root, build_horizon_agent_config_file_path), "w") as build_horizon_file:
            build_horizon_file.writelines(build_horizon_lines_updated)

    def _add_new_config(self):
        """
        Generate RM component config for new apphost vertical
        """
        logging.info("Adding new config file")
        configs_dir_path_local = os.path.join(ARC_COMMON_PATH, APPHOST_CONFIGS_DIR)
        config_file_name = "{comp_name_snake_case}.py".format(
            comp_name_snake_case=self.comp_name_snake_case,
        )
        config_file_path = os.path.join(
            configs_dir_path_local,
            config_file_name,
        )
        Arcadia.update(configs_dir_path_local, set_depth="infinity", parents=True)

        with open(os.path.join(self.ya_root, config_file_path), "w") as config_file:
            if self.Parameters.rmci:
                self._create_new_rmci_config(config_file)
            else:
                self._create_new_rmte_config(config_file)

        config_file_path_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, APPHOST_CONFIGS_DIR, config_file_name)

        if not Arcadia.info(config_file_path_url):
            Arcadia.add(config_file_path)

        self._add_file_in_ya_make(configs_dir_path_local, config_file_name)

    def _create_new_rmte_config(self, config_file):

        logging.info("Creating new RMTE config")

        with open(self._arc_path(CREATE_NEW_RM_TASK_DIR, CONFIG_FILE_TEMPLATE_NAME), "r") as config_template:
            env = jinja2.Environment()
            config_file.writelines(
                env.from_string(config_template.read()).render({
                    "comp_name": self.component_name,
                    "comp_name_camel_case": self.comp_name_camel_case,
                    "responsible": self.Parameters.responsible,
                    "startrek_queue": self.Parameters.startrek_queue,
                    "trunk_task_owner": self.Parameters.trunk_task_owner,
                    "vertical_name_upper": self.Parameters.vertical_name.upper(),
                    "vertical_name": self.Parameters.vertical_name,
                    "deploy_service": self.deploy_service,
                    "dashboard": self.deploy_dashboard,
                })
            )
            config_file.write("\n")

    def _create_new_rmci_config(self, config_file):

        from release_machine.public.config_templates import renderer

        logging.info("Creating new RMCI config")

        config_data = renderer.render_template(
            "rmci_config_template.jinja2",
            vertical_name=self.Parameters.vertical_name,
            component_name=self.component_name,
            responsible_abc_service_name=self.Parameters.responsible_abc_group_id,
            responsible_user=self.Parameters.responsible,
            startrek_queue=self.Parameters.startrek_queue,
            sandbox_task_owner_group=self.Parameters.trunk_task_owner,
            deploy_service=self.deploy_service,
            deploy_dashboard=self.deploy_dashboard,
        )

        config_file.writelines(config_data)

    def _create_dir_for_graph_generator(self):
        logging.debug("Creating directory apphost/conf/verticals/%s", self.vertical_name_upper_case)
        Arcadia.update(
            self._arc_path(apphost_vertical.GRAPH_GENERATOR_DIR),
            set_depth="infinity",
            parents=True,
        )
        if not Arcadia.check(self.vertical_dir_url):
            Arcadia.mkdir(
                self.vertical_dir,
                parents=True,
            )

        self._add_vertical_settings_file()
        for file_name in [PING_GRAPH_FILE_NAME, REQUIRED_GRAPHS_FILE_NAME, ROUTING_CONFIG_FILE_NAME]:
            self._copy_file_to_repo(file_name)

    def _add_vertical_settings_file(self):
        vertical_settings_file_path = os.path.join(
            self.vertical_dir,
            VERTICAL_SETTINGS_FILE_NAME,
        )
        with open(os.path.join(self.ya_root, vertical_settings_file_path), "w") as vertical_settings_file:
            with open(
                self._arc_path(CREATE_NEW_RM_TASK_DIR, VERTICAL_SETTINGS_FILE_TEMPLATE_NAME),
            ) as vertical_settings_template:
                required_locations = []

                for ctype in self.Context.ctypes:
                    location_ctype = ctype
                    if ctype == apphost_vertical.Ctypes.prod.value:
                        location_ctype = "prod"
                    for location in self.Context.locations:
                        required_locations.append("\"ctype={ctype};geo={location}\"".format(
                            ctype=location_ctype,
                            location=location,
                        ))

                role_scopes = []
                for scope in self.Parameters.responsible_abc_role_scopes:
                    role_scopes.append("\"{role}\"".format(role=scope))

                duty_slugs = []
                for duty in self.Parameters.responsible_abc_duty_slugs:
                    duty_slugs.append("\"{slug}\"".format(slug=duty))

                env = jinja2.Environment()
                vertical_settings_file.writelines(
                    env.from_string(vertical_settings_template.read()).render({
                        "vertical_name": self.Parameters.vertical_name,
                        "component_name": self.component_name,
                        "required_locations": ",\n\t\t".join(required_locations),
                        "abc_slug": self.Parameters.responsible_abc_group_id,
                        "abc_role_scopes": ", ".join(role_scopes),
                        "abc_duty_slugs": ", ".join(duty_slugs),
                    })
                )
                vertical_settings_file.write("\n")
        vertical_settings_file_path_url = os.path.join(self.vertical_dir_url, VERTICAL_SETTINGS_FILE_NAME)
        if not Arcadia.check(vertical_settings_file_path_url):
            Arcadia.add(vertical_settings_file_path)

    def _copy_file_to_repo(self, file_name):
        """
        Copy one of required immutable files for apphost vertical to Arcadia
        """
        repo_file_path = os.path.join(
            self.ya_root,
            self.vertical_dir,
            file_name,
        )
        local_file_path = self._arc_path(CREATE_NEW_RM_TASK_DIR, file_name)
        shutil.copy(local_file_path, repo_file_path)
        file_path_url = os.path.join(self.vertical_dir_url, file_name)
        if not Arcadia.info(file_path_url):
            Arcadia.add(repo_file_path)

    def _add_new_resources(self):
        resources_dir_path_local = os.path.join(ARC_COMMON_PATH, RESOURCES_DIR_PATH)
        resources_file_name = "{comp_name_snake_case}_resources.py".format(
            comp_name_snake_case=self.comp_name_snake_case,
        )
        resources_file_path = os.path.join(
            resources_dir_path_local,
            resources_file_name,
        )
        Arcadia.update(resources_dir_path_local, set_depth="infinity", parents=True)
        with open(os.path.join(self.ya_root, resources_file_path), "w") as resources_file:
            resources_file_template = open(
                self._arc_path(CREATE_NEW_RM_TASK_DIR, RESOURCES_FILE_TEMPLATE_NAME), "r",
            )
            env = jinja2.Environment()
            resources_file.writelines(
                env.from_string(resources_file_template.read()).render({
                    "vertical_name_camel_case": self.vertical_name_camel_case,
                    "vertical_name_upper": self.vertical_name_upper_case,
                    "trunk_task_owner": self.Parameters.trunk_task_owner,
                })
            )
            resources_file.write("\n")
        logging.info("Add resources file to Arcadia")
        resources_file_path_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, RESOURCES_DIR_PATH, resources_file_name)
        if not Arcadia.info(resources_file_path_url):
            Arcadia.add(resources_file_path)
        self._create_dir_for_graph_generator()
        self._add_file_in_ya_make(resources_dir_path_local, resources_file_name)
        self._update_build_horizon_agent_config_task()
        self._update_test_rm_components_file()

    def _checkout_and_update_arcadia(self):
        Arcadia.checkout(os.path.join(rm_svn.TRUNK_PATH, "arcadia"), ARC_COMMON_PATH, depth="immediates")
        Arcadia.update(os.path.join(ARC_COMMON_PATH, CREATE_NEW_RM_TASK_DIR), set_depth="files", parents=True)
