import json
import logging
import os
import six
import yaml

import sandbox.sdk2 as sdk2
import sandbox.projects.app_host.vertical_by_button.CreateNannyServicesForApphostVertical as create_nanny_services_task
import sandbox.projects.app_host.vertical_by_button.CreateNewRmForApphostVertical as create_new_rm_task
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.tasks.base_task as rm_bt
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.app_host.vertical_by_button.mixins import apphost_vertical
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.nanny import client as nanny_client
from sandbox.projects.release_machine import security as rm_sec


class RemoveApphostVertical(create_nanny_services_task.CreateNannyServicesForApphostVertical):
    """
    Remove apphost vertical's dashboard, balancer upstream, nanny services, macros for yp pods and RM config.
    """

    class Parameters(rm_params.BaseReleaseMachineParameters):
        vertical_name = sdk2.parameters.String("Vertical name to remove", required=True)
        kill_timeout = 2 * len(rm_const.ALL_LOCATIONS) * 60 * 60  # 10 hours

    def revert_ya_make_resources_file_changes(self, resources_dir_path_local, resources_file_name):
        ya_make_resources_file_path = os.path.join(resources_dir_path_local, "ya.make")
        logging.debug("Updating %s, remove %s", ya_make_resources_file_path, resources_file_name)
        with open(ya_make_resources_file_path, "r") as ya_make_file:
            ya_make_lines = ya_make_file.readlines()
        resources_file_line = "    {}\n".format(resources_file_name)
        if resources_file_line in ya_make_lines:
            ya_make_lines_updated = [line for line in ya_make_lines if line != resources_file_line]
        else:
            ya_make_lines_updated = ya_make_lines
        with open(ya_make_resources_file_path, "w") as ya_make_file:
            ya_make_file.writelines(ya_make_lines_updated)

    def revert_woland_changes(self):
        logging.debug("Revert woland changes")
        Arcadia.update(
            self._arc_path(create_new_rm_task.WOLAND_PANELS_DIR),
            set_depth="infinity",
            parents=True,
        )

        with open(
            self._arc_path(create_new_rm_task.WOLAND_PANELS_DIR, create_new_rm_task.WOLAND_SERVER_FILE_NAME), "r",
        ) as server_file:
            server_config = create_new_rm_task.ordered_load(server_file.read())
        if any([nanny_service["prj"] == self.vertical_name_snake_case for nanny_service in server_config["nanny"]]):
            server_config["nanny"].remove(
                {
                    "prj": str(self.vertical_name_snake_case),
                    "service": "production_app_host_{{geo}}_{vertical}".format(vertical=self.vertical_name_snake_case),
                }
            )

        vertical_service = "apphost/verticals/{}".format(self.vertical_name_snake_case)
        if vertical_service in server_config["services"]:
            server_config["services"].remove(vertical_service)

        with open(
            self._arc_path(create_new_rm_task.WOLAND_PANELS_DIR, create_new_rm_task.WOLAND_SERVER_FILE_NAME), "w",
        ) as server_file:
            server_file.write(
                create_new_rm_task.ordered_dump(server_config, Dumper=yaml.SafeDumper, default_flow_style=False)
            )

        panel_path = self._arc_path(
            create_new_rm_task.WOLAND_PANELS_DIR,
            "verticals",
            self.vertical_name_snake_case + ".yaml",
        )
        if Arcadia.check(panel_path, look_parent=False):
            Arcadia.delete(panel_path)

    def remove_graph_dir(self, graph_generator_dir):
        Arcadia.update(
            self._arc_path(graph_generator_dir),
            set_depth="infinity",
            parents=True,
        )
        vertical_dir = self._arc_path(graph_generator_dir, self.vertical_name_upper_case)
        vertical_dir_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, graph_generator_dir, self.vertical_name_upper_case)
        if Arcadia.check(vertical_dir_url):
            logging.debug("Remove %s", vertical_dir_url)
            Arcadia.delete(vertical_dir, user=rm_const.ROBOT_RELEASER_USER_NAME)

    def revert_horizon_agent_task_changes(self):
        build_horizon_agent_config_file_path = os.path.join(
            create_new_rm_task.ARC_COMMON_PATH, create_new_rm_task.BUILD_HORIZON_AGENT_CONFIG_PATH,
        )
        Arcadia.update(build_horizon_agent_config_file_path, set_depth="infinity", parents=True)
        with open(os.path.join(self.ya_root, build_horizon_agent_config_file_path), "r") as build_horizon_file:
            build_horizon_lines = build_horizon_file.readlines()
        comp_resources_file = "{}_resources".format(self.comp_name_snake_case)
        import_resources_line = "from sandbox.projects.horizon.resources import {}\n".format(comp_resources_file)
        if import_resources_line in build_horizon_lines:
            build_horizon_lines_updated = [line for line in build_horizon_lines if line != import_resources_line]
            build_horizon_lines_updated = [
                line for line in build_horizon_lines_updated
                if not line.startswith("    '{vert}': (".format(vert=self.vertical_name_upper_case))
            ]
        else:
            build_horizon_lines_updated = build_horizon_lines
        with open(os.path.join(self.ya_root, build_horizon_agent_config_file_path), "w") as build_horizon_file:
            build_horizon_file.writelines(build_horizon_lines_updated)

    def revert_test_rm_component_file_changes(self):
        test_rm_components_file_path = os.path.join(
            create_new_rm_task.ARC_COMMON_PATH, create_new_rm_task.TEST_RM_COMPONENTS_FILE_PATH,
        )
        Arcadia.update(test_rm_components_file_path, parents=True)
        with open(os.path.join(self.ya_root, test_rm_components_file_path), "r") as test_rm_file:
            test_rm_file_lines = test_rm_file.readlines()
        resources_file_name = "{comp_name_snake_case}_resources".format(comp_name_snake_case=self.comp_name_snake_case)
        line_to_add = "from sandbox.projects.horizon.resources import {}  # noqa: UnusedImport\n".format(
            resources_file_name,
        )
        if line_to_add in test_rm_file_lines:
            test_rm_file_lines_updated = [line for line in test_rm_file_lines if line != line_to_add]
        else:
            test_rm_file_lines_updated = test_rm_file_lines
        with open(os.path.join(self.ya_root, test_rm_components_file_path), "w") as test_rm_file:
            test_rm_file.writelines(test_rm_file_lines_updated)

    def remove_vertical_rm_config(self):
        resources_dir_path_local = os.path.join(
            create_new_rm_task.ARC_COMMON_PATH, create_new_rm_task.RESOURCES_DIR_PATH,
        )
        resources_file_name = "{comp_name_snake_case}_resources.py".format(
            comp_name_snake_case=self.comp_name_snake_case,
        )
        resources_file_path = os.path.join(resources_dir_path_local, resources_file_name)
        resources_file_path_url = os.path.join(
            Arcadia.ARCADIA_TRUNK_URL, create_new_rm_task.RESOURCES_DIR_PATH, resources_file_name,
        )
        Arcadia.update(
            resources_dir_path_local,
            set_depth="infinity",
            parents=True,
        )
        if Arcadia.info(resources_file_path_url):
            logging.debug("Delete resources file %s", resources_file_name)
            Arcadia.delete(resources_file_path, user=rm_const.ROBOT_RELEASER_USER_NAME)

        self.remove_graph_dir(apphost_vertical.GRAPH_GENERATOR_DIR)
        self.revert_ya_make_resources_file_changes(resources_dir_path_local, resources_file_name)
        self.revert_horizon_agent_task_changes()
        self.revert_test_rm_component_file_changes()
        self.revert_woland_changes()

        configs_dir_path_local = os.path.join(create_new_rm_task.ARC_COMMON_PATH, create_new_rm_task.APPHOST_CONFIGS_DIR)
        config_file_name = "{comp_name_snake_case}.py".format(
            comp_name_snake_case=self.comp_name_snake_case,
        )
        config_file_path = os.path.join(
            configs_dir_path_local,
            config_file_name,
        )
        config_file_path_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, create_new_rm_task.APPHOST_CONFIGS_DIR, config_file_name)
        Arcadia.update(
            configs_dir_path_local,
            set_depth="infinity",
            parents=True,
        )
        if Arcadia.info(config_file_path_url):
            logging.debug("Delete config file %s", config_file_name)
            Arcadia.delete(config_file_path, user=rm_const.ROBOT_RELEASER_USER_NAME)

        configs_dir_path_local = os.path.join(create_new_rm_task.ARC_COMMON_PATH, create_new_rm_task.APPHOST_CONFIGS_DIR)
        self.revert_ya_make_resources_file_changes(configs_dir_path_local, config_file_name)

    def remove_generated_config(self):
        apphost_combinator_path_local = os.path.join(
            create_new_rm_task.ARC_COMMON_PATH, create_nanny_services_task.APPHOST_COMBINATOR,
        )
        Arcadia.update(
            apphost_combinator_path_local,
            set_depth="infinity",
            parents=True,
        )
        generated_vertical_files_path = os.path.join(
            apphost_combinator_path_local,
            "generated",
            self.vertical_name_upper_case,
        )
        generated_vertical_files_path_url = os.path.join(
            Arcadia.ARCADIA_TRUNK_URL,
            create_nanny_services_task.APPHOST_COMBINATOR,
            "generated",
            self.vertical_name_upper_case,
        )
        if Arcadia.info(generated_vertical_files_path_url):
            logging.debug("Delete directory %s", generated_vertical_files_path_url)
            Arcadia.delete(generated_vertical_files_path, user=rm_const.ROBOT_RELEASER_USER_NAME)

        gen_config_path = os.path.join(
            self.ya_root,
            apphost_combinator_path_local,
            create_nanny_services_task.GEN_CONFIG_PATH,
        )
        with open(gen_config_path, "r") as gen_config:
            gen_config_file = json.load(gen_config)
            gen_config_file["ctypes_mapping"].pop(self.vertical_name_upper_case, None)

        with open(gen_config_path, "w") as gen_config:
            json.dump(gen_config_file, gen_config, indent=4, sort_keys=True)

        ya_make_generated_file_path = os.path.join(apphost_combinator_path_local, "generated", "ya.make")
        with open(ya_make_generated_file_path, "r") as ya_make_file:
            ya_make_lines = ya_make_file.readlines()
        ya_make_lines_updated = []

        for line in ya_make_lines:
            if not line.strip().startswith("{vertical}/".format(vertical=self.vertical_name_upper_case)):
                ya_make_lines_updated.append(line)
        with open(ya_make_generated_file_path, "w") as ya_make_file:
            ya_make_file.writelines(ya_make_lines_updated)

        installation_config_file = os.path.join(
            apphost_combinator_path_local,
            create_nanny_services_task.INSTALLATION_CONFIG.format(vertical=self.vertical_name_snake_case),
        )
        installation_config_file_url = os.path.join(
            Arcadia.ARCADIA_TRUNK_URL,
            create_nanny_services_task.APPHOST_COMBINATOR,
            create_nanny_services_task.INSTALLATION_CONFIG.format(vertical=self.vertical_name_snake_case),
        )
        if Arcadia.info(installation_config_file_url):
            logging.debug("Delete installation file %s", installation_config_file_url)
            Arcadia.delete(installation_config_file, user=rm_const.ROBOT_RELEASER_USER_NAME)

        with open(
            os.path.join(apphost_combinator_path_local, create_nanny_services_task.APPHOST_JSON_CONFIG), "r"
        ) as json_config:
            json_config_file = json.load(json_config)
            for config_element in json_config_file:
                if isinstance(config_element, dict) and isinstance(config_element.items()[0][1], six.string_types):
                    if config_element.items()[0][1].startswith("app_host/installation/"):
                        logging.debug("Installation configs found")
                        for vertical in config_element:
                            if vertical == "+{vertical}".format(vertical=self.vertical_name_upper_case):
                                config_element.pop(vertical, None)
                                break
        with open(
            os.path.join(apphost_combinator_path_local, create_nanny_services_task.APPHOST_JSON_CONFIG), "w"
        ) as json_config:
            json.dump(json_config_file, json_config, indent=4, sort_keys=True)

    def on_enqueue(self):
        apphost_vertical.ApphostVerticalBaseTask.on_enqueue(self)

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        import nanny_rpc_client
        from infra.nanny.yp_lite_api.py_stubs import pod_sets_api_stub
        import saas.library.python.racktables.project_networks as prj_networks
        from app_host.tools.nanny_tools.lib.common import NannyTools, Arguments

        self.ya_root = os.path.realpath(".")
        self._token = rm_sec.get_rm_token(self)
        self.nanny_client = nanny_client.NannyClient(rm_const.Urls.NANNY_BASE_URL, self._token)
        self.prj_networks_api = prj_networks.ProjectNetworksApi(self._token)
        self.pod_sets_rpc_client = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(
            nanny_rpc_client.RetryingRpcClient(
                rpc_url=create_nanny_services_task.POD_SETS_NANNY_URL, oauth_token=self._token,
            )
        )
        self.awacs_client = nanny_rpc_client.RetryingRpcClient(
            rpc_url=create_nanny_services_task.AWACS_API_URL, oauth_token=self._token,
        )

        arguments = Arguments()
        arguments.confirmed = True
        arguments.category = None
        nanny = NannyTools(arguments)
        services = nanny.get_services_with_info(labels={
            "itype": "apphost",
            "prj": self.Parameters.vertical_name,
        })
        hamster = False
        for _, service in services.items():
            self.Context.locations.append(service["labels"]["geo"])
            if service["labels"]["ctype"] == apphost_vertical.Ctypes.hamster.value:
                hamster = True
        self.Context.locations = list(set(self.Context.locations))
        self.Context.ctypes = [apphost_vertical.Ctypes.prod.value]
        if hamster:
            self.Context.ctypes.append(apphost_vertical.Ctypes.hamster.value)
        self.Context.dashboard_name = "apphost_{vertical}".format(vertical=self.Parameters.vertical_name)
        self.set_info("Locations are {locations} and ctypes are {ctypes}".format(
            locations=self.Context.locations,
            ctypes=self.Context.ctypes,
        ))
        Arcadia.checkout(
            os.path.join(rm_svn.TRUNK_PATH, "arcadia"),
            create_new_rm_task.ARC_COMMON_PATH,
            depth="immediates",
        )
        self.remove_dashboard()
        self.remove_balancer_upstream()
        self.remove_all_nanny_services()
        self.delete_all_macros_for_yp_pods()

        commit_message = "Remove {vertical} vertical RM config, SB task {task_link}".format(
            vertical=self.Parameters.vertical_name,
            task_link=lb.task_link(self.id, link_name=self.id, plain=True),
        )
        self.remove_generated_config()
        self.remove_vertical_rm_config()
        if Arcadia.diff(os.path.join(self.ya_root, create_new_rm_task.ARC_COMMON_PATH)):
            review_id = self._create_review(commit_message=commit_message)
            self.skip_check_in_review_and_commit(review_id=review_id)
        else:
            self.set_info("RM config and other files have been reverted already")
