import enum
import functools
import itertools
import logging
import os
import re
import time

from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import decorators
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers import arcanum_helper
from sandbox.projects.release_machine.helpers import commit
from sandbox.sdk2.vcs.svn import Arcadia
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.projects.release_machine.input_params2 as rm_params

ARC_COMMON_PATH = "arcadia"
GRAPH_GENERATOR_DIR = "apphost/conf/verticals"


class Ctypes(enum.Enum):
    prod = "production"
    hamster = "hamster"


def waiter(
    init_wait_time_sec=0,
    wait_time_cycle_sec=10,
    total_wait_time_cycles_sec=100,
):
    def dec(func):
        @functools.wraps(func)
        def f2(*args, **kwargs):
            time.sleep(init_wait_time_sec)
            start_time = int(time.time())
            for step in itertools.count():
                result = func(*args, **kwargs)
                if result.ok:
                    return result.result
                elif int(time.time()) - start_time > total_wait_time_cycles_sec:
                    eh.check_failed("Timeout {} reached".format(total_wait_time_cycles_sec))
                logging.debug("Step %s, sleep for %s", step, wait_time_cycle_sec)
                time.sleep(wait_time_cycle_sec)
        return f2
    return dec


class ApphostVerticalBaseTask(rm_bt.BaseReleaseMachineTask):
    """Default class for tasks, related to apphost vertical creation."""

    def _arc_path(self, *subpaths):
        return os.path.join(self.ya_root, ARC_COMMON_PATH, *subpaths)

    @property
    def component_name(self):
        return "{}_graphs".format(self.Parameters.vertical_name.lower())

    @property
    def comp_name_upper_case(self):
        return self.component_name.upper()

    @property
    def vertical_name_upper_case(self):
        return self.Parameters.vertical_name.upper()

    @property
    def comp_name_snake_case(self):
        return self.component_name

    @property
    def vertical_name_snake_case(self):
        return self.Parameters.vertical_name.lower()

    @property
    def comp_name_camel_case(self):
        return self._convert_snake_case_to_camel_case(self.component_name)

    @property
    def vertical_name_camel_case(self):
        return self._convert_snake_case_to_camel_case(self.Parameters.vertical_name)

    @property
    def vertical_dir(self):
        return self._arc_path(GRAPH_GENERATOR_DIR, self.vertical_name_upper_case)

    @property
    def vertical_dir_url(self):
        return os.path.join(Arcadia.ARCADIA_TRUNK_URL, GRAPH_GENERATOR_DIR, self.vertical_name_upper_case)

    @decorators.memoized_property
    def ya_root(self):
        return os.path.realpath(".")

    def _convert_snake_case_to_camel_case(self, string_to_convert):
        string_parts = string_to_convert.split("_")
        return "".join([part.capitalize() for part in string_parts])

    def _create_review(self, commit_message):
        ssh_key = commit.get_arc_ssh_key(self)
        with ssh_key:
            created_review = None
            try:
                commit_result = Arcadia.commit(
                    ARC_COMMON_PATH,
                    commit_message,
                    user=rm_const.ROBOT_RELEASER_USER_NAME,
                    with_revprop=["arcanum:review-skip=yes", "arcanum:check=yes"],
                )
                if not commit_result:
                    self.set_info("Nothing to commit")
                    return
                logging.debug("Got commit_result while creating review: %s", commit_result)
            except Exception as exc:
                logging.exception("Got exception, while committing", exc)
                created_review = re.findall(
                    r"Check status can be monitored using this special review request: ([0-9]+)",
                    str(exc),
                )
            if created_review:
                created_review = created_review[0]
                self.set_info("Created review {review_link}".format(
                    review_link=lb.review_link(review=created_review, link_type=lb.LinkType.href)),
                    do_escape=False,
                )
                return created_review
            eh.check_failed("Can't create review, failing")

    class Parameters(rm_params.BaseReleaseMachineParameters):
        cleanup_on_error = sdk2.parameters.Bool("Delete created artifacts in case of failure", default=False)
        hamster = sdk2.parameters.Bool("Create also hamster nanny services")
        locations = sdk2.parameters.List("Locations for new services with deploy order, from first to last")
        responsible = sdk2.parameters.String("Responsible, staff nickname", required=True)
        responsible_abc_duty_slugs = sdk2.parameters.List("ABC duty scopes with responsibles for apphost", default=[])
        responsible_abc_group_id = sdk2.parameters.String("ABC group slug with responsibles for apphost", required=True)
        responsible_abc_role_scopes = sdk2.parameters.List("ABC roles with responsibles for apphost", default=[])
        startrek_queue = sdk2.parameters.String("Startrek queue for release tickets", required=True)
        trunk_task_owner = sdk2.parameters.String("Trunk task owner, sandbox group", required=True)
        vertical_name = sdk2.parameters.String("Vertical name", required=True)
        rmci = sdk2.parameters.Bool("Use RMCI (instead of RMTE)", default_value=True)

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        locations = []
        ctypes = []

    def skip_check_in_review_and_commit(self, review_id):
        """
        Disable review checks.

        :param review_id: review id
        :return: committed_revision
        """

        arcanum_api = arcanum_helper.ArcanumApi(token=self._token)
        arcanum_api.update_review_checks(
            rr_id=review_id, content={"system": "ci", "type": "large_tests", "required": "False"}
        )
        arcanum_api.update_review_checks(
            rr_id=review_id, content={"system": "ci", "type": "tests", "required": "False"}
        )
        arcanum_api.update_review_checks(
            rr_id=review_id, content={"system": "ci", "type": "build", "required": "False"}
        )
        arcanum_api.update_review_checks(
            rr_id=review_id, content={"system": "arcanum", "type": "approved", "required": "False"}
        )
        arcanum_api.enable_commit_button(rr_id=review_id)

        # sleep, because next checks can be created with unexpected delay by CI, which can cause timeout fail on merge waiting
        time.sleep(60)
        arcanum_api.update_review_checks(
            rr_id=review_id, content={"system": "CI", "type": "Arcadia Common Checks", "required": "False"}
        )
        arcanum_api.update_review_checks(
            rr_id=review_id, content={"system": "CI", "type": "autocheck: Autocheck Precommit Trunk", "required": "False"}
        )

        commit_result = commit.wait_review_for_merge(review_id=review_id, arcanum_api=arcanum_api)
        committed_revision = commit_result.revision
        if committed_revision is not None:
            logging.debug("New RM config committed as %s revision", committed_revision)
            return committed_revision
        else:
            eh.check_failed("Could not commit new RM config")

    def get_locations(self):
        """Split self.Parameters.locations into tokens and check their correctness."""
        locations = [location.lower() for location in self.Parameters.locations if location]
        logging.debug("Got locations %s", locations)
        for location in locations:
            if location not in rm_const.ALL_LOCATIONS:
                eh.check_failed(
                    "Got location {location} not possible locations {locations}".format(
                        location=location,
                        locations=rm_const.ALL_LOCATIONS,
                    )
                )
        self.Context.locations = locations

    def on_execute(self):
        super(ApphostVerticalBaseTask, self).on_execute()

        self._token = rm_sec.get_rm_token(self)
        self.Context.ctypes = [Ctypes.prod.value]
        if self.Parameters.hamster:
            self.Context.ctypes.append(Ctypes.hamster.value)
        self.get_locations()
        self.set_info("Locations are {locations} and ctypes are {ctypes}".format(
            locations=self.Context.locations,
            ctypes=self.Context.ctypes,
        ))
