# -*- coding: utf-8 -*-

import logging
import os
from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.common.arcadia.sdk import mount_arc_path
from sandbox.projects.common.build.parameters import ArcadiaUrl, ArcSecret, UseArcInsteadOfArcadiaApi
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_LXC_IMAGE
from sandbox.projects.sandbox_ci.task import ManagersTaskMixin
from sandbox.projects.sandbox_ci.utils.context import Node


class APRIL_NODE_MODULES_BUNDLE(sdk2.Resource):
    """
    Bundle with npm dependencies
    """
    auto_backup = True
    any_arch = False
    __default_attribute__ = sdk2.parameters.String


class AprilFrontendJest(ManagersTaskMixin, sdk2.Task):
    class Requirements(sdk2.Requirements):
        ram = 2 * 1024
        cores = 4

        dns = DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container(
            'Build environment',
            resource_type=SANDBOX_CI_LXC_IMAGE,
            platform='linux_ubuntu_18.04_bionic',  # necessary for correct default resource searching
            required=True,
        )
        kill_timeout = 30 * 60

        project_path = sdk2.parameters.String(
            'Path to the project where tests should be run in',
            required=True,
        )
        checkout_arcadia_from_url = ArcadiaUrl()
        arc_secret = ArcSecret()
        use_arc_instead_of_aapi = UseArcInsteadOfArcadiaApi(default=True)
        node_js_version = sdk2.parameters.String(
            'Node-js version',
            description='Override default node-js version used in LXC-container',
        )
        reuse_dependencies_cache = sdk2.parameters.Bool(
            'Reuse dependencies cache (libs, node_modules, etc)',
            default=True,
        )
        pass_with_no_tests = sdk2.parameters.Bool(
            'Pass with no tests',
            description='Allows the test suite to pass when no files are found',
        )

    class Context(sdk2.Context):
        reused_resources = []

    lifecycle_steps = {
        'npm_install_yarn': 'mkdir .build.tmp && npm i --prefix ./.build.tmp yarn@1.22',
        'yarn': './.build.tmp/node_modules/.bin/yarn install',
        'jest': './.build.tmp/node_modules/.bin/yarn jest --ci',
    }

    arcadia_path = None

    # Lifecycle manager requires the following props/methods: `project_name`, `project_dir`, `working_path`, `project_path`.
    @property
    def project_name(self):
        return 'april'

    @property
    def april_dir(self):
        if self.arcadia_path:
            return self.arcadia_path / 'april'

    @property
    def project_dir(self):
        if self.april_dir:
            return self.working_path() / self.Parameters.project_path

        return self.working_path()

    def working_path(self, *args):
        if self.april_dir:
            return self.april_dir

        return self.path(*args)

    def project_path(self, *args):
        return self.project_dir.joinpath(*args)

    def install_deps(self, target_dir, target_archive=sdk2.Path('node_modules.tar.gz'),
                     lifecycle_step='yarn'):
        deps_files = [
            self.april_dir / 'yarn.lock',
            self.april_dir / 'tools' / 'django-pog' / 'package.json',
            self.april_dir / 'web' / 'package.json',
        ]

        return self.dependencies.install_deps(
            deps_files,
            target_dir=target_dir,
            target_archive=target_archive,
            lifecycle_step=lifecycle_step,
            resource_type=APRIL_NODE_MODULES_BUNDLE,
        )

    def on_execute(self):
        if self.Parameters.pass_with_no_tests:
            self.lifecycle_steps['jest'] += ' --passWithNoTests'

        url = self.Parameters.checkout_arcadia_from_url
        use_arc = self.Parameters.use_arc_instead_of_aapi
        with mount_arc_path(url, use_arc_instead_of_aapi=use_arc) as path:
            self.arcadia_path = sdk2.Path(path)
            with Node(self.Parameters.node_js_version):
                os.environ.setdefault('NPM_CONFIG_REGISTRY', 'https://npm.yandex-team.ru')

                logging.info('Install yarn')
                self.lifecycle('npm_install_yarn')

                logging.info('Install modules')
                self.install_deps(self.april_dir / 'node_modules')

                logging.info('Run tests')
                self.lifecycle('jest')
        self.arcadia_path = None

        logging.info('The End')
