import logging
import os
import shutil

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.april.multik.resources import MultikStandFrontend
import sandbox.projects.common.arcadia.sdk as arcadiasdk
from sandbox.projects.common.vcs import aapi
from sandbox.projects.common.ya_deploy.release_integration import ReleaseToYaDeployTask2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.misc as ctm


class BuildMultikServerFrontend(ReleaseToYaDeployTask2, sdk2.Task):
    """
    Class for building MultiK frontend
    """
    YP_TOKEN_YAV_SECRET_ID = 'sec-01fy9cyrxbrkv405eja5g8ddtb'

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('10.20.1'),
            environments.GCCEnvironment(version='5.3.0'),
        ]

    class Parameters(sdk2.Parameters):
        debug = sdk2.parameters.Bool('Debug build', required=False)
        project_name = sdk2.parameters.String('Name of deploy project, that resource is built for', default=None, required=False)

        arcadia_revision = sdk2.parameters.String('Arcadia revision', default=None, required=False)
        checkout_arcadia_from_url = sdk2.parameters.String('Svn url for arc with revision', default=sdk2.svn.Arcadia.ARCADIA_TRUNK_URL, required=True)
        arcadia_patch = sdk2.parameters.String('Apply patch (text diff or rbtorrent)', multiline=True, default='')

    def _prepare_env(self):
        """
        Calling prepare for environments

        :return:
        """
        for env in self.Requirements.environments:
            env.prepare()

    def build_npm(self, target, revision=None):
        """
        Function for building NPM package. Will move to abstract class in future

        :param target: Target to build
        :param revision: arcadia revision or None for HEAD
        :return:
        """
        logger = logging.getLogger('npm_builder')

        revision = revision or aapi.ArcadiaApi.svn_head()

        logger.info('Building {} revision: {}'.format(target, revision))

        build_path = os.path.join(os.getcwd(), 'build_dir')

        with arcadiasdk.mount_arc_path(sdk2.svn.Arcadia.replace(self.Parameters.checkout_arcadia_from_url, revision=revision)) as a_arcadia_path:
            if getattr(self.Parameters, 'arcadia_patch', ''):
                arcadiasdk.apply_patch(self, a_arcadia_path, self.Parameters.arcadia_patch, self.path())
            shutil.copytree(os.path.join(a_arcadia_path, target), build_path)

        run_environment = 'development' if self.Parameters.debug else 'production'

        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            sp.check_call(
                ['npm', '--version'],
                cwd=build_path,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            sp.check_call(
                ['npm', '--registry=https://npm.yandex-team.ru', 'install', '-g', 'npm@~6.14.5'],
                cwd=build_path,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

            sp.check_call(
                ['npm', '--registry=https://npm.yandex-team.ru', 'install'],
                cwd=build_path,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

            sp.check_call(
                ['npm', 'run', 'build:{}'.format(run_environment)],
                cwd=build_path,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            result = sdk2.ResourceData(MultikStandFrontend(
                self,
                'Multik frontend version {}:{}'.format(revision, run_environment),
                'multik_frontend',
            ))
            result_dir = str(result.path)
            os.mkdir(result_dir)

            shutil.copytree(os.path.join(build_path, 'build'), os.path.join(result_dir, 'build'))
            shutil.copy(os.path.join(build_path, 'public', 'env.json'), os.path.join(result_dir, 'build', 'env.json'))

    def on_execute(self):
        self._prepare_env()
        self.build_npm('april/multik/stand/web', revision=self.Parameters.arcadia_revision)
