import contextlib
import logging
import time

import requests

import sandbox.sdk2

from sandbox.projects.common.build import YaPackage2


logger = logging.getLogger(__name__)


SAMOGON_PARAMS_GROUP_NAME = 'Samogon settings'
ARC_RELEASE_PARAMS_GROUP_NAME = 'Arc release settings'

INFRA_API_URL = 'https://infra-api.yandex-team.ru'
INFRA_SERVICE = 51  # Arcadia
INFRA_ENVIRONMENT = 806  # Arc

INFRA_MESSAGE = """\
Arc server component deploy

Name: {name}
Resource: https://sandbox.yandex-team.ru/resource/{resource}
"""


class ArcReleaseCreateInfra(sandbox.sdk2.parameters.Bool):
    name = 'arc_release_create_infra'
    description = 'Create Infra release event'
    default_value = False
    group = ARC_RELEASE_PARAMS_GROUP_NAME


class ArcReleaseComponent(sandbox.sdk2.parameters.String):
    name = 'arc_release_component'
    description = 'Arc component name'
    required = True
    group = ARC_RELEASE_PARAMS_GROUP_NAME


class ArcReleaseDuration(sandbox.sdk2.parameters.Integer):
    name = 'arc_release_duration'
    description = 'Infra event fixed duration (minutes, zero to track duration automatically)'
    required = True
    default_value = 0
    group = ARC_RELEASE_PARAMS_GROUP_NAME


ArcReleaseCreateInfra.sub_fields = {
    'true': [
        ArcReleaseDuration.name,
        ArcReleaseComponent.name,
    ]
}


class SamogonWaitDeploy(sandbox.sdk2.parameters.Bool):
    name = 'samogon_wait_deploy'
    description = 'Wait for Samogon to deploy package release'
    default_value = False
    group = SAMOGON_PARAMS_GROUP_NAME


class SamogonService(sandbox.sdk2.parameters.String):
    name = 'samogon_service'
    description = 'Samogon service name'
    required = True
    default_value = "arc2"
    group = SAMOGON_PARAMS_GROUP_NAME


class SamogonServant(sandbox.sdk2.parameters.String):
    name = 'samogon_servant'
    description = 'Samogon servant name'
    required = True
    group = SAMOGON_PARAMS_GROUP_NAME


SamogonWaitDeploy.sub_fields = {
    'true': [
        SamogonService.name,
        SamogonServant.name,
    ]
}


@contextlib.contextmanager
def create_infra_event(task, resource):
    logger.info('Creating Infra event')
    infra_session = requests.Session()
    infra_session.headers['Authorization'] = 'OAuth {}'.format(sandbox.sdk2.Vault.data('INFRA_TOKEN'))
    event_data = {
        'title': 'Arc {} release'.format(task.Parameters.arc_release_component),
        'description': INFRA_MESSAGE.format(
            name=task.Parameters.arc_release_component,
            resource=resource,
        ),
        'serviceId': INFRA_SERVICE,
        'environmentId': INFRA_ENVIRONMENT,
        'type': 'maintenance',
        'severity': 'minor',
    }
    if task.Parameters.arc_release_duration:
        event_data['duration'] = task.Parameters.arc_release_duration * 60
    logger.debug('Creating infra event: %s', event_data)
    r = infra_session.post('{}/v1/events', INFRA_API_URL, data=event_data)
    r.raise_for_status()
    event_info = r.json()
    logger.debug('Infra event created: %s', event_info)
    try:
        task.set_info('Infra event: https://infra.yandex-team.ru/event/{}'.format(event_info['id']))
        yield event_info
    finally:
        if not task.parameters.arc_release_duration:
            event_data = {'finishTime': time.time()}
            logger.debug('Finishing infra event %d: %s', event_info['id'], event_data)
            r = infra_session.put('{}/v1/events/{}'.format(INFRA_API_URL, event_info['id']), data=event_data)
            if r.status_code == requests.codes.ok:
                task.set_info('Infra event finished')
            else:
                task.set_info('Failed to finish infra event: {}'.format(r.text))


class ArcServerPackage(YaPackage2.YaPackage2):
    class Parameters(YaPackage2.YaPackage2.Parameters):
        arc_release_block = sandbox.sdk2.parameters.Info('Arc release settings')
        arc_release_create_infra = ArcReleaseCreateInfra()
        arc_release_component = ArcReleaseComponent()
        arc_release_duration = ArcReleaseDuration()

        samogon_block = sandbox.sdk2.parameters.Info('Samogon settings')
        samogon_service = SamogonService()
        samogon_servant = SamogonServant()

    def on_release(self, parameters):
        releaseable_resources = self.get_release_resources()
        assert len(releaseable_resources) == 1, 'Unsupported number of releaseable resources'
        resource_info = releaseable_resources[0]

        with create_infra_event(self, resource_info['id']):
            super(self).on_release()
