import logging
import uuid

from sandbox import sdk2
from sandbox.projects.common.vcs import arc
from sandbox.sdk2.helpers import process
import sandbox.common.types.misc as ctm
from sandbox.common.types import resource as ctr

import sandbox.agentr.types as agentr_types

import os


class ArchivedObjectStore(sdk2.Resource):
    commit_id = sdk2.parameters.String
    warmed_paths = sdk2.parameters.String
    rocksdb_cache = sdk2.parameters.Bool

    any_arch = False
    auto_backup = True
    calc_md5 = True
    restart_policy = ctr.RestartPolicy.RESET
    release_subscribers = ['kikht', 'ilikepugs']
    share = True
    ttl = 7


class WarmingArcCache(sdk2.Task):
    class Requirements(sdk2.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 100 * 1024, None)

    class Parameters(sdk2.Task.Parameters):
        commit_id = sdk2.parameters.String('Commit id', default=None)
        paths_to_warm = sdk2.parameters.List('Warm paths', default=[])
        rocksdb_cache = sdk2.parameters.Bool('Rocksdb cache', default=False)
        rocksdb_path = sdk2.parameters.String('Rocksdb path', default='arcd/store/rocks_blob_cache')
        enable_ramdrive = sdk2.parameters.Bool('Enable ramdrive', default=True)

    def on_execute(self):
        self.Parameters.tags = ["ARC", "WARM_DISK_CACHE"]
        commit_id = self.Parameters.commit_id

        logging.info("Start mounting arc repository" + ("" if commit_id is None else " at {}".format(commit_id)))
        arc_repo = arc.Arc()
        workdir_path = str(sdk2.task.Task.current.path(agentr_types.FUSE_DIRNAME))
        if self.Parameters.enable_ramdrive and self.ramdrive:
            logging.info("Objects store will be placed on tmpfs")
            ramdrive_path = self.ramdrive.path
        else:
            logging.info("Tmpfs is disabled, so objects store will be placed on hard drive")
            ramdrive_path = workdir_path

        object_store_path = os.path.join(str(ramdrive_path), "object_store_" + str(uuid.uuid4()))
        with arc_repo.mount_path("", commit_id, object_store_path=object_store_path, fetch_all=False) as mount_path:
            try:
                logging.info("Trying to do rev-parse on " + commit_id)
                output = process.subprocess.check_output(
                    [arc_repo.binary_path, "rev-parse", (commit_id if commit_id else "HEAD")],
                    cwd=mount_path,
                    stderr=process.subprocess.STDOUT
                )
            except process.subprocess.CalledProcessError as err:
                logging.error("Failed to get commit hash: {}".format(err.output))
                raise err

            commit_id = output.strip()

            if self.Parameters.rocksdb_cache:
                archive_cwd = os.path.join(str(ramdrive_path), "rocks")
                result_dir = os.path.join(archive_cwd,  self.Parameters.rocksdb_path)
                os.makedirs(result_dir)
                prefetch_args = ['--rocks', result_dir]
                archive_path = os.path.join(workdir_path, "rocks_cache.tar")
                archive_command = "tar -cf {} ."
            else:
                archive_cwd = object_store_path
                prefetch_args = []
                archive_path = os.path.join(workdir_path, "object-store.tar.gz")
                archive_command = "GZIP=-2 tar -zcf {} ."

            try:
                logging.info("Start warming disk cache")
                output = process.subprocess.check_output(
                    [arc_repo.binary_path, "prefetch-files",  "--commit", commit_id] + prefetch_args + self.Parameters.paths_to_warm,
                    cwd=mount_path,
                    stderr=process.subprocess.STDOUT
                )
            except process.subprocess.CalledProcessError as err:
                logging.error("Failure while warming arc disk cache: {}".format(err.output))
                raise err

            try:
                logging.info("Archiving ready object-store {}".format(archive_cwd))
                output = process.subprocess.check_output(
                    archive_command.format(archive_path),
                    stderr=process.subprocess.STDOUT,
                    cwd=archive_cwd,
                    shell=True
                )
            except process.subprocess.CalledProcessError as err:
                logging.error("Error during archiving: {}".format(err.output))
                raise err
        out_cache = ArchivedObjectStore(
            self,
            description="Archived (using '{}') object-store for arc".format(archive_command),
            path=archive_path,
            commit_id=commit_id,
            warmed_paths=", ".join(self.Parameters.paths_to_warm),
            rocksdb_cache=self.Parameters.rocksdb_cache
        )
        out_resource = sdk2.ResourceData(out_cache)
        out_resource.ready()
        logging.info("Removing temporary objects store")
        if not self.ramdrive:
            import shutil
            shutil.rmtree(object_store_path, ignore_errors=True)
