import logging
import os

from sandbox import sdk2
from sandbox.projects.common.vcs import arc
from sandbox.sdk2.helpers import process
import sandbox.common.types.misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr
from sandbox.common.types import notification as ctn


def get_current_sb_url():
    return "https://sandbox.yandex-team.ru/task/{}".format(sdk2.Task.current.id)


class WarmedRevisionGraph(sdk2.Resource):
    hash_id = sdk2.parameters.String

    restart_policy = ctr.RestartPolicy.RESET
    release_subscribers = ['kikht', 'ilikepugs']
    share = True
    ttl = 7


class WarmingArcRevisionGraph(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 2048
        disk_space = 16 * 1024
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 14 * 1024, None)

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        load_from = sdk2.parameters.String('Previos revision graph state: blobref, commit or remote branch name', required=True)
        trunk_only = sdk2.parameters.Bool('Whether to warm only trunks commits', default=False)
        prestable = sdk2.parameters.Bool('Whether to warm graph for prestable', default=False)
        force_no_ramdrive = sdk2.parameters.Bool('Force do not use ramdrive', default=False)

    def on_save(self):
        super(WarmingArcRevisionGraph, self).on_save()
        server = "test" if self.Parameters.prestable else "prod"
        service = "trunk" if self.Parameters.trunk_only else "full"
        self.juggler_destination = "host=arc-{}&service=warm-graph-{}".format(server, service)
        self.Parameters.notifications = [
            sdk2.Notification(
                [ctt.Status.SUCCESS],
                [self.juggler_destination],
                ctn.Transport.JUGGLER,
                check_status=ctn.JugglerStatus.OK
            )
        ]

    def _juggler_crit_notification(self, text):
        self.server.notification(
            body=text,
            recipients=[self.juggler_destination],
            transport=ctn.Transport.JUGGLER,
            check_status=ctn.JugglerStatus.CRIT
        )

    def on_exception(self, exception):
        self._juggler_crit_notification(
            "Task {} failed with EXCEPTION:\n {}".format(get_current_sb_url(), exception)
        )

    def on_timeout(self):
        self._juggler_crit_notification(
            "Task {}: TIMEOUT exceeded".format(get_current_sb_url())
        )

    def on_execute(self):
        self.Parameters.tags = [
            "ARC",
            "WARM_REVISION_GRAPH",
            "TRUNK-ONLY" if self.Parameters.trunk_only else "FULL"
        ]
        if self.Parameters.prestable:
            self.Parameters.tags.append("PRESTABLE")

        load_from = str(self.Parameters.load_from)

        logging.info("Start mounting arc repository")
        arc_repo = arc.Arc()
        objects_path = None
        if not self.Parameters.force_no_ramdrive and self.ramdrive:
            logging.info("Objects store will be placed on tmpfs")
            objects_path = str(self.ramdrive.path / "objects")
        else:
            logging.info("Tmpfs is disabled, so objects store will be placed on hard drive")

        server = "arc-prestable.sas.yp-c.yandex.net" if self.Parameters.prestable else None
        extra_params = []
        if self.Parameters.prestable:
            extra_params = ["--server", server]

        with arc_repo.mount_path("", "trunk", object_store_path=objects_path, fetch_all=False, extra_params=extra_params) as mount_path:
            # TODO(ilikepugs): remove such ad-hoc and set up timeout by client
            config_lines = []
            config_path = os.path.join(mount_path, ".arc", "config")
            with open(config_path, "r") as config_file:
                config_lines = config_file.read().splitlines()
            for i, line in enumerate(config_lines):
                if "Server {" in line:
                    config_lines.insert(i + 1, "  RpcTimeout: 600000")
                    break
            with open(config_path, "w") as config_file:
                config_file.write("\n".join(config_lines))

            logging.info("Checking out to source ref")
            arc.Arc().checkout(mount_path, load_from, track=True)

            warm_args = ["--make-commit", "--load-from", load_from]
            if self.Parameters.trunk_only:
                warm_args += ["--trunk-only"]

            try:
                logging.info("Start warming revision graph")
                out = process.subprocess.check_output(
                    [arc_repo.binary_path, "dump", "rev-graph"] + warm_args,
                    cwd=mount_path,
                    stderr=process.subprocess.STDOUT
                )
                logging.debug(out)
                rev_graph_id = out.splitlines()[-1]
            except process.subprocess.CalledProcessError as err:
                logging.error(err.output)
                raise arc.ArcCommandFailed("Unable to warm arc revision graph")

            logging.info("Resetting source branch to warmed revision graph commit {}".format(rev_graph_id))
            arc.Arc().reset(mount_path, arc.ResetMode.HARD, rev_graph_id)
            logging.info("Updating source branch on the server")
            arc.Arc().push(mount_path, load_from, force=True)
