# -*- coding: utf-8 -*-

import os
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox import sdk2

from sandbox.sandboxsdk import svn

import sandbox.sandboxsdk.ssh as sdk_ssh
import sandbox.sandboxsdk.parameters as parameters
import sandbox.sandboxsdk.environments as environments
import sandbox.sandboxsdk.process as process

import sandbox.projects.common.build.ArcadiaTask as arcadia_task
import sandbox.projects.common.utils as utils


class SYNC_RESULT(sdk2.Resource):
    auto_backup = True


class GitUrlParam(parameters.SandboxStringParameter):
    name = 'git_url'
    description = 'Git url'
    default_value = ''


class ArcRelPathParam(parameters.SandboxStringParameter):
    name = 'arc_rel_path'
    description = 'Arcadia related project path to export'
    default_value = ''


class CiProjectIdParam(parameters.SandboxStringParameter):
    name = 'ci_project_id'
    description = 'Ci project id to get affected commits from'
    default_value = ''


class MappingResIdParam(parameters.SandboxStringParameter):
    name = 'mapping_res_id'
    description = 'Ya-bin small mapping resource id'
    default_value = ''


class OpensourceToolsIdParam(parameters.SandboxStringParameter):
    name = 'opensource_tools_id'
    description = 'Id of oss scripts resource'
    default_value = '500821960'


class StartSyncRevisionParam(parameters.SandboxStringParameter):
    name = 'sync_revision_from'
    description = 'Revision to start syncing from'
    default_value = 'HEAD'


class SyncScript(parameters.SandboxStringParameter):
    name = 'sync_script'
    description = 'Sync script'
    default_value = 'put_arcadia_to_git.sh'


class GitDir(parameters.SandboxStringParameter):
    name = 'git_dir'
    description = 'Git directory to sync'
    default_value = ''


class CheckResult(parameters.SandboxBoolParameter):
    name = 'check_resule'
    description = 'Check sync result'
    default_value = True


class DebugModeParam(parameters.SandboxBoolParameter):
    name = 'debug_mode'
    description = 'Debug mode'
    default_value = False


class DNS64Param(parameters.SandboxBoolParameter):
    name = 'dns64'
    description = 'Use DNS64'


class SaveResult(parameters.SandboxBoolParameter):
    name = 'save_result'
    description = 'Save sync result'
    default_value = True


class TraceOutput(parameters.SandboxBoolParameter):
    name = 'trace_output'
    description = 'Trace sync output'
    default_value = False


class RequireBitbucketToken(parameters.SandboxBoolParameter):
    name = 'bb_token_required'
    description = 'Pass bb token'
    default_value = False


class RequireS3Tokens(parameters.SandboxBoolParameter):
    name = 's3_tokens_required'
    description = 'Pass s3 token'
    default_value = False


class ArcadiaGitSync(arcadia_task.ArcadiaTask):
    """
        Arc->Git sync
    """

    type = "ARCADIA_GIT_SYNC"

    environment = [environments.SvnEnvironment()]
    client_tags = ctc.Tag.GENERIC | ctc.Tag.Group.OSX
    cores = 1
    required_ram = 2048
    execution_space = 2048

    input_parameters = [
        ArcRelPathParam,
        CiProjectIdParam,
        StartSyncRevisionParam,
        GitUrlParam,
        MappingResIdParam,
        OpensourceToolsIdParam,
        DebugModeParam,
        SyncScript,
        GitDir,
        CheckResult,
        SaveResult,
        TraceOutput,
        RequireBitbucketToken,
        RequireS3Tokens,
    ]

    def __init__(self, *args, **kwargs):
        arcadia_task.ArcadiaTask.__init__(self, *args, **kwargs)

    @property
    def dns(self):
        return ctm.DnsType.DNS64 if self.ctx.get(DNS64Param.name) else ctm.DnsType.DEFAULT

    def _get_opensource_dir_path(self):
        dest = self.path('opensource')
        tools = self.sync_resource(utils.get_or_default(self.ctx, OpensourceToolsIdParam))
        with tarfile.open(tools) as tf:
            tf.extractall(dest)
        svn.Arcadia.export(
            'arcadia:/arc/trunk/arcadia/opensource/github.login',
            dest,
        )
        return dest

    def _get_start_sync_revision(self):
        return utils.get_or_default(self.ctx, StartSyncRevisionParam)

    def _get_mapping_res_id(self):
        return utils.get_or_default(self.ctx, MappingResIdParam)

    def on_execute(self):
        work_dir = self.path("sync")
        os.makedirs(work_dir)

        if utils.get_or_default(self.ctx, SaveResult):
            sync_output_res = self.create_resource(
                description='Sync output',
                resource_path=work_dir,
                resource_type=SYNC_RESULT,
                attributes={
                    'ttl': 3
                }
            )
        else:
            sync_output_res = None

        #  XXX - move to input parameters
        with sdk_ssh.Key(self, "akastornov", "arcadia-devtools-ssh-key"):
            mds_token = self.get_vault_data("YATOOL", "MDS_DEVTOOLS_OPENSOURCE_TOKEN_PROD")
            staff_token = self.get_vault_data("YATOOL", "STAFF_DEVTOOLS_OAUTH_TOKEN")
            arc_token = self.get_vault_data("YATOOL", "ARC_TOKEN")
            env = os.environ.copy()
            env["MDS_DEVTOOLS_OPENSOURCE_TOKEN"] = mds_token
            env["STAFF_DEVTOOLS_OAUTH_TOKEN"] = staff_token
            env["ARC_TOKEN"] = arc_token

            if utils.get_or_default(self.ctx, RequireBitbucketToken):
                env["BB_TOKEN"] = self.get_vault_data("YATOOL", "BB_TOKEN")

            if utils.get_or_default(self.ctx, RequireS3Tokens):
                env["AWS_ACCESS_KEY_ID"] = self.get_vault_data("YATOOL", "NZ_SYNC_KEY_ID")
                env["AWS_SECRET_ACCESS_KEY"] = self.get_vault_data("YATOOL", "NZ_SYNC_ACCESS_KEY")

            os_dir_path = self._get_opensource_dir_path()

            cmd = [
                os.path.join(os_dir_path, "bin", "arc2git"),
                "--scripts-dir", os.path.join(os_dir_path, "scripts"),
                "--arc-project-path", utils.get_or_default(self.ctx, ArcRelPathParam),
                "--login-mapping", os.path.join(os_dir_path, "github.login"),
                "--revision-from", self._get_start_sync_revision(),
                "--git", utils.get_or_default(self.ctx, GitUrlParam),
                "--mapping-res-id", self._get_mapping_res_id(),
                "--work-dir", work_dir,
                "--logs-dir", work_dir,
                "--sync-script", utils.get_or_default(self.ctx, SyncScript),
            ]

            project_id = utils.get_or_default(self.ctx, CiProjectIdParam)
            if project_id:
                cmd += ["--project-id", project_id]

            if sync_output_res:
                cmd += ["--logs-url", "https://proxy.sandbox.yandex-team.ru/{}".format(sync_output_res.id)]

            if utils.get_or_default(self.ctx, CheckResult):
                cmd += ["--check"]

            _git_dir = utils.get_or_default(self.ctx, GitDir)
            if _git_dir:
                cmd += ["--git-dir", _git_dir]

            if utils.get_or_default(self.ctx, TraceOutput):
                cmd += ["--trace-output"]

            process.run_process(
                cmd,
                log_prefix='sync', outputs_to_one_file=True, work_dir=work_dir, environment=env,
            )


__Task__ = ArcadiaGitSync
