# -*- coding: utf-8 -*-
import logging
import os
import tarfile
import tempfile

import sandbox.sdk2 as sdk2
from sandbox.projects import resource_types
from sandbox.projects.common.arcadia import sdk
from sandbox.sdk2.helpers import subprocess as sp

import test_stat
import json

logger = logging.getLogger(__name__)


class CoverageDiffReportResource(sdk2.Resource):
    """
    Report for coverage diff
    """


class CoverageDiffReport(sdk2.Task):
    """
    Make Report on Coverage Diff
    """

    class Parameters(sdk2.Parameters):
        diff_res_id = sdk2.parameters.Resource(
            "Coverage diff binary archive",
            resource_type=resource_types.ARCADIA_PROJECT_TGZ,
            required=True,
        )
        build_output_before = sdk2.parameters.Resource(
            "COVERAGE_YA_MAKE_TASK output",
            resource_type=resource_types.BUILD_OUTPUT,
            required=True,
        )
        build_output_after = sdk2.parameters.Resource(
            "COVERAGE_YA_MAKE_TASK output with patch",
            resource_type=resource_types.BUILD_OUTPUT,
            required=True,
        )
        filter = sdk2.parameters.String("Coverage source path", required=True)
        test_paths = sdk2.parameters.String("Target test path (semicolon separated)", required=True)
        review_id = sdk2.parameters.Integer("Review ID")
        revision = sdk2.parameters.String("Revision")
        zipatch_link = sdk2.parameters.String("Link to zipatch")
        arcadia = sdk2.parameters.ArcadiaUrl('Arcadia url')

    class Context(sdk2.Task.Context):
        arcanum_comment = ""
        has_diff = False

    def make_path_to_coverage(self):
        return self.Parameters.test_paths

    def unzip_tgz(self, file, to_dir):
        os.makedirs(to_dir)

        binary_path = None
        with tarfile.open(file, 'r:gz') as tar:
            tar.extractall(path=to_dir)
            binary_path = os.path.join(to_dir, "coverage_diff")
            logger.info("Binary extracted: %s", str(os.path.exists(binary_path)))

        return binary_path

    def on_execute(self):
        tmp_dir = tempfile.mkdtemp(prefix="cov_diff")
        logger.info("Make tmp directory %s", tmp_dir)
        binary_dir = os.path.join(tmp_dir, "bin")

        diff_tgz_path = str(sdk2.ResourceData(self.Parameters.diff_res_id).path)
        logger.info("Path to diff binary: %s", diff_tgz_path)
        binary_path = self.unzip_tgz(diff_tgz_path, binary_dir)

        path_to_coverage = self.make_path_to_coverage()
        output_before_path = str(sdk2.ResourceData(self.Parameters.build_output_before).path)
        logger.info("Path to before output: %s", os.path.join(output_before_path, path_to_coverage))
        logger.info("Coverage before found: %s", str(os.path.exists(os.path.join(output_before_path, path_to_coverage))))
        output_after_path = str(sdk2.ResourceData(self.Parameters.build_output_after).path)
        logger.info("Path to after output: %s", os.path.join(output_after_path, path_to_coverage))
        logger.info("Coverage after found: %s", str(os.path.exists(os.path.join(output_after_path, path_to_coverage))))

        out_resource = CoverageDiffReportResource(self, "Coverage diff", "resource_directory", ttl=15)

        resource_data = sdk2.ResourceData(out_resource)
        resource_data.path.mkdir(0o755, parents=True, exist_ok=True)

        out_path = str(resource_data.path)
        logger.info("Output path: %s", out_path)
        logger.info("Output URL: %s", out_resource.http_proxy)

        zipatch = self.Parameters.zipatch_link or self.Context.arcadia_patch
        review_id = self.Parameters.review_id or self.Context.arcanum_review_id

        with sdk.mount_arc_path(self.Parameters.arcadia, fallback=True) as src_dir:
            logger.info("Arcadia source directory: {}".format(src_dir))
            source_root = str(src_dir)
            logger.info("Arcadia root: %s", source_root)

            with sdk2.helpers.ProcessLog(self, logger="coverage_diff") as pl:
                sp.Popen([binary_path,
                          "--filter", str(self.Parameters.filter),
                          "--source-dir",  source_root,
                          "--zipatch", str(zipatch),
                          "--old-coverage", os.path.join(output_before_path, path_to_coverage),
                          "--new-coverage", os.path.join(output_after_path, path_to_coverage),
                          "--review", str(review_id),
                          "--out-dir", out_path,
                          "--revision", str(self.Parameters.revision)],
                         stdout=pl.stdout, stderr=pl.stderr).wait()

        resource_data.ready()

        index_url = os.path.join(out_resource.http_proxy, "index.html")
        logger.info("URL to report index page: %s", index_url)

        summary_file_name = os.path.join(out_path, "summary.txt")
        if os.path.exists(summary_file_name):
            with open(summary_file_name) as summary:
                status = summary.read()
        else:
            logger.warning("Failed to get summary")
            status = ""

        self.Context.arcanum_comment = ""
        try:
            if self.Context.__te_apiargs:
                j = json.loads(self.Context.__te_apiargs)
                if "tags" in j:
                    for tag in j["tags"]:
                        if "COVERAGE" in tag:
                            self.Context.arcanum_comment = "`{}`\n\n".format(tag)
                            break
        except Exception:
            logger.error("Failed to parse TE args")

        self.Context.arcanum_comment += "{status}*Coverage diff report [here]({url} )*\n".format(status=status, url=index_url)
        self.Context.arcanum_comment += "\n---\nTest results:\n"
        self.Context.arcanum_comment += test_stat.change_stat_msg(test_stat.get_test_stat(output_before_path),
                                                                  test_stat.get_test_stat(output_after_path))
