import dateutil.parser
import json
import logging
import os
import pytz

from sandbox import sdk2
from sandbox.common import patterns as common_patterns
from sandbox.common.types import misc as ctm
from sandbox.common import errors as ce
from sandbox.common import urls as common_urls
from sandbox.sdk2 import parameters as sb_parameters
from sandbox.sdk2 import paths
from sandbox.sdk2.vcs import svn

from sandbox.projects.common import binary_task
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.build import YaMake2
from sandbox.projects.common.build.sdk import sdk_compat as ya_sdk_compat
from sandbox.projects.common.vcs import arc


logger = logging.getLogger(__name__)


class ArcadiaDocsS3(sdk2.Resource):
    executable = False
    auto_backup = True

    target = sdk2.parameters.String


class ArcadiaDocsDaas(sdk2.Resource):
    executable = False
    auto_backup = True

    target = sdk2.parameters.String


class DeployArcadiaDocs(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Deploy Arcadia DOCS artifact
    """

    ARCANUM_TOKEN = 'robot-cozmo:robot-cozmo-arcanum-token'

    S3_ARTIFACT_MD = 'preprocessed.tar.gz'
    S3_ARTIFACT_HTML = 'docs.tar.gz'
    S3_BUCKET_NAME = 'docs-test'.lower()
    S3_ENDPOINT_URL = 'https://s3.mds.yandex.net'
    YAV_S3_KEY = 'Access-key-prod'
    YAV_TOKEN = 'robot-cozmo:robot-cozmo-yav-token'
    YAV_UUID = 'sec-01drh3z694nqvd4wa9qwc0y39z'

    YFM_URL_PATTERN = 'https://{version}.testing.docs.yandex-team.ru/{project}/'
    YFM_EXTERNAL_URL_PATTERN = 'https://{project}.docs-ext.yandex-team.ru/?revision={version}'

    class Requirements(sdk2.Requirements):
        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        targets = sdk2.parameters.String('Targets (semicolon separated)', required=True)
        checkout_arcadia_from_url = build_parameters.ArcadiaUrl()
        arcadia_patch = build_parameters.ArcadiaPatch()

        deploy_to = sb_parameters.String('Deploy to', choices=(('cloud', 'cloud'), ('daas', 'daas')), default='cloud')

        arcanum_review_id = sb_parameters.Integer('Arcanum PR id', default=None)
        notify = sb_parameters.Bool('Post deployed url to Arcanum comments (either for commit or pull request)', default=True)
        is_precommit = sb_parameters.Bool('Task run for precommit check', default=False)
        ignore_auto_release_settings = sb_parameters.Bool('Ignore deploy.auto-release setting in .yfm config', default=False)
        direct_api_upload = sb_parameters.Bool('Upload using docs API', default=False)
        docs_token = sb_parameters.YavSecretWithKey('OAuth token to authenticate at docs.y-t API')

        with sdk2.parameters.Group('Runtime') as runtime:
            use_aapi_fuse = build_parameters.UseArcadiaApiFuse(default=True)
            use_arc_instead_of_aapi = build_parameters.UseArcInsteadOfArcadiaApi(default=True)
            with use_arc_instead_of_aapi.value[True]:
                arc_secret = build_parameters.ArcSecret(default='sec-01eyxes68z3wmr6k1kkjddv27g#arc-token')
            aapi_fallback = build_parameters.AllowArcadiaApiFallback(default=True)
            checkout = build_parameters.CheckoutParameter(default=True)
            checkout_mode = build_parameters.CheckoutModeParameter()

            ram_drive_size = build_parameters.RamDriveSize()

        with sdk2.parameters.Group('Task executor') as task_executor:
            ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output:
            projects = sb_parameters.JSON('Deployed projects')


    class AutoLinks(sdk.AutoLinks):
        def __init__(self, base_dir):
            self._files = []
            self._base_dir = base_dir

        def add(self, name, resource_type, description=None, ttl=4):
            fname = paths.get_unique_file_name(self._base_dir, name)
            logger.debug('Add auto link %s', (fname, resource_type, description))
            self._files.append((fname, resource_type, description, ttl))

            return fname

        def finalize(self):
            pass

    def on_save(self):
        super(DeployArcadiaDocs, self).on_save()

        if self.Parameters.deploy_to == 'cloud':
            self.Requirements.cores = 1
            self.Requirements.ram = 8 * 1024

        if self.Parameters.ram_drive_size:
            self.ramdrive = self.RamDrive(
                ctm.RamDriveType.TMPFS,
                self.Parameters.ram_drive_size,
                None
            )
        else:
            self.ramdrive = None

    def on_enqueue(self):
        self.Context.ap_packs = {}

    @property
    def from_arc_path(self):
        return YaMake2.ya_sdk_compat.is_arc_path(self)

    def pre_execute(self):
        checkout_arcadia_from_url = self.Parameters.checkout_arcadia_from_url
        try:
            checkout_arcadia_from_url = sdk2.svn.Arcadia.freeze_url_revision(checkout_arcadia_from_url)
        except sdk2.svn.SvnError as e:
            raise ce.TaskFailure('Arcadia URL {0} does not exist. Error: {1}'.format(checkout_arcadia_from_url, e))

        parsed_url = sdk2.svn.Arcadia.parse_url(checkout_arcadia_from_url)
        self.Context.ap_arcadia_revision = parsed_url.revision
        self.Context.ap_arcadia_trunk = parsed_url.trunk and not self.Parameters.arcadia_patch
        self.Context.ap_arcadia_branch = parsed_url.branch
        self.Context.ap_arcadia_tag = parsed_url.tag
        self.Context.checkout_arcadia_from_url = checkout_arcadia_from_url

        if not self.from_arc_path:
            # these properties will be filled then repo is mounted
            self.Context.ap_arcadia_arc_svn_revision = None
            self.Context.merge_base = None
            self.Context.merge_base_svn_revision = None

            info_from_svn = svn.Arcadia.info(self.Context.checkout_arcadia_from_url)
            self.Context.ap_author = info_from_svn['author']
            self.Context.ap_timestamp = info_from_svn['date']

    def on_execute(self):
        if self.Parameters.is_precommit and not self.Parameters.arcadia_patch and not self.from_arc_path:
            self.set_info('Skip run for base revision in precommit check')
            return

        if self.from_arc_path and self.Parameters.arcadia_patch:
            ce.TaskFailure('Arc based runs with patch not supported')

        if self.Parameters.is_precommit and self.Parameters.arcadia_patch and not self.Parameters.arcanum_review_id:
            ce.TaskFailure('Arcanum PR id must be set for svn based precommit runs with patch')

        super(DeployArcadiaDocs, self).on_execute()

        self.pre_execute()

        build_path, output_dir = YaMake2.ya_sdk_compat.get_build_path(self)
        self.Context.output_dir_path = output_dir

        local_output_dir = YaMake2.ya_sdk_compat.get_local_output_dir(self)
        logger.debug('local_output_dir: %s', local_output_dir)
        results_dir = output_dir if local_output_dir else None

        targets = YaMake2.ya_sdk_compat.get_targets(self)

        patch = self.Parameters.arcadia_patch

        if self.Parameters.deploy_to == 'cloud':
            build_def_flags = '-DDOCSBUILDER=yfm'
            build_system = 'ya_force'
        else:
            build_def_flags = '-DDOCSBUILDER=mkdocs'
            build_system = 'semi_distbuild'
        build_def_flags += ' -DNO_SRCLINK=yes'

        make_context_on_distbuild = False

        YaMake2.ya_sdk_compat.check_dist_build_parameters(self, build_system, make_context_on_distbuild)
        checkout, mode = YaMake2.ya_sdk_compat.check_parameters_compatibility(self, make_context_on_distbuild)
        self.Context.checkout = checkout
        self.Context.mode = mode

        arcadia_ctx, arcadia_tests_data_ctx = YaMake2.ya_sdk_compat.get_source_dirs(self, mode)

        YaMake2.ya_sdk_compat.process_resources(self, self.get_resources_params())

        with arcadia_ctx as arcadia_path:
            if self.from_arc_path:
                arc_info = arc.Arc().log(arcadia_path, max_count=1, as_dict=True)[0]
                self.Context.ap_arcadia_revision = arc_info['commit']
                self.Context.ap_arcadia_arc_svn_revision = arc_info.get('revision')

                local_branches = arc_info.get('branches', {}).get('local', [])
                if local_branches:
                    self.Context.ap_arcadia_trunk = 'trunk' in local_branches
                    self.Context.ap_arcadia_branch = local_branches[-1]
                else:
                    self.Context.ap_arcadia_trunk = True
                    self.Context.ap_arcadia_branch = None

                self.Context.ap_arcadia_tag = None
                self.Context.ap_author = arc_info['author']
                self.Context.ap_timestamp = arc_info['date']

                self.Context.merge_base = arc.Arc().get_merge_base(arcadia_path)
                merge_base_details = arc.Arc().show(arcadia_path, commit=self.Context.merge_base, as_dict=True, name_status=True)
                logger.debug('merge_base_details:\n%s', merge_base_details)
                self.Context.merge_base_svn_revision = merge_base_details[0].get('commits', [])[0].get('revision')

            logger.info('Arcadia source dir: %s', arcadia_path)

            if patch:
                sdk.apply_patch(self, arcadia_path, patch, self.abs_path(), checkout)

            import traceback
            try:
                env = {'YA_DUMP_RAW_RESULTS': '1'}
                logger.debug('Start do_build')

                sdk.do_build(
                    build_system,
                    arcadia_path,
                    targets,
                    build_execution_time=600,
                    build_type='release',
                    check_rc=True,
                    checkout=checkout,
                    clear_build=False,
                    def_flags=build_def_flags,
                    download_artifacts=True,
                    env=env,
                    patch=patch,
                    ram_drive_path=ya_sdk_compat.get_ram_drive_path(self),
                    results_dir=results_dir,
                    auto_links=self.AutoLinks(str(self.log_path().absolute())),
                    dump_graph_for_debug=False,
                )
            except Exception:
                logger.debug(traceback.format_exc())
                raise
            finally:
                try:
                    from sandbox.sandboxsdk import process
                    p = process.run_process(['chmod', '--recursive', '755', output_dir], log_prefix='chmod_output_dir')
                    p.communicate()
                except Exception as e:
                    logger.error('Error while chmod: %s', e)

            project_docs_properties = self.mine_docs_properties_from_ya_make_files(arcadia_path)

            if self.Parameters.direct_api_upload:
                deployed_projects = self.send_results_to_storage_via_api(arcadia_path, project_docs_properties)
            else:
                deployed_projects = self.send_results_to_storage_via_s3(arcadia_path, project_docs_properties)
            logger.debug('Results uploaded: %s', deployed_projects)
            for project_name, project in deployed_projects.iteritems():
                self.set_info('Preview: <a href="{}">{}</a>'.format(project['url'], project_name), do_escape=False)

            self.auto_release(deployed_projects)
            self.push_comments(deployed_projects)

            self.Parameters.projects = dict((
                (project_name, {
                    'revision': d['revision'],
                    'is_trunk': self.Context.ap_arcadia_trunk,
                    'is_external': d.get('is_external', False)
                }) \
                for project_name, d in deployed_projects.iteritems()
            ))

        YaMake2.ya_sdk_compat.pack_arts([{'path': t, 'dest': t} for t in YaMake2.ya_sdk_compat.get_targets(self)], YaMake2.PACK_DIR, self.Context.output_dir_path)
        YaMake2.ya_sdk_compat.finalize_resources(self)

    def abs_path(self, *args):
        return str(self.log_path(*args).absolute())

    def get_resources_params(self):
        resources = {}
        for art in YaMake2.ya_sdk_compat.get_targets(self):
            project = {
                'resource_type': ArcadiaDocsS3.name,
                'description': 'DOCS S3 for {} r{}'.format(art, self.Context.ap_arcadia_revision),
                'resource_path': art,
            }
            resources[art] = project
        return resources

    def get_base_resource_attrs(self):
        return {
            'arcadia_revision': self.Context.ap_arcadia_revision,
            'arcadia_trunk': self.Context.ap_arcadia_trunk,
            'arcadia_branch': self.Context.ap_arcadia_branch,
            'arcadia_tag': self.Context.ap_arcadia_tag,
        }

    def get_resources_attrs(self):
        return dict([
            (art, {'target': art})
            for art in YaMake2.ya_sdk_compat.get_targets(self)
        ])

    def create_resource(self, description, resource_path, resource_type, arch=None, **kwargs):
        resource = sdk2.Resource[resource_type](
            task=self,
            description=description,
            path=resource_path,
            arch=arch,
        )
        logger.debug('create_resource: unused parameters:', kwargs)

        return resource

    def _arcanum_pr_meta(self, version, docs_dir):
        """
        Collect data from VCS to JSON (author, date, etc.)

        :param version:     documentation bundle version
        :param docs_dir:    bundle's source path

        :return: JSON with PR meta
        """

        if self.Parameters.arcanum_review_id:
            arcanum_pr = self.arcanum_client.pullrequest(self.Parameters.arcanum_review_id).get()
            iteration_info = filter(
                lambda x: x['zipatch'] == self.Parameters.arcadia_patch,
                arcanum_pr['patches'])
            if not iteration_info:
                # arcanum stores the only one patch for unpublished iteration.
                # if the task was started for unpublished iteration, the user may have uploaded a new version of draft
                # and in this case arcanum api won't say a word about the initial draft version the task was created for.
                # so let's just take the closest iteration.
                svn_base_rev = int(
                    self.Context.merge_base_svn_revision if self.from_arc_path else self.Context.ap_arcadia_revision)
                iteration_info = sorted(
                    filter(
                        lambda x: int(x['svn_base_rev']) >= svn_base_rev,
                        arcanum_pr['patches']),
                    key=lambda x: x['svn_base_rev'],
                )
            date = iteration_info[0]['created_time']
            pr_iteration = iteration_info[0]['revision']

            revision_info = {
                'author': arcanum_pr['author'],
                'timestamp': date,
                'revision': version,
                'pr_id': self.Parameters.arcanum_review_id,
                'pr_iteration': pr_iteration,
            }
        else:
            revision_info = {
                'author': self.Context.ap_author,
                'timestamp': self.Context.ap_timestamp,
                'revision': version,
            }

        ts = dateutil.parser.parse(revision_info['timestamp'])
        revision_info['timestamp'] = ts.astimezone(tz=pytz.utc).isoformat()
        revision_info['date'] = ts.strftime('%d.%m.%Y')
        revision_info['arcadia-path'] = docs_dir
        logger.debug('revisions_info %s', revision_info)

        return revision_info

    def send_results_to_storage_via_s3(self, source_root, project_docs_properties):
        """
        :param source_root:             (string) path to arcadia dir
        :param project_docs_properties: (dict)   project name -> project name -> (dict):
            - docs_dir          relative path to source root
            - docs_config_path  relative path to `.yfm` file
        :return: deployed_projects      (dict)   project name -> (dict):
            - revision:     (string) identifier of project version on docs-viewer hosting
            - settings:     (dict)   content of project's .yfm
            - url:          (string) deployed url
        """

        import boto3
        import shutil
        import tarfile
        import yaml
        from library.python.vault_client import instances as yav_instances
        from . import lib as release_lib

        logger.debug('Trying to get S3 secret from yav')
        yav_token = sdk2.Vault.data(*self.YAV_TOKEN.split(':'))

        yav_client = yav_instances.Production(
            rsa_auth=False,
            authorization='OAuth {}'.format(yav_token),
            decode_files=True,
        )

        s3_secret = yav_client.get_version(self.YAV_UUID)
        s3_access_key = json.loads(s3_secret['value'][self.YAV_S3_KEY])

        logger.debug('Creating S3 client')
        session = boto3.session.Session(
            aws_access_key_id=s3_access_key.get('AccessKeyId'),
            aws_secret_access_key=s3_access_key.get('AccessSecretKey'),
        )
        s3 = session.client(
            service_name='s3',
            endpoint_url=self.S3_ENDPOINT_URL,
        )

        results = {}
        projects_to_register = {}
        for target_dir in YaMake2.ya_sdk_compat.get_targets(self):

            self.path('extracted').mkdir(exist_ok=True)
            result_dir = os.path.join('extracted', target_dir)
            with tarfile.open(os.path.join(self.Context.output_dir_path, target_dir, self.S3_ARTIFACT_MD)) as tared_dir:
                tared_dir.extractall(result_dir)

            config_path = os.path.join(source_root, project_docs_properties[target_dir]['docs_config_path'])
            shutil.copy(config_path, os.path.join(result_dir, '.yfm'))  # why is this?

            logger.debug('Loading yfm config from %s', config_path)
            with open(config_path, 'r') as f:
                yfm_settings = yaml.load(f)

            try:
                project_name = yfm_settings['docs-viewer']['project-name'].lower()
            except KeyError:
                project_name = yfm_settings['project_name'].lower()

            langs = yfm_settings['docs-viewer'].get('langs', [])

            version = '{}{}'.format(
                str(self.Parameters.arcanum_review_id or 'custom-patch') + '-' if self.Parameters.arcadia_patch else '',
                self.Context.ap_arcadia_revision,
            )

            for lang in langs or ['ru']:
                s3_object_key = '{project}/{version}/{lang}/'.format(project=project_name, version=version, lang=lang)
                logger.debug('Uploading %s/%s to key %s', target_dir, lang, s3_object_key)

                # 'ru' is assumed to be default and not required to present in result_dir
                result_dir_lang = os.path.join(result_dir, lang) if langs else result_dir
                if not os.path.exists(result_dir_lang):
                    raise ce.TaskFailure('Language directory for "{}" not found in results'.format(lang))
                for root, _, files in os.walk(result_dir_lang):
                    for f in files:
                        filename = os.path.join(root, f)
                        s3.upload_file(Filename=filename, Bucket=self.S3_BUCKET_NAME, Key=s3_object_key + os.path.relpath(filename, result_dir_lang))
            if langs:
                for filename in ('.yfm', 'redirects.yaml'):
                    if not os.path.exists(os.path.join(result_dir, filename)):
                        logger.debug('File %s not found in results dir - skip uploading to S3', filename)
                        continue
                    s3_object_key = '{project}/{version}/'.format(project=project_name, version=version)
                    logger.debug('Uploading %s/%s to key %s', target_dir, filename, s3_object_key)
                    s3.upload_file(Filename=os.path.join(result_dir, filename), Bucket=self.S3_BUCKET_NAME, Key=s3_object_key + filename)

            projects_to_register[project_name] = self._arcanum_pr_meta(version, project_docs_properties[target_dir]['docs_dir'])

            results[project_name] = {
                'revision': version,
                'settings': yfm_settings,
                'url': self.YFM_URL_PATTERN.format(project=project_name, version=version),
            }

        release_lib.deploy(
            projects_to_register,
            user_agent='{}:{}'.format(self.type.name, self.id),
            token=sdk2.Vault.data(*self.ARCANUM_TOKEN.split(':')),
        )
        return results

    def send_results_to_storage_via_api(self, source_root, project_docs_properties):
        '''
        :param source_root:             (string) path to arcadia dir
        :param project_docs_properties: (dict)   project name -> project name -> (dict):
            - docs_dir          relative path to source root
            - docs_config_path  relative path to `.yfm` file
            - docs_archive_name name of the output archive
        :return: deployed_projects      (dict)   project name -> (dict):
            - revision:     (string) identifier of project version on docs-viewer hosting
            - settings:     (dict)   content of project's .yfm
            - url:          (string) deployed url
        '''
        import yaml
        from . import lib as release_lib

        yfm_api_token = (
            self.Parameters.docs_token.value()
            if self.Parameters.docs_token else
            sdk2.Vault.data(*self.ARCANUM_TOKEN.split(':'))
        )

        results = {}
        projects_to_register = {}
        for target_dir in YaMake2.ya_sdk_compat.get_targets(self):
            artifact_md = sdk2.path.Path(self.Context.output_dir_path) / target_dir /  self.S3_ARTIFACT_MD
            artifact_html = sdk2.path.Path(self.Context.output_dir_path) / target_dir /  project_docs_properties[target_dir]['docs_archive_name']
            config_path = sdk2.path.Path(source_root) / project_docs_properties[target_dir]['docs_config_path']
            logger.debug('Loading yfm config from %s', config_path)
            with config_path.open() as f:
                yfm_settings = yaml.load(f)

            try:
                project_name = yfm_settings['docs-viewer']['project-name'].lower()
            except KeyError:
                project_name = yfm_settings['project_name'].lower()

            is_external = yfm_settings['docs-viewer'].get('is-external', False)
            langs = yfm_settings['docs-viewer'].get('langs', [])
            def_lang_prefix = not bool(langs)

            version = '{}{}'.format(
                str(self.Parameters.arcanum_review_id or 'custom-patch') + '-' if self.Parameters.arcadia_patch else '',
                self.Context.ap_arcadia_revision,
            )

            logger.debug("Uploading documentation tarball for project %r revision %r", project_name, version)
            for st in release_lib.upload(yfm_api_token, project_name, version, artifact_md, artifact_html, def_lang_prefix, is_external):
                logger.debug("Current uploading state is %r", st)

            projects_to_register[project_name] = self._arcanum_pr_meta(version, project_docs_properties[target_dir]['docs_dir'])
            projects_to_register[project_name]['is_external'] = is_external

            url = self.YFM_EXTERNAL_URL_PATTERN if is_external else self.YFM_URL_PATTERN

            results[project_name] = {
                'revision': version,
                'settings': yfm_settings,
                'url': url.format(project=project_name, version=version),
                'is_external': is_external
            }

        release_lib.deploy(
            projects_to_register,
            user_agent='{}:{}'.format(self.type.name, self.id),
            token=yfm_api_token,
        )
        return results

    def push_comments(self, projects):
        import vcs.arcanum.comments

        task_link = '_by [Sandbox task]({server_url}/task/{task_id})_'.format(task_id=self.id, server_url=common_urls.server_url())
        if len(projects) == 1:
            project_name, project = list(projects.items())[0]
            message = 'Docs deployed: [open docs {project_name}]({project_url})\n{task_link}'.format(
                project_name=project_name, project_url=project['url'], task_link=task_link
            )
        else:
            message = 'Docs deployed: \n{projects}\n\n{task_link}'.format(
                projects='\n'.join(['* [Open docs {project_name}]({project_url})'.format(project_name=project_name, project_url=d['url']) for project_name, d in projects.iteritems()]),
                task_link=task_link
            )

        # notify if self.Parameters.notify is set and
        # (when deploying from trunk) there is at least one project with `deploy.comment-on-trunk-commits: true` in .yfm
        need_notify_trunk = False
        for project in projects.values():
            need_notify_trunk = need_notify_trunk or project['settings'].get('deploy', {}).get('comment-on-trunk-commits', False)
        need_notify = need_notify_trunk and self.Context.ap_arcadia_trunk or self.Parameters.arcanum_review_id
        need_notify = need_notify and self.Parameters.notify

        if not need_notify:
            self.set_info('Skip posting comment\n{}'.format(message))
            return

        comments_api = vcs.arcanum.comments.CommentsApi(self.arcanum_client)
        if self.Parameters.arcanum_review_id:
            comments_api.post_review_requests_comments(
                review_request_id=self.Parameters.arcanum_review_id,
                content=message,
                draft=False,
            )
        elif not self.Parameters.arcadia_patch:
            comments_api.post_commits_comments(
                repo_name='arc_vcs' if self.from_arc_path else 'arc',
                commit_id=self.Context.ap_arcadia_revision,
                content=message)

    def auto_release(self, projects):
        if self.Parameters.ignore_auto_release_settings:
            self.set_info('Skip auto release:\r\n{}'.format(projects))
            return

        from . import lib as release_lib

        # check if auto release demanded
        release_projects = {}
        for project_name, deploy_data in projects.iteritems():
            auto_release_setting = deploy_data['settings']['docs-viewer'].get('auto-release-to')
            envs = []
            if isinstance(auto_release_setting, dict):
                if auto_release_setting.get('testing', False):
                    envs.append(release_lib.VIEWER_TESTING_ENV)
                if auto_release_setting.get('prod', False):
                    envs.append(release_lib.VIEWER_PROD_ENV)
            release_projects[project_name] = {'environments': envs, 'revision': deploy_data['revision'], 'is_external': deploy_data.get('is_external', False)}

        release_lib.release(
            release_projects,
            self.Context.ap_arcadia_trunk,
            user_agent='{}:{}'.format(self.type.name, self.id),
            token=(
                self.Parameters.docs_token.value()
                if self.Parameters.docs_token else
                sdk2.Vault.data(*self.ARCANUM_TOKEN.split(':'))
            ),
        )

    @common_patterns.singleton_property
    def arcanum_client(self):
        import vcs.arcanum

        # In case of `docs_token` was provided it means the task is created via new version of CI process, so
        # we will use arc token to communicate with arcanum too - it should be user's robot instead of default
        arcanum_token = (
            self.Parameters.arc_secret.value()
            if self.Parameters.docs_token else
            sdk2.Vault.data(*self.ARCANUM_TOKEN.split(':'))
        )
        return vcs.arcanum.Arcanum(auth=arcanum_token, user_agent='Sandbox task {}'.format(self.type.name))

    def mine_docs_properties_from_ya_make_files(self, source_root):
        '''
        :param source_root:        (string) path to arcadia dir
        :return: docs_properties   (dict) project name -> (dict):
            - docs_dir          relative path to source root
            - docs_config_path  relative path to `.yfm` file
            - docs_archive_name name of the output archive
        '''
        docs_properties = {}

        for target_dir in YaMake2.ya_sdk_compat.get_targets(self):
            docs_dir, docs_config, docs_archive_name = _parse_ya_make(os.path.join(source_root, target_dir, 'ya.make'))

            docs_config = os.path.normpath(docs_config or os.path.join(target_dir, '.yfm'))
            if os.path.sep not in docs_config:
                docs_config = os.path.join(target_dir, docs_config)
            
            docs_archive_name = '${0}.tar.gz'.format(docs_archive_name) if docs_archive_name else self.S3_ARTIFACT_HTML

            docs_properties[target_dir] = {
                'docs_dir': docs_dir or target_dir,
                'docs_config_path': docs_config,
                'docs_archive_name': docs_archive_name
            }

        logger.debug('yfm docs_properties: %s', docs_properties)
        return docs_properties


# helper methods to work with ya.make files
def _collect_variables(make_list):
    vars = {}
    for set_macro in make_list.project.find_nodes('SET'):
        set_values = set_macro.get_values()
        if not set_values:
            continue
        vars[set_values[0].name] = set_values[1].name
    return vars


def _expand_variables(value, variables):
    import re
    var_re = re.compile(r'(?P<entry>\$\{(?P<var_name>[^\}]+)\})', re.UNICODE)
    for (entry, var_name) in var_re.findall(value, 0):
        if var_name in variables:
            value = value.replace(entry, variables[var_name])
    return value


def _parse_ya_make(ya_make_path):
    from yalibrary import makelists

    make_list = makelists.from_file(ya_make_path)
    vars = _collect_variables(make_list)

    docs_dir_macros = make_list.project.docs_dir.get_values()
    docs_dir = _expand_variables(docs_dir_macros[0].name, vars) if len(docs_dir_macros) > 0 else None

    docs_config_macros = make_list.project.find_nodes('DOCS_CONFIG')
    docs_config_path = _expand_variables(docs_config_macros[0].get_values()[0].name, vars) if len(docs_config_macros) > 0 else None

    docs_macros = make_list.project.docs.get_values()
    docs_archive_name = _expand_variables(docs_macros[0].name, vars) if len(docs_macros) > 0 else None

    return docs_dir, docs_config_path, docs_archive_name
