import contextlib
import json
import logging
import requests
import multiprocessing.dummy

import sandbox.common


VIEWER_PROD_ENV = 'production'
VIEWER_TESTING_ENV = 'testing'
_YFM_BASE_URL = 'https://docs.yandex-team.ru'
_YFM_API_URL = _YFM_BASE_URL + '/api/revisions'
_YFM_UPLOAD_URL = _YFM_BASE_URL + '/docs-api'
_YFM_TARGET_UPLOAD_STATUS = 'FINISHED'
_YFM_FAKE_NOT_FOUND_STATUS = 'NOT_FOUND'
_YFM_MAX_JOB_REGISTRATION_WAIT = 15
_YFM_API_MAX_THREADS = 5

logger = logging.getLogger(__name__)


def _setup(user_agent, token, request_timeout=20):
    import library.python.retry
    import requests
    import requests.adapters
    import requests.exceptions

    http_session = requests.Session()
    http_adapter = requests.adapters.HTTPAdapter(pool_connections=1, pool_maxsize=_YFM_API_MAX_THREADS)
    http_session.mount('https://', http_adapter)
    http_session.headers['Authorization'] = 'OAuth {}'.format(token)
    http_session.headers['User-Agent'] = user_agent

    retry_conf = library.python.retry.DEFAULT_CONF.clone(
        retriable=lambda e: isinstance(e, (requests.exceptions.ConnectionError, requests.exceptions.Timeout))
                            or (isinstance(e, requests.exceptions.HTTPError) and e.response.status_code in (500, 502, 504))) \
        .upto(request_timeout * 3)

    return http_session, retry_conf


def deploy(projects, user_agent, token):
    """Sends requests to YFM API to deploy projects (register a new version previously pushed to the underlying storage)

    :param projects: dict yfm project name -> deploy meta information:
                         - 'author':       string, // commit author
                         - 'timestamp':    string, // UTC ts of commit in ISO format
                         - 'revision':     string, // "version" of the document: svn revision, arc commit hash, some unique identifier for pull requests
                         - 'pr_id':        int,    // Arcanum pull request id (optional)
                         - 'pr_iteration': int,    // number of iteration in Arcanum pull request (optional)
                         - 'date':         string, // commit date (timezone provided by vcs)
                         - 'arcadia-path': string, // Arcadia root relative path to project
    :param user_agent: string to pass as User-Agent request header
    :param token: string with token to use for request authorization
    """
    request_timeout = 20
    http_session, retry_conf = _setup(user_agent, token, request_timeout)
    with _autojoining_pool(threads=_YFM_API_MAX_THREADS) as pool:
        for project, data in projects.iteritems():

            logger.debug('Deploying project {p}, info {i}'.format(p=project, i=data))
            request_data = {
                'project': project,
                'revision': data,
                'type': 'revisions',
                'isExternal': data.get('is_external', False)
            }
            pool.apply_async(_post_release, args=[http_session, request_timeout, request_data, retry_conf])


def release(projects, is_trunk, user_agent, token):
    """Sends requests to YFM API to release projects

    :param projects: dict yfm project name -> release properties,
           which are "revision" - id of version previously deployed to YFM viewer,
                     "environments" - list of VIEWER_PROD_ENV or/and VIEWER_TESTING_ENV values
    :param is_trunk: shows whether "revision" belongs to trunk
    :param user_agent: string to pass as User-Agent request header
    :param token: string with token to use for request authorization
    """

    request_timeout = 20
    http_session, retry_conf = _setup(user_agent, token, request_timeout)
    with _autojoining_pool(threads=_YFM_API_MAX_THREADS) as pool:
        for project, data in projects.iteritems():
            environments = data.get('environments', [])
            if not environments:
                continue

            logger.debug('Releasing project {p} to {env}, revision {rev}'.format(p=project, env=environments, rev=data['revision']))
            for env in environments:
                if not is_trunk and env != VIEWER_TESTING_ENV:
                    logger.debug('Skip release to %s for non trunk deploy', env)
                    continue

                request_data = {
                    'project': project,
                    'revision': {
                        'revision': data['revision'],
                    },
                    'type': env,
                    'isExternal': data.get('is_external', False)
                }
                pool.apply_async(_post_release, args=[http_session, request_timeout, request_data, retry_conf])


def _post_release(http_session, request_timeout, data, retry_conf):
    import library.python.retry

    logger.debug(data)
    response = library.python.retry.retry_call(
        http_session.request,
        f_args=['post', _YFM_API_URL],
        f_kwargs=dict(timeout=(5, request_timeout), json=data),
        conf=retry_conf,
    )
    logger.debug(response)


def upload(token, project, revision, archive_md, archive_html, def_lang_prefix=True, is_external=False):
    """
    Uploads given documentation archive via YFM API.

    :param token:           string with OAuth token for YFM API request authorization
    :param project:         project name
    :param revision:        documentation revision
    :param archive_md:      file object of `pathlib.Path` type with path to documentation archive with md files to be uploaded
    :param archive_html:    file object of `pathlib.Path` type with path to documentation archive with html files to be uploaded
    :param def_lang_prefix: should be set to `True` in case of no `langs` section in the archive's `.yfm` file
    :param is_external:     load documentation on external or internal s3
    """

    from requests_toolbelt.multipart import encoder

    data = encoder.MultipartEncoder(
        fields={
            'docsArchive': (archive_md.name, archive_md.open('rb'), 'text/plain'),
            'docsArchiveHtml': (archive_html.name, archive_html.open('rb'), 'text/plain'),
            'project': project,
            'revision': str(revision),
            'addDefaultLangPrefix': json.dumps(def_lang_prefix),
            'isExternal': json.dumps(is_external),
        }
    )

    logger.debug('Upload data: {d}'.format(d=data))

    response = requests.post(_YFM_UPLOAD_URL + '/upload', data=data, headers={
        'Content-Type': data.content_type,
        'Authorization': 'OAuth ' + token,
    })
    response.raise_for_status()

    prev_status = None
    upload_id = response.json()["id"]

    for slept, _ in sandbox.common.itertools.progressive_yielder(1, 10, float('inf'), sleep_first=False):
        response = requests.get(_YFM_UPLOAD_URL + '/check-upload-status', json={'id': upload_id})  # uff
        if response.status_code == requests.codes.NOT_FOUND and slept < _YFM_MAX_JOB_REGISTRATION_WAIT:
            # Assume the 404 response means the upload job not yet registered - just wait for it
            status = _YFM_FAKE_NOT_FOUND_STATUS
        else:
            response.raise_for_status()
            status = response.json()["status"]
        if status != prev_status:
            yield status
        prev_status = status
        if status == _YFM_TARGET_UPLOAD_STATUS:
            return

@contextlib.contextmanager
def _autojoining_pool(threads):
    pool = multiprocessing.dummy.Pool(threads)
    yield pool

    pool.close()
    pool.join()
