# -*- coding: utf-8 -*-

import os
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.ssh as sdk_ssh
import sandbox.sandboxsdk.parameters as parameters
import sandbox.sandboxsdk.environments as environments
import sandbox.sandboxsdk.process as process

import sandbox.projects.common.build.ArcadiaTask as arcadia_task
import sandbox.projects.common.utils as utils

from sandbox.projects.arcadia.ArcadiaGitSync import SYNC_RESULT


class GitSourceUrlParam(parameters.SandboxStringParameter):
    name = 'git_source_url'
    description = 'Git source url'
    default_value = ''
    required = True


class GitTargetUrlParam(parameters.SandboxStringParameter):
    name = 'git_target_url'
    description = 'Git target url'
    default_value = ''
    required = True


class StartSyncRevisionParam(parameters.SandboxStringParameter):
    name = 'sync_revision_from'
    description = 'Revision to start syncing from'
    default_value = 'HEAD'


class SkipErrorsIntervalsParam(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'skip_errors_intervals'
    description = 'Revision intervals where bad commits should be considered as good ones'
    default_value = []


class OpensourceToolsIdParam(parameters.SandboxStringParameter):
    name = 'opensource_tools_id'
    description = 'Id of oss scripts resource'
    default_value = '500821960'


class DNS64Param(parameters.SandboxBoolParameter):
    name = 'dns64'
    description = 'Use DNS64'


class HideAuthorParam(parameters.SandboxBoolParameter):
    name = 'hide_author'
    description = 'Set committer to default'
    default_value = False


class CommitMessageParam(parameters.SandboxStringParameter):
    name = 'commit_message'
    description = 'Commit message'
    default_value = None


class GitGitSync(arcadia_task.ArcadiaTask):
    """
        Git->Git sync
    """

    type = "GIT_GIT_SYNC"

    environment = [environments.SvnEnvironment()]
    client_tags = ctc.Tag.GENERIC | ctc.Tag.Group.OSX

    cores = 1
    required_ram = 2048
    execution_space = 2048

    input_parameters = [
        GitSourceUrlParam,
        GitTargetUrlParam,
        StartSyncRevisionParam,
        SkipErrorsIntervalsParam,
        OpensourceToolsIdParam,
        DNS64Param,
        HideAuthorParam,
        CommitMessageParam,
    ]

    @property
    def dns(self):
        return ctm.DnsType.DNS64 if self.ctx.get(DNS64Param.name) else ctm.DnsType.DEFAULT

    def _get_opensource_dir_path(self):
        dest = self.path('opensource')
        tools = self.sync_resource(utils.get_or_default(self.ctx, OpensourceToolsIdParam))
        with tarfile.open(tools) as tf:
            tf.extractall(dest)
        return dest

    def _get_start_sync_revision(self):
        return utils.get_or_default(self.ctx, StartSyncRevisionParam)

    def on_execute(self):
        work_dir = self.path("sync")
        os.makedirs(work_dir)

        sync_output_res = self.create_resource(
            description='Sync output',
            resource_path=work_dir,
            resource_type=SYNC_RESULT,
            attributes={
                'ttl': 4
            }
        )

        skip_errors_intervals = []
        for interval in utils.get_or_default(self.ctx, SkipErrorsIntervalsParam) or []:
            skip_errors_intervals += ["--skip-errors-interval", interval.strip()]

        # XXX - move to input parameters

        env = os.environ.copy()
        env["GITHUB_OAUTH_DEVTOOLS_TOKEN"] = self.get_vault_data("YATOOL", "GITHUB_OAUTH_DEVTOOLS_TOKEN")
        with sdk_ssh.Key(self, "akastornov", "arcadia-devtools-ssh-key"):
            os_dir_path = self._get_opensource_dir_path()

            proc_params = [
                    os.path.join(os_dir_path, "bin", "git2git"),
                    "--git-from",  utils.get_or_default(self.ctx, GitSourceUrlParam),
                    "--git-to",  utils.get_or_default(self.ctx, GitTargetUrlParam),
                    "--revision-from", utils.get_or_default(self.ctx, StartSyncRevisionParam),
                    "--work-dir", work_dir,
                    "--logs-dir", work_dir,
                    "--logs-url", "https://proxy.sandbox.yandex-team.ru/{}".format(sync_output_res.id),
                ]

            if utils.get_or_default(self.ctx, HideAuthorParam):
                proc_params += ["--hide-author"]

            commit_msg = utils.get_or_default(self.ctx, CommitMessageParam)
            if commit_msg:
                proc_params += ["--commit-message", commit_msg]

            proc_params += skip_errors_intervals

            process.run_process(
                proc_params,
                log_prefix='sync', outputs_to_one_file=True, work_dir=work_dir, environment=env,
            )


__Task__ = GitGitSync
