# -*- coding: utf-8 -*-

import os
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.ssh as sdk_ssh
import sandbox.sandboxsdk.parameters as parameters
import sandbox.sandboxsdk.environments as environments
import sandbox.sandboxsdk.process as process

import sandbox.projects.common.build.ArcadiaTask as arcadia_task
import sandbox.projects.common.utils as utils

from sandbox.projects.arcadia.ArcadiaGitSync import SYNC_RESULT


class GitUrlParam(parameters.SandboxStringParameter):
    name = 'git_url'
    description = 'Git url to watch PR'
    default_value = ''
    required = True


class GitApiUrlParam(parameters.SandboxStringParameter):
    name = 'git_api_url'
    description = 'Git API url'
    default_value = ''
    required = True


class CiProjectIdParam(parameters.SandboxStringParameter):
    name = 'ci_project_id'
    description = 'Ci project id'
    default_value = ''
    required = True


class OpensourceToolsIdParam(parameters.SandboxStringParameter):
    name = 'opensource_tools_id'
    description = 'Id of oss scripts resource'
    default_value = '501370999'
    required = True


class DNS64Param(parameters.SandboxBoolParameter):
    name = 'dns64'
    description = 'Use DNS64'


class PullRequestSync(arcadia_task.ArcadiaTask):
    """
        Pull-request sync
    """

    type = "PULL_REQUEST_SYNC"

    environment = [environments.SvnEnvironment()]
    client_tags = ctc.Tag.GENERIC | ctc.Tag.Group.OSX
    cores = 17
    input_parameters = [
        GitUrlParam,
        GitApiUrlParam,
        CiProjectIdParam,
        OpensourceToolsIdParam,
        DNS64Param,
    ]

    @property
    def dns(self):
        return ctm.DnsType.DNS64 if self.ctx.get(DNS64Param.name) else ctm.DnsType.DEFAULT

    def _get_opensource_dir_path(self):
        dest = self.path('opensource')
        tools = self.sync_resource(utils.get_or_default(self.ctx, OpensourceToolsIdParam))
        with tarfile.open(tools) as tf:
            tf.extractall(dest)
        return dest

    def on_execute(self):
        work_dir = self.path("sync")
        os.makedirs(work_dir)

        sync_output_res = self.create_resource(
            description='Sync output',
            resource_path=work_dir,
            resource_type=SYNC_RESULT,
            attributes={
                'ttl': 4
            }
        )

        env = os.environ.copy()
        env["GITHUB_OAUTH_DEVTOOLS_TOKEN"] = self.get_vault_data("YATOOL", "GITHUB_OAUTH_DEVTOOLS_TOKEN")
        # XXX - move to input parameters
        with sdk_ssh.Key(self, "akastornov", "arcadia-devtools-ssh-key"):
            os_dir_path = self._get_opensource_dir_path()
            process.run_process(
                [
                    os.path.join(os_dir_path, "bin", "pr"),
                    "--repo", utils.get_or_default(self.ctx, GitUrlParam),
                    "--project", utils.get_or_default(self.ctx, CiProjectIdParam),
                    "--work-dir", work_dir,
                    "--logs-dir", work_dir,
                    "--logs-url", "https://proxy.sandbox.yandex-team.ru/{}".format(sync_output_res.id),
                    "--scripts-dir", os.path.join(os_dir_path, "scripts"),
                ],
                log_prefix='sync', outputs_to_one_file=True, work_dir=work_dir, environment=env
            )


__Task__ = PullRequestSync
