# -*- coding: utf-8 -*-

import time
from datetime import datetime
import logging

from sandbox import sdk2
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk import environments


class AugurPublicRelationsReport(sdk2.Task):
    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        environments = (
            environments.PipEnvironment('grequests', version='0.3.0', use_wheel=True),
            # environments.PipEnvironment('openpyxl', version='2.4.9', use_wheel=True),
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings'),
            environments.PipEnvironment('psycopg2'),
        )

    class Parameters(sdk2.Parameters):
        urls = sdk2.parameters.String(
            "Список URL",
            multiline=True,
            required=True,
        )
        date = sdk2.parameters.String(
            "Дата",
            required=True,
        )
        browser_share = sdk2.parameters.Float(
            "Доля Браузера",
            default=0.166,
            required=True,
        )
        barnavi_log_share = sdk2.parameters.Float(
            "Доля барнавиг лога",
            default=0.5,
            required=True,
        )

    def on_execute(self):
        logging.info("-"*80 + " Started" + "-"*80)
        import grequests
        import requests
        from yt.wrapper.client import YtClient

        setattr(self, '_grequests', grequests)
        setattr(self, '_requests', requests)
        setattr(self, '_YtClient', YtClient)

        dt = datetime.strptime(self.Parameters.date, '%Y-%m-%d')

        urls = self.resolve_redirects()
        self.put_to_alp_inbox(urls, dt)

        logging.info(urls)
        logging.info("-"*80 + " Finished " + "-"*80)

    def resolve_redirects(self):
        headers = self._requests.utils.default_headers()
        headers.update({
            'User-Agent': 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.110 Safari/537.36 OPR/45.0.2552.8888',
        })
        urls = set()
        for url in self.Parameters.urls.splitlines():
            url = url.strip()
            if url:
                urls.add(url)

        rs = (self._grequests.get(url, timeout=5, headers=headers) for url in urls)

        def exc_handler(request, exc):
            logging.info("Request for url %s has been failed: %s" % (request.url, str(exc)))

        responses = self._grequests.map(rs, size=50, exception_handler=exc_handler)
        redirects = {}

        no_response = 0
        response_4xx = []

        for url, response in zip(urls, responses):
            if response is None:
                logging.info("No response for %s" % url)
                no_response += 1
            else:
                if response.status_code % 100 == 4:
                    response_4xx.append(url)
                if response.url != url:
                    redirects[url] = response.url

        if response_4xx:
            retryable = []
            hopeless = []
            for url in response_4xx:
                if url != url.lower():
                    retryable.append(url)
                else:
                    hopeless.append(url)

            for url in hopeless:
                logging.info("Hopeless 4xx for: %s" % url)

            for url in retryable:
                logging.info("Retryable 4xx for: %s" % url)

            if retryable:
                rs = (self._grequests.get(url.lower(), timeout=30) for url in retryable)
                responses = self._grequests.map(rs, size=50)
                for url, response in zip(retryable, responses):
                    if response is None:
                        logging.info("No response for %s" % url)
                        no_response += 1
                    else:
                        if response.status_code % 100 == 4:
                            logging.info("Again 4xx for %s" % (url.lower()))
                        redirects[url] = response.url
        logging.info("No responses count: %s" % no_response)
        return redirects

    def put_to_alp_inbox(self, urls, dt):
        now = datetime.now()

        stream = 'dozor-%04d-%02d-%02d_4' % (now.year, now.month, now.day)
        path = '//home/augur/pr/alp/urls_inbox/%s' % stream

        timestamp = time.mktime(dt.timetuple())

        yt = self._YtClient(proxy='hahn', token=sdk2.Vault.data('robot-augur-yt'))
        rows = []
        for url in urls:
            rows.append({
                'Url': url,
                'Stream': stream,
                'Date': timestamp
            })
        yt.write_table(path, rows)
