import logging
import datetime

from sandbox import sdk2
from sandbox.common.types.task import Status as TaskStatus
from sandbox.sdk2 import parameters
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.autobudget.audit.lib.query import AuditType
from sandbox.projects.autobudget.audit.lib.query import DEFAULT_QUERIES
from sandbox.projects.autobudget.audit.lib.query import DEFAULT_PRE_QUERIES
from sandbox.projects.autobudget.audit.lib.query import DEFAULT_POST_QUERIES

logging.basicConfig(level=logging.DEBUG)


class StringList(parameters.List):
    supported_types = (parameters.String,)


class Text(parameters.String):
    multiline = True


SBYT_TABLE_FRESHNESS = 'SBYT_TABLE_FRESHNESS'
CYPRESS_NODE_AVAILABILITY = 'CYPRESS_NODE_AVAILABILITY'
DATETIME_FORMAT = '%Y-%m-%d %H:%M:%S'

GOLEM_HOSTNAME_SOURCE = 'GOLEM_HOSTNAME_SOURCE'
JUGGLER_ESCALATION_RULE_SOURCE = 'JUGGLER_ESCALATION_RULE_SOURCE'
NO_RESPONSIBLE_USERS_SOURCE = 'NO_RESPONSIBLE_USERS_SOURCE'


class AutobudgetAudit(BaseBinTask):
    """
    Run autobudget audit monitoring
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4096
        disk_space = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        yt_token_vault = parameters.String('YT token', description='Sandbox Vault secret name for YT token', required=True)
        solomon_token_vault = parameters.String('Solomon token', description='Sandbox Vault secret name for Solomon token', default='AUTOBUDGET_AUDIT_SOLOMON_TOKEN')

        name = parameters.String('Name', description='Monitoring name', required=True)

        with parameters.RadioGroup('Cluster group', description='YT cluster group to run monitoring on (Available: "hahn" -> hahn/arnold, "hahn-pre" -> freud)', required=True) as cluster_group:
            cluster_group.values['hahn'] = cluster_group.Value(value='hahn/arnold')
            cluster_group.values['hahn-pre'] = cluster_group.Value(value='freud (hahn-pre)')

        run_time = sdk2.parameters.String('Run time ({})'.format(DATETIME_FORMAT))

        with parameters.String('Cluster resolving type', ui=parameters.String.UI('select')) as cluster_resolving_type:
            cluster_resolving_type.values[SBYT_TABLE_FRESHNESS] = SBYT_TABLE_FRESHNESS
            cluster_resolving_type.values[CYPRESS_NODE_AVAILABILITY] = CYPRESS_NODE_AVAILABILITY
            cluster_resolving_type.default_value = SBYT_TABLE_FRESHNESS

        with cluster_resolving_type.value[SBYT_TABLE_FRESHNESS]:
            source_table_path = parameters.String('Source table', description='Path to YT table with stats for monitoring', required=True)

        with cluster_resolving_type.value[CYPRESS_NODE_AVAILABILITY]:
            cypress_node_path = parameters.String('Cypress node path', description='Path to cypress node to check if cluster is available', required=True)
            lookup_number = parameters.Integer('Number of tables too lookup', default=0)

        dest_cluster = parameters.String('Destination cluster', description='YT cluster for saving results', required=True)
        dest_table_path = parameters.String('Destination table', description='Path to YT table with results', required=True)

        with parameters.String('Audit type', required=True) as audit_type:
            audit_type.values[AuditType.CUSTOM.value] = audit_type.Value(AuditType.CUSTOM.value, default=True)
            for query_type in DEFAULT_QUERIES.keys():
                audit_type.values[query_type.value] = query_type.value

        with audit_type.value[AuditType.CUSTOM.value]:
            custom_yql_query = Text('Custom YQL query', description='Custom YQL query of monitoring')
            custom_yql_pre_query = Text('Custom YQL pre query', description='Custom YQL query to count values to be inserted in main query in {{ ResultColumnName }} format')

        use_custom_post_query = sdk2.parameters.Bool('Use custom query for pretty output')
        with use_custom_post_query.value[True]:
            custom_yql_post_query = Text('Custom YQL post query', description='Custom YQL query to pretty output audit result')
        yql_post_query_delay = parameters.Integer('Post query delay in seconds', default=30)

        yql_syntax_version = parameters.Integer('YQL syntax version', default=0)

        st_queue = parameters.String('ST queue', description='Startrek queue for tickets with problems', required=True)
        st_components = StringList('ST components', description='Startrek components for ticket', required=True)
        st_tags = StringList('ST tags', description='Startrek ticket tags (comma separated)', default=['auto_audit'])

        with parameters.String('Responsible users source', ui=parameters.String.UI('select')) as responsible_users_source:
            responsible_users_source.values[GOLEM_HOSTNAME_SOURCE] = GOLEM_HOSTNAME_SOURCE
            responsible_users_source.values[JUGGLER_ESCALATION_RULE_SOURCE] = JUGGLER_ESCALATION_RULE_SOURCE
            responsible_users_source.values[NO_RESPONSIBLE_USERS_SOURCE] = NO_RESPONSIBLE_USERS_SOURCE
            responsible_users_source.default_value = GOLEM_HOSTNAME_SOURCE

        with responsible_users_source.value[GOLEM_HOSTNAME_SOURCE]:
            golem_hostname = parameters.String('Golem hostname', description='Hostname with responsilbe list in Golem', required=True)

        with responsible_users_source.value[JUGGLER_ESCALATION_RULE_SOURCE]:
            rule_id = parameters.String('Juggler rule_id', description='Escalation rule with list of logins', required=True)

        threshold = parameters.Integer('Threshold for monitoring to fire', description='Monitoring creates ticket only if count of bad rows exceeds threshold', default_value=0)
        solomon_cluster = parameters.String('Solomon cluster', default_value='debug')
        solomon_audit_name = parameters.String('Solomon audit name', description='Audit name for solomon sensors', required=True)
        yql_pool = parameters.String('Yql pool name', required=False)

    def on_execute(self):
        from yabs.autobudget.pylibs.audit import AutobudgetAuditApp
        from yabs.autobudget.pylibs.audit import ClusterResolvingType
        from yabs.autobudget.pylibs.audit import NoDataToProcess
        from yabs.autobudget.pylibs.audit import GolemHostName
        from yabs.autobudget.pylibs.audit import JugglerEscalationRuleId

        prev_last_processed_table = None
        if self.scheduler and self.scheduler > 0:
            last_succeeded_task = sdk2.Task.find(scheduler=self.scheduler, status=TaskStatus.SUCCESS, order='-id', limit=1).first()
            if last_succeeded_task:
                prev_last_processed_table = last_succeeded_task.Context.last_processed_table

        audit_type = AuditType(self.Parameters.audit_type)
        if audit_type == AuditType.CUSTOM:
            yql_query = self.Parameters.custom_yql_query
            yql_pre_query = self.Parameters.custom_yql_pre_query
            yql_post_query = None
        else:
            yql_query = DEFAULT_QUERIES[audit_type]
            yql_pre_query = DEFAULT_PRE_QUERIES.get(audit_type)
            yql_post_query = DEFAULT_POST_QUERIES.get(audit_type)
        if self.Parameters.use_custom_post_query:
            yql_post_query = self.Parameters.custom_yql_post_query

        cluster_resolving_type = ClusterResolvingType(self.Parameters.cluster_resolving_type)
        if cluster_resolving_type is ClusterResolvingType.SBYT_TABLE_FRESHNESS:
            cluster_resolving_argument = self.Parameters.source_table_path
        elif cluster_resolving_type is ClusterResolvingType.CYPRESS_NODE_AVAILABILITY:
            cluster_resolving_argument = self.Parameters.cypress_node_path
        else:
            raise ValueError('Unsupported cluster resolving type {}'.format(self.Parameters.cluster_resolving_type))

        if self.Parameters.run_time:
            run_time = datetime.datetime.strptime(self.Parameters.run_time, DATETIME_FORMAT)
        else:
            run_time = datetime.datetime.now()
        self.Context.run_time = run_time.strftime(DATETIME_FORMAT)

        responsible_users_src = None
        if self.Parameters.responsible_users_source == GOLEM_HOSTNAME_SOURCE:
            responsible_users_src = GolemHostName(hostname=str(self.Parameters.golem_hostname))
        elif self.Parameters.responsible_users_source == JUGGLER_ESCALATION_RULE_SOURCE:
            responsible_users_src = JugglerEscalationRuleId(rule_id=str(self.Parameters.rule_id))

        try:
            app = AutobudgetAuditApp.create(
                token=sdk2.Vault.data(self.Parameters.yt_token_vault),
                name=self.Parameters.name,
                cluster_group=self.Parameters.cluster_group,
                cluster_resolving_type=cluster_resolving_type,
                cluster_resolving_argument=cluster_resolving_argument,
                dest_cluster=self.Parameters.dest_cluster,
                dest_table_path=self.Parameters.dest_table_path,
                st_queue=self.Parameters.st_queue,
                st_components=self.Parameters.st_components,
                st_tags=self.Parameters.st_tags,
                responsible_users_src=responsible_users_src,
                threshold=self.Parameters.threshold,
                solomon_token=sdk2.Vault.data(self.Parameters.solomon_token_vault),
                solomon_cluster=self.Parameters.solomon_cluster,
                solomon_audit_name=self.Parameters.solomon_audit_name,
                logger=logging,
                yql_pool=self.Parameters.yql_pool,
                prev_last_processed_table=prev_last_processed_table,
                lookup_number=self.Parameters.lookup_number,
                yql_post_query=yql_post_query,
                yql_post_query_delay=self.Parameters.yql_post_query_delay,
            )
        except NoDataToProcess:
            self.Context.last_processed_table = prev_last_processed_table
        else:
            app.run(
                yql_query=yql_query,
                yql_pre_query=yql_pre_query,
                run_time=run_time,
            )
            self.Context.last_processed_table = app.last_processed_table
