QUERY = '''
-- CPC audit --
---------------

-- Helpers:

$ToDatetime = ($seconds) -> {
    RETURN DateTime::MakeTzDatetime(AddTimezone(DateTime::FromSeconds(CAST($seconds AS Uint32)), 'Europe/Moscow'));
};

$RoundToWeekStart = ($timestamp) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfWeek($timestamp));
};

$ToSeconds = ($timestamp) -> {
    RETURN DateTime::ToSeconds($timestamp);
};

$MinOfExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, MIN_OF(COALESCE($a, $b), COALESCE($b, $a)));
};

$MaxOfExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, MAX_OF(COALESCE($a, $b), COALESCE($b, $a)));
};

-- Arguments:

$Now = DateTime::MakeTzDatetime(TzDatetime('{{ Now }},Europe/Moscow'));

$CurWeek = $RoundToWeekStart($Now);

$PrevWeek = $CurWeek - DateTime::IntervalFromDays(7);

-- Audit logic:

$CheckIfHasRestart = ($start_time, $finish_time, $next_start_time) -> {
    RETURN ($next_start_time < $finish_time)
        OR (DateTime::ToDays($ToDatetime($finish_time) - $ToDatetime($start_time)) < 7);
};

$GetAcceptableExceedRate = ($clicks, $start_time, $next_start_time) -> {
    RETURN IF(
        ($clicks < 100)
        OR $CheckIfHasRestart($start_time, $ToSeconds($CurWeek), $next_start_time),
        104.0,
        4.0);
};

$GetExceed = ($expected, $actual) -> {
    RETURN $actual - $expected;
};

$GetExceedRate = ($target, $cost, $units) -> {
    $fraction = IF($target > 0 AND $units > 0, 1.0 * $cost / ($units * $target), 1) - 1;
    RETURN Math::Round($fraction * 100, -2);
};

$GetCPU = ($cost, $units) -> {
    RETURN 1.0 * $cost / $units;
};

$IsCPC = ($row) -> {
    RETURN $row.OptimizeType = 0 AND ($row.AvgCost > 0 OR $row.AvgCostCur > 0);
};

$CPCStrategyModifyTime = ($row) -> {
    RETURN IF($IsCPC($row), $row.ModifyTime, 0);
};

$CPCOrderVersions = SELECT
    OrderID,
    MAX_BY(StartTime, $CPCStrategyModifyTime(TableRow())) AS StartTime,
    MAX_BY(LastUpdateTime, $CPCStrategyModifyTime(TableRow())) AS LastUpdateTime,
    MAX_BY(LimitWeek, $CPCStrategyModifyTime(TableRow())) AS LimitWeek,
    MAX_BY(LimitWeekCur, $CPCStrategyModifyTime(TableRow())) AS LimitWeekCur,
    MAX_BY(AvgCost, $CPCStrategyModifyTime(TableRow())) AS LimitAvgCPC,
    MAX_BY(AvgCostCur, $CPCStrategyModifyTime(TableRow())) AS LimitAvgCPCCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE
    $ToDatetime(StartTime) < $PrevWeek
    AND $ToDatetime(LastUpdateTime) < $CurWeek
GROUP BY
    OrderID
HAVING  -- Last cpc strategy for the beginning of the week
    MAX_BY(
        $IsCPC(TableRow())
      , ModifyTime
    )
UNION ALL
SELECT
    OrderID,
    StartTime,
    MAX_BY(LastUpdateTime, $CPCStrategyModifyTime(TableRow())) AS LastUpdateTime,
    MAX_BY(LimitWeek, $CPCStrategyModifyTime(TableRow())) AS LimitWeek,
    MAX_BY(LimitWeekCur, $CPCStrategyModifyTime(TableRow())) AS LimitWeekCur,
    MAX_BY(AvgCost, $CPCStrategyModifyTime(TableRow())) AS LimitAvgCPC,
    MAX_BY(AvgCostCur, $CPCStrategyModifyTime(TableRow())) AS LimitAvgCPCCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE  -- Was started or restarted during the week
    $ToDatetime(StartTime) >= $PrevWeek
    AND $ToDatetime(StartTime) < $CurWeek
GROUP BY
    OrderID, StartTime
;

$CPCOrderVersions = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    LimitWeek,
    LimitWeekCur,
    LimitAvgCPC,
    LimitAvgCPCCur,
    COALESCE(MIN(IF(B.StartTime > A.StartTime, B.StartTime, $ToSeconds($CurWeek))), $ToSeconds($CurWeek)) AS NextStartTime,
FROM
    $CPCOrderVersions AS A
LEFT JOIN
    $CPCOrderVersions AS B
ON
    A.OrderID = B.OrderID
GROUP BY
    A.OrderID AS OrderID
  , A.StartTime AS StartTime
  , A.LastUpdateTime AS LastUpdateTime
  , A.LimitWeek AS LimitWeek
  , A.LimitWeekCur AS LimitWeekCur
  , A.LimitAvgCPC AS LimitAvgCPC
  , A.LimitAvgCPCCur AS LimitAvgCPCCur
HAVING
    LimitAvgCPC > 0 OR LimitAvgCPCCur > 0
;

$Log = SELECT
    orderid,
    autobudgetstarttime,
    autobudgetsoftrestarttime,
    eventcost,
    costcur,
    eventtime,
    countertype,
FROM
    RANGE(
        `//logs/bs-chevent-log/1d`
      , DateTime::Format('%Y-%m-%d')($PrevWeek)
      , DateTime::Format('%Y-%m-%d')($CurWeek - DateTime::IntervalFromDays(1))
    )
WHERE
    fraudbits = '0'
;

$UndoLog = SELECT
    orderid,
    autobudgetstarttime,
    autobudgetsoftrestarttime,
    oldeventcost,
    oldcostcur,
    neweventcost,
    newcostcur,
    eventtime,
    countertype,
    fraudbits,
FROM
    RANGE(
        `//logs/bs-undochevent-log/1d`
      , DateTime::Format('%Y-%m-%d')($PrevWeek)
      , DateTime::Format('%Y-%m-%d')($CurWeek - DateTime::IntervalFromDays(1))
    )
;

$CPCOrderVersions = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    LimitWeek,
    LimitWeekCur,
    LimitAvgCPC,
    LimitAvgCPCCur,
    NextStartTime,
    SUM(CAST(L.eventcost AS Int64)) AS Cost,
    SUM(CAST(L.costcur AS Int64)) AS CostCur,
    SUM(CAST(L.countertype = '2' AS Int64)) AS Clicks,
    MIN(CAST(L.eventtime AS Int64)) AS FirstEvent,
    MAX(CAST(L.eventtime AS Int64)) AS LastEvent,
FROM
    $CPCOrderVersions AS O
LEFT JOIN
    $Log AS L
ON
    (
        CAST(L.orderid AS Int64) = O.OrderID
        AND CAST(L.autobudgetstarttime AS Int64) = O.StartTime
    )
WHERE
    L.orderid IS NULL OR (
        $ToDatetime(eventtime) >= $ToDatetime(MAX_OF($ToSeconds($PrevWeek), O.StartTime))
        AND $ToDatetime(eventtime) < $CurWeek
    )
GROUP BY
    O.OrderID AS OrderID
  , O.StartTime AS StartTime
  , O.LastUpdateTime AS LastUpdateTime
  , O.LimitWeek AS LimitWeek
  , O.LimitWeekCur AS LimitWeekCur
  , O.LimitAvgCPC AS LimitAvgCPC
  , O.LimitAvgCPCCur AS LimitAvgCPCCur
  , O.NextStartTime AS NextStartTime
;

$CPCOrderVersions = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    LimitWeek,
    LimitWeekCur,
    LimitAvgCPC,
    LimitAvgCPCCur,
    NextStartTime,
    Cost - COALESCE(SUM(CAST(L.oldeventcost AS Int64) - CAST(L.neweventcost AS Int64)), 0l) AS Cost,
    CostCur - COALESCE(SUM(CAST(L.oldcostcur AS Int64) - CAST(L.newcostcur AS Int64)), 0l) AS CostCur,
    Clicks - COALESCE(SUM(CAST(L.countertype = '2' AND fraudbits <> '0' AS Int64)), 0l) AS Clicks,
    $MinOfExisted(FirstEvent, MIN(CAST(L.eventtime AS Int64))) AS FirstEvent,
    $MaxOfExisted(LastEvent, Max(CAST(L.eventtime AS Int64))) AS LastEvent,
FROM
    $CPCOrderVersions AS O
LEFT JOIN
    $UndoLog AS L
ON
    (
        CAST(L.orderid AS Int64) = O.OrderID
        AND CAST(L.autobudgetstarttime AS Int64) = O.StartTime
    )
WHERE
    L.orderid IS NULL OR (
        $ToDatetime(eventtime) >= $ToDatetime(MAX_OF($ToSeconds($PrevWeek), O.StartTime))
        AND $ToDatetime(eventtime) < $CurWeek
    )
GROUP BY
    O.OrderID AS OrderID
  , O.StartTime AS StartTime
  , O.LastUpdateTime AS LastUpdateTime
  , O.LimitWeek AS LimitWeek
  , O.LimitWeekCur AS LimitWeekCur
  , O.LimitAvgCPC AS LimitAvgCPC
  , O.LimitAvgCPCCur AS LimitAvgCPCCur
  , O.NextStartTime AS NextStartTime
  , O.Cost AS Cost
  , O.CostCur AS CostCur
  , O.Clicks AS Clicks
  , O.FirstEvent AS FirstEvent
  , O.LastEvent AS LastEvent
;

SELECT
    $ToSeconds($Now) AS RunTime,
    O.OrderID AS OrderID,
    $GetExceed(O.LimitAvgCPC, $GetCPU(O.Cost, O.Clicks)) AS Exceed,
    $GetExceed(O.LimitAvgCPCCur, $GetCPU(O.CostCur, O.Clicks)) AS ExceedCur,
    $GetExceedRate(O.LimitAvgCPC, O.Cost, O.Clicks) AS ExceedRate,
    $GetExceedRate(O.LimitAvgCPCCur, O.CostCur, O.Clicks) AS ExceedCurRate,
    'week' AS LimitType,
    LimitWeek AS LimitBudget,
    LimitWeekCur AS LimitBudgetCur,
    O.LimitAvgCPC AS LimitAvgCPC,
    O.LimitAvgCPCCur AS LimitAvgCPCCur,
    O.Cost AS Budget,
    O.CostCur AS BudgetCur,
    $GetCPU(O.Cost, O.Clicks) AS AvgCPC,
    $GetCPU(O.CostCur, O.Clicks) AS AvgCPCCur,
    O.Clicks AS Clicks,
    O.StartTime AS AutobudgetStartTime,
    MAX_OF(O.StartTime, $ToSeconds($PrevWeek)) AS PeriodStartTime,
    O.NextStartTime AS PeriodFinishTime,
    O.FirstEvent AS FirstEvent,
    O.LastEvent AS LastEvent,
FROM
    $CPCOrderVersions AS O
INNER JOIN
    `//home/yabs/dict/CaesarOrderInfo` AS I
ON
    O.OrderID = I.OrderID
WHERE
    (LimitAvgCPC > 0 OR LimitAvgCPCCur > 0)
    AND (
        (NOT (I.CurrencyID != 0) AND $GetExceedRate(O.LimitAvgCPC, O.Cost, O.Clicks) > $GetAcceptableExceedRate(O.Clicks, O.StartTime, O.NextStartTime))
        OR ((I.CurrencyID != 0) AND $GetExceedRate(O.LimitAvgCPCCur, O.CostCur, O.Clicks) > $GetAcceptableExceedRate(O.Clicks, O.StartTime, O.NextStartTime))
    )
;

SELECT
    $ToSeconds($Now) AS RunTime,
    COUNT(*) AS TotalCheckedRows,
FROM
    $CPCOrderVersions
;
'''
