PRE_QUERY = '''
-- CPM Audit MinStartTime --
----------------------------

-- Helpers:

$ToDatetime = ($seconds) -> {
    RETURN DateTime::MakeTzDatetime(AddTimezone(DateTime::FromSeconds(CAST($seconds AS Uint32)), 'Europe/Moscow'));
};

$RoundToWeekStart = ($datetime) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfWeek($datetime));
};

-- Arguments:

$Now = DateTime::MakeTzDatetime(TzDatetime('{{ Now }},Europe/Moscow'));

$Today = DateTime::MakeTzDatetime(DateTime::StartOfDay($Now));

$Yesterday = $Today - DateTime::IntervalFromDays(1);

$CurWeek = $RoundToWeekStart($Now);

$PrevWeek = $CurWeek - DateTime::IntervalFromDays(7);

-- Audit logic:

$IsCPMLimited = ($row) -> {
    RETURN ($row.AvgCPMCur > 0 OR $row.AvgCPM > 0);
};

SELECT
    DateTime::Format('%Y-%m-%dT%H:%M:%S,%Z')(COALESCE($ToDatetime(MIN(StartTime)), $PrevWeek)) AS MinStartTime,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE
    $IsCPMLimited(TableRow())
    AND $ToDatetime(ModifyTime) < $Today
    AND $ToDatetime(PeriodBudgetFinish) < $Today
    AND $ToDatetime(PeriodBudgetFinish) >= $Yesterday
;
'''

QUERY = '''
PRAGMA AnsiInForEmptyOrNullableItemsCollections;
-- CPM audit with versioning --
-------------------------------

-- Helpers:

$ToDatetime = ($seconds) -> {
    RETURN DateTime::MakeTzDatetime(AddTimezone(DateTime::FromSeconds(CAST($seconds AS Uint32)), 'Europe/Moscow'));
};

$ToSeconds = ($datetime) -> {
    RETURN DateTime::ToSeconds($datetime);
};

$RoundToWeekStart = ($datetime) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfWeek($datetime));
};

$RoundToDayStart = ($datetime) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfDay($datetime));
};

$script = @@
from yql.typing import List
from yql.typing import Optional
from yql.typing import Uint32


def range_days(fr: Optional[Uint32], to: Optional[Uint32]) -> List[Uint32]:
    result = []
    if fr is None or to is None:
        return result
    while fr <= to:
        result.append(fr)
        fr += 24 * 60 * 60
    return result
@@;

$RangeDays = Python3::range_days($script);

-- Arguments:

$MinStartTime = DateTime::MakeTzDatetime(TzDatetime('{{ MinStartTime }}'));

$Now = DateTime::MakeTzDatetime(TzDatetime('{{ Now }},Europe/Moscow'));

$Today = $RoundToDayStart($Now);

$Yesterday = $Today - DateTime::IntervalFromDays(1);

$CurWeek = $RoundToWeekStart($Now);

$PrevWeek = $CurWeek - DateTime::IntervalFromDays(7);

$MinStartDay = MIN_OF($RoundToDayStart($MinStartTime), $PrevWeek);

-- Audit logic:

$CheckIfHasRestart = ($row) -> {
    RETURN (
        $row.PeriodFinishTimeReal - $row.PeriodStartTimeReal
    ) < (
        $row.PeriodFinishTime - $row.PeriodStartTime
    );
};

$GetAcceptableExceedRate = ($row) -> {
    RETURN IF(
        ($row.Shows < 10000)
        OR $CheckIfHasRestart($row)
      , 100.0
      , 4.0
    );
};

$GetExceed = ($expected, $actual) -> {
    RETURN $actual - $expected;
};

$GetExceedRate = ($target, $cost, $units) -> {
    $fraction = IF($target > 0 AND $units > 0, 1.0 * $cost / ($units * $target), 1) - 1;
    RETURN Math::Round($fraction * 100, -2);
};

$GetCPU = ($cost, $units) -> {
    RETURN 1.0 * $cost / $units;
};

$GetLimitType = ($weekLimit, $periodLimit) -> {
    RETURN IF($weekLimit > 0, 'week', IF($periodLimit > 0, 'period', 'unknown'));
};

$GetLimitBudget = ($weekLimit, $periodLimit) -> {
    RETURN MAX_OF($weekLimit, $periodLimit);
};

$IsCPMLimited = ($row) -> {
    RETURN ($row.AvgCPMCur > 0 OR $row.AvgCPM > 0);
};

$IsPeriod = ($row) -> {
    RETURN ($row.PeriodBudgetLimit > 0 OR $row.PeriodBudgetLimitCur > 0);
};

$CPMOrdersVersions = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    MAX_BY(PeriodBudgetFinish, ModifyTime) AS PeriodBudgetFinish,
    MAX_BY(PeriodBudgetLimit, ModifyTime) AS PeriodBudgetLimit,
    MAX_BY(PeriodBudgetLimitCur, ModifyTime) AS PeriodBudgetLimitCur,
    MAX_BY(LimitWeek, ModifyTime) AS LimitWeek,
    MAX_BY(LimitWeekCur, ModifyTime) AS LimitWeekCur,
    MAX_BY(AvgCPM, ModifyTime) AS LimitAvgCPM,
    MAX_BY(AvgCPMCur, ModifyTime) AS LimitAvgCPMCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE
    $IsCPMLimited(TableRow())
    AND $IsPeriod(TableRow())
GROUP BY
    OrderID, StartTime, LastUpdateTime
HAVING
    $ToDatetime(MAX_BY(PeriodBudgetFinish, ModifyTime)) <= $Today
    AND $ToDatetime(MAX_BY(PeriodBudgetFinish, ModifyTime)) > $Yesterday
UNION ALL
SELECT
    OrderID,
    MAX_BY(StartTime, ModifyTime) AS StartTime,
    MAX_BY(LastUpdateTime, ModifyTime) AS LastUpdateTime,
    MAX_BY(PeriodBudgetFinish, ModifyTime) AS PeriodBudgetFinish,
    MAX_BY(PeriodBudgetLimit, ModifyTime) AS PeriodBudgetLimit,
    MAX_BY(PeriodBudgetLimitCur, ModifyTime) AS PeriodBudgetLimitCur,
    MAX_BY(LimitWeek, ModifyTime) AS LimitWeek,
    MAX_BY(LimitWeekCur, ModifyTime) AS LimitWeekCur,
    MAX_BY(LimitAvgCPM, ModifyTime) AS LimitAvgCPM,
    MAX_BY(LimitAvgCPMCur, ModifyTime) AS LimitAvgCPMCur,
FROM (
    SELECT
        OrderID,
        MAX(ModifyTime) AS ModifyTime,
        MAX_BY(StartTime, ModifyTime) AS StartTime,
        MAX_BY(LastUpdateTime, ModifyTime) AS LastUpdateTime,
        MAX_BY(PeriodBudgetFinish, ModifyTime) AS PeriodBudgetFinish,
        MAX_BY(PeriodBudgetLimit, ModifyTime) AS PeriodBudgetLimit,
        MAX_BY(PeriodBudgetLimitCur, ModifyTime) AS PeriodBudgetLimitCur,
        MAX_BY(LimitWeek, ModifyTime) AS LimitWeek,
        MAX_BY(LimitWeekCur, ModifyTime) AS LimitWeekCur,
        MAX_BY(AvgCPM, ModifyTime) AS LimitAvgCPM,
        MAX_BY(AvgCPMCur, ModifyTime) AS LimitAvgCPMCur,
    FROM
        `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
    WHERE
        $CurWeek = $Today
        AND $ToDatetime(StartTime) < $PrevWeek
    GROUP BY
        OrderID
    HAVING
        (
            MAX_BY(AvgCPM, ModifyTime) > 0
            OR MAX_BY(AvgCPMCur, ModifyTime) > 0
        ) AND (
            MAX_BY(PeriodBudgetLimit, ModifyTime) = 0
            AND MAX_BY(PeriodBudgetLimitCur, ModifyTime) = 0
        )
    UNION ALL
    SELECT
        OrderID,
        MAX(ModifyTime) AS ModifyTime,
        MAX_BY(StartTime, ModifyTime) AS StartTime,
        MAX_BY(LastUpdateTime, ModifyTime) AS LastUpdateTime,
        MAX_BY(PeriodBudgetFinish, ModifyTime) AS PeriodBudgetFinish,
        MAX_BY(PeriodBudgetLimit, ModifyTime) AS PeriodBudgetLimit,
        MAX_BY(PeriodBudgetLimitCur, ModifyTime) AS PeriodBudgetLimitCur,
        MAX_BY(LimitWeek, ModifyTime) AS LimitWeek,
        MAX_BY(LimitWeekCur, ModifyTime) AS LimitWeekCur,
        MAX_BY(AvgCPM, ModifyTime) AS LimitAvgCPM,
        MAX_BY(AvgCPMCur, ModifyTime) AS LimitAvgCPMCur,
    FROM
        `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
    WHERE
        $CurWeek = $Today
        AND $IsCPMLimited(TableRow())
        AND NOT $IsPeriod(TableRow())
        AND $ToDatetime(StartTime) >= $PrevWeek
        AND $ToDatetime(StartTime) < $CurWeek
    GROUP BY
        OrderID, StartTime, LastUpdateTime
)
GROUP BY
    OrderID, StartTime, LastUpdateTime
;

$PeriodStartTime = ($periodBudgetLimit, $periodBudgetLimitCur, $startTime) -> {
    RETURN IF($periodBudgetLimit > 0 OR $periodBudgetLimitCur > 0, $startTime, $ToSeconds($PrevWeek));
};

$PeriodFinishTime = ($periodBudgetLimit, $periodBudgetLimitCur, $periodBudgetFinish) -> {
    RETURN IF($periodBudgetLimit > 0 OR $periodBudgetLimitCur > 0, $periodBudgetFinish, $ToSeconds($CurWeek));
};

$CPMOrders = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    PeriodBudgetFinish,
    PeriodBudgetLimit,
    PeriodBudgetLimitCur,
    LimitWeek,
    LimitWeekCur,
    LimitAvgCPM,
    LimitAvgCPMCur,
    $PeriodStartTime(PeriodBudgetLimit, PeriodBudgetLimitCur, LastUpdateTime) AS PeriodStartTime,
    $PeriodFinishTime(PeriodBudgetLimit, PeriodBudgetLimitCur, PeriodBudgetFinish) AS PeriodFinishTime,
    COALESCE(MIN(
        IF(
            B.LastUpdateTime > A.LastUpdateTime
          , B.LastUpdateTime
          , $PeriodFinishTime(PeriodBudgetLimit, PeriodBudgetLimitCur, PeriodBudgetFinish)
        )
    ), $PeriodFinishTime(PeriodBudgetLimit, PeriodBudgetLimitCur, PeriodBudgetFinish)) AS NextStartTime,
FROM
    $CPMOrdersVersions AS A
LEFT JOIN
    $CPMOrdersVersions AS B
ON
    A.OrderID = B.OrderID
GROUP BY
    A.OrderID AS OrderID
  , A.StartTime AS StartTime
  , A.LastUpdateTime AS LastUpdateTime
  , A.PeriodBudgetFinish AS PeriodBudgetFinish
  , A.PeriodBudgetLimit AS PeriodBudgetLimit
  , A.PeriodBudgetLimitCur AS PeriodBudgetLimitCur
  , A.LimitWeek AS LimitWeek
  , A.LimitWeekCur AS LimitWeekCur
  , A.LimitAvgCPM AS LimitAvgCPM
  , A.LimitAvgCPMCur AS LimitAvgCPMCur
;

$CPMOrders = SELECT
    O.OrderID AS OrderID,
    StartTime,
    LastUpdateTime,
    PeriodBudgetFinish,
    PeriodBudgetLimit,
    PeriodBudgetLimitCur,
    LimitWeek,
    LimitWeekCur,
    LimitAvgCPM,
    LimitAvgCPMCur,
    PeriodStartTime,
    PeriodFinishTime,
    MAX_OF(PeriodStartTime, LastUpdateTime) AS PeriodStartTimeReal,
    MIN_OF(PeriodFinishTime, NextStartTime) AS PeriodFinishTimeReal,
    NextStartTime,
    CurrencyID,
    $RangeDays(DateTime::ToSeconds($MinStartDay) , DateTime::ToSeconds($Now)) AS Day,
FROM
    $CPMOrders AS O
INNER JOIN
    (SELECT OrderID, CurrencyID FROM `//home/yabs/dict/CaesarOrderInfo`) AS I
ON
    O.OrderID = I.OrderID
;

$CPMOrders = SELECT
    *
FROM
    $CPMOrders
FLATTEN BY
    Day
;

$Logs = SELECT
    orderid,
    autobudgetstarttime,
    autobudgetsoftrestarttime,
    eventcost,
    costcur,
    eventtime,
    countertype,
    producttype,
    ots,
    $ToSeconds($RoundToDayStart($ToDatetime(eventtime))) AS Day,
FROM
    RANGE(
        `//cooked_logs/bs-chevent-cooked-log/1d`
      , DateTime::Format('%Y-%m-%d')($ToDatetime($MinStartTime))
      , DateTime::Format('%Y-%m-%d')($Yesterday)
    )
WHERE
    fraudbits = 0
;

$CPMOrdersExpenses = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    PeriodBudgetFinish,
    PeriodBudgetLimit,
    PeriodBudgetLimitCur,
    LimitWeek,
    LimitWeekCur,
    LimitAvgCPM,
    LimitAvgCPMCur,
    PeriodStartTime,
    PeriodFinishTime,
    PeriodStartTimeReal,
    PeriodFinishTimeReal,
    NextStartTime,
    CurrencyID,
    SUM(L.eventcost) AS Cost,
    SUM(L.costcur) AS CostCur,
    MIN(L.eventtime) AS FirstEvent,
    MAX(L.eventtime) AS LastEvent,
    SUM(IF(
          L.producttype IN ('video-creative-reach-indoor', 'video-creative-reach-outdoor')
        , CAST(L.countertype == 1 AS Int64) * L.ots / 1000000
        , CAST(L.countertype == 1 AS Int64)
    )) AS Shows,
    Day,
FROM
    $CPMOrders AS O
LEFT JOIN
    $Logs AS L
ON
    (
        L.orderid = O.OrderID
        AND L.autobudgetstarttime = O.StartTime
        AND L.autobudgetsoftrestarttime = O.LastUpdateTime
        AND L.Day = O.Day
    )
WHERE
    L.orderid IS NULL OR (
        $ToDatetime(L.eventtime) >= $ToDatetime(O.PeriodStartTime)
        AND $ToDatetime(L.eventtime) < $ToDatetime(O.PeriodFinishTime)
    )
GROUP BY
    OrderID AS OrderID
  , StartTime AS StartTime
  , LastUpdateTime AS LastUpdateTime
  , PeriodBudgetLimit AS PeriodBudgetLimit
  , PeriodBudgetLimitCur AS PeriodBudgetLimitCur
  , PeriodBudgetFinish AS PeriodBudgetFinish
  , CurrencyID AS CurrencyID
  , LimitWeek AS LimitWeek
  , LimitWeekCur AS LimitWeekCur
  , LimitAvgCPM AS LimitAvgCPM
  , LimitAvgCPMCur AS LimitAvgCPMCur
  , PeriodStartTime AS PeriodStartTime
  , PeriodFinishTime AS PeriodFinishTime
  , PeriodStartTimeReal AS PeriodStartTimeReal
  , PeriodFinishTimeReal AS PeriodFinishTimeReal
  , NextStartTime AS NextStartTime
  , O.Day AS Day
;

$CPMOrdersExpenses = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    PeriodBudgetFinish,
    PeriodBudgetLimit,
    PeriodBudgetLimitCur,
    LimitWeek,
    LimitWeekCur,
    LimitAvgCPM,
    LimitAvgCPMCur,
    PeriodStartTime,
    PeriodFinishTime,
    PeriodStartTimeReal,
    PeriodFinishTimeReal,
    NextStartTime,
    CurrencyID,
    SUM(COALESCE(Cost, 0)) AS Cost,
    SUM(COALESCE(CostCur, 0)) AS CostCur,
    SUM(COALESCE(Shows, 0)) AS Shows,
    MIN(FirstEvent) AS FirstEvent,
    MAX(LastEvent) AS LastEvent,
FROM
    $CPMOrdersExpenses
GROUP BY
    OrderID AS OrderID
  , StartTime AS StartTime
  , LastUpdateTime AS LastUpdateTime
  , PeriodBudgetLimit AS PeriodBudgetLimit
  , PeriodBudgetLimitCur AS PeriodBudgetLimitCur
  , PeriodBudgetFinish AS PeriodBudgetFinish
  , LimitWeek AS LimitWeek
  , LimitWeekCur AS LimitWeekCur
  , LimitAvgCPM AS LimitAvgCPM
  , LimitAvgCPMCur AS LimitAvgCPMCur
  , PeriodStartTime AS PeriodStartTime
  , PeriodFinishTime AS PeriodFinishTime
  , PeriodStartTimeReal AS PeriodStartTimeReal
  , PeriodFinishTimeReal AS PeriodFinishTimeReal
  , NextStartTime AS NextStartTime
  , CurrencyID AS CurrencyID
;

SELECT
    $ToSeconds($Now) as RunTime,
    OrderID,
    CurrencyID,

    $GetExceed(LimitAvgCPM, $GetCPU(Cost, Shows)) AS Exceed,
    $GetExceed(LimitAvgCPMCur, $GetCPU(CostCur, Shows)) AS ExceedCur,
    $GetExceedRate(LimitAvgCPM, Cost, Shows) AS ExceedRate,
    $GetExceedRate(LimitAvgCPMCur, CostCur, Shows) AS ExceedCurRate,

    $GetLimitType(LimitWeek, PeriodBudgetLimit) AS LimitType,
    $GetLimitBudget(LimitWeek, PeriodBudgetLimit) AS LimitBudget,
    $GetLimitBudget(LimitWeekCur, PeriodBudgetLimitCur) AS LimitBudgetCur,
    LimitAvgCPM,
    LimitAvgCPMCur,

    Cost AS Budget,
    CostCur AS BudgetCur,

    $GetCPU(Cost, Shows) AS AvgCPM,
    $GetCPU(CostCur, Shows) AS AvgCPMCur,
    Shows,
    StartTime AS AutobudgetStartTime,
    LastUpdateTime AS AutobudgetSoftRestartTime,
    PeriodStartTime,
    PeriodFinishTime,
    PeriodStartTimeReal,
    PeriodFinishTimeReal,

    FirstEvent,
    LastEvent,
FROM
    $CPMOrdersExpenses
WHERE
    (
        (
            CurrencyID == 0
            AND $GetExceedRate(LimitAvgCPM, Cost, Shows) > $GetAcceptableExceedRate(TableRow())
        ) OR (
            CurrencyID != 0
            AND $GetExceedRate(LimitAvgCPMCur, CostCur, Shows) > $GetAcceptableExceedRate(TableRow())
        )
    )
;

SELECT
    $ToSeconds($Now) AS RunTime,
    COUNT(*) AS TotalCheckedRows,
FROM
    $CPMOrdersExpenses
;
'''
