QUERY = '''
-- CPV audit --
---------------

-- Helpers:

$ToDatetime = ($seconds) -> {
    RETURN DateTime::MakeTzDatetime(AddTimezone(DateTime::FromSeconds(CAST($seconds AS Uint32)), 'Europe/Moscow'));
};

$RoundToWeekStart = ($timestamp) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfWeek($timestamp));
};

$ToSeconds = ($timestamp) -> {
    RETURN DateTime::ToSeconds($timestamp);
};

$MinOfExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, MIN_OF(COALESCE($a, $b), COALESCE($b, $a)));
};

$MaxOfExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, MAX_OF(COALESCE($a, $b), COALESCE($b, $a)));
};

-- Arguments:

$Now = DateTime::MakeTzDatetime(TzDatetime('2021-09-18T12:46:41,Europe/Moscow'));

$CurWeek = $RoundToWeekStart($Now);

$PrevWeek = $CurWeek - DateTime::IntervalFromDays(7);

-- Audit logic:

$CheckIfHasRestart = ($start_time, $finish_time, $next_start_time) -> {
    RETURN ($next_start_time < $finish_time)
        OR (DateTime::ToDays($ToDatetime($finish_time) - $ToDatetime($start_time)) < 7);
};

$GetAcceptableExceedRate = ($true_views, $start_time, $next_start_time) -> {
    RETURN IF(
        ($true_views < 100)
        OR $CheckIfHasRestart($start_time, $ToSeconds($CurWeek), $next_start_time),
        104.0,
        4.0);
};

$GetExceed = ($expected, $actual) -> {
    RETURN $actual - $expected;
};

$GetExceedRate = ($target, $cost, $units) -> {
    $fraction = IF($target > 0 AND $units > 0, 1.0 * $cost / ($units * $target), 1) - 1;
    RETURN Math::Round($fraction * 100, -2);
};

$GetCPU = ($cost, $units) -> {
    RETURN 1.0 * $cost / $units;
};

$IsCPV = ($row) -> {
    RETURN $row.OptionsPaidActions AND ($row.AvgCPV > 0 OR $row.AvgCPVCur > 0);
};

$CPVStrategyModifyTime = ($row) -> {
    RETURN IF($IsCPV($row), $row.ModifyTime, 0);
};

$CPVOrderVersions = SELECT
    OrderID,
    MAX_BY(StartTime, $CPVStrategyModifyTime(TableRow())) AS StartTime,
    MAX_BY(LastUpdateTime, $CPVStrategyModifyTime(TableRow())) AS LastUpdateTime,
    MAX_BY(LimitWeek, $CPVStrategyModifyTime(TableRow())) AS LimitWeek,
    MAX_BY(LimitWeekCur, $CPVStrategyModifyTime(TableRow())) AS LimitWeekCur,
    MAX_BY(AvgCPV, $CPVStrategyModifyTime(TableRow())) AS LimitAvgCPV,
    MAX_BY(AvgCPVCur, $CPVStrategyModifyTime(TableRow())) AS LimitAvgCPVCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE
    $ToDatetime(StartTime) < $PrevWeek
    AND $ToDatetime(LastUpdateTime) < $CurWeek
    AND OptionsPaidActions
GROUP BY
    OrderID
HAVING  -- Last cpv strategy for the beginning of the week
    MAX_BY(
        $IsCPV(TableRow())
      , ModifyTime
    )
UNION ALL
SELECT
    OrderID,
    StartTime,
    MAX_BY(LastUpdateTime, $CPVStrategyModifyTime(TableRow())) AS LastUpdateTime,
    MAX_BY(LimitWeek, $CPVStrategyModifyTime(TableRow())) AS LimitWeek,
    MAX_BY(LimitWeekCur, $CPVStrategyModifyTime(TableRow())) AS LimitWeekCur,
    MAX_BY(AvgCPV, $CPVStrategyModifyTime(TableRow())) AS LimitAvgCPV,
    MAX_BY(AvgCPVCur, $CPVStrategyModifyTime(TableRow())) AS LimitAvgCPVCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE  -- Was started or restarted during the week
    $ToDatetime(StartTime) >= $PrevWeek
    AND $ToDatetime(StartTime) < $CurWeek
    AND OptionsPaidActions
GROUP BY
    OrderID, StartTime
;

$CPVOrderVersions = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    LimitWeek,
    LimitWeekCur,
    LimitAvgCPV,
    LimitAvgCPVCur,
    COALESCE(MIN(IF(B.StartTime > A.StartTime, B.StartTime, $ToSeconds($CurWeek))), $ToSeconds($CurWeek)) AS NextStartTime,
FROM
    $CPVOrderVersions AS A
LEFT JOIN
    $CPVOrderVersions AS B
ON
    A.OrderID = B.OrderID
GROUP BY
    A.OrderID AS OrderID
  , A.StartTime AS StartTime
  , A.LastUpdateTime AS LastUpdateTime
  , A.LimitWeek AS LimitWeek
  , A.LimitWeekCur AS LimitWeekCur
  , A.LimitAvgCPV AS LimitAvgCPV
  , A.LimitAvgCPVCur AS LimitAvgCPVCur
HAVING
    LimitAvgCPV > 0 OR LimitAvgCPVCur > 0
;

$Log = SELECT
    orderid,
    autobudgetstarttime,
    actioncost,
    actioncostcur,
    eventtime,
    sign,
FROM
    RANGE(
        `//logs/bs-action-checked-log/1d`
      , DateTime::Format('%Y-%m-%d')($PrevWeek)
      , DateTime::Format('%Y-%m-%d')($CurWeek - DateTime::IntervalFromDays(1))
    )
WHERE
    fraudbits = 0
    AND String::Contains(autobudgetoptions, "avg-cpv")
    AND NOT String::Contains(autobudgetoptions, "limit-avg-cpv")
;

$CPVOrderVersions = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    LimitWeek,
    LimitWeekCur,
    LimitAvgCPV,
    LimitAvgCPVCur,
    NextStartTime,
    SUM(actioncost * sign) AS Cost,
    SUM(actioncostcur * sign) AS CostCur,
    SUM(CAST(1 AS Int64)) AS TrueViews,
    MIN(CAST(L.eventtime AS Int64)) AS FirstEvent,
    MAX(CAST(L.eventtime AS Int64)) AS LastEvent,
FROM
    $CPVOrderVersions AS O
LEFT JOIN
    $Log AS L
ON
    (
        CAST(L.orderid AS Int64) = O.OrderID
        AND CAST(L.autobudgetstarttime AS Int64) = O.StartTime
    )
WHERE
    L.orderid IS NULL OR (
        $ToDatetime(eventtime) >= $ToDatetime(MAX_OF($ToSeconds($PrevWeek), O.StartTime))
        AND $ToDatetime(eventtime) < $CurWeek
    )
GROUP BY
    O.OrderID AS OrderID
  , O.StartTime AS StartTime
  , O.LastUpdateTime AS LastUpdateTime
  , O.LimitWeek AS LimitWeek
  , O.LimitWeekCur AS LimitWeekCur
  , O.LimitAvgCPV AS LimitAvgCPV
  , O.LimitAvgCPVCur AS LimitAvgCPVCur
  , O.NextStartTime AS NextStartTime
;


SELECT
    $ToSeconds($Now) AS RunTime,
    O.OrderID AS OrderID,
    $GetExceed(O.LimitAvgCPV, $GetCPU(O.Cost, O.TrueViews)) AS Exceed,
    $GetExceed(O.LimitAvgCPVCur, $GetCPU(O.CostCur, O.TrueViews)) AS ExceedCur,
    $GetExceedRate(O.LimitAvgCPV, O.Cost, O.TrueViews) AS ExceedRate,
    $GetExceedRate(O.LimitAvgCPVCur, O.CostCur, O.TrueViews) AS ExceedCurRate,
    'week' AS LimitType,
    LimitWeek AS LimitBudget,
    LimitWeekCur AS LimitBudgetCur,
    O.LimitAvgCPV AS LimitAvgCPV,
    O.LimitAvgCPVCur AS LimitAvgCPVCur,
    O.Cost AS Budget,
    O.CostCur AS BudgetCur,
    $GetCPU(O.Cost, O.TrueViews) AS AvgCPV,
    $GetCPU(O.CostCur, O.TrueViews) AS AvgCPVCur,
    O.TrueViews AS TrueViews,
    O.StartTime AS AutobudgetStartTime,
    MAX_OF(O.StartTime, $ToSeconds($PrevWeek)) AS PeriodStartTime,
    O.NextStartTime AS PeriodFinishTime,
    O.FirstEvent AS FirstEvent,
    O.LastEvent AS LastEvent,
FROM
    $CPVOrderVersions AS O
INNER JOIN
    `//home/yabs/dict/CaesarOrderInfo` AS I
ON
    O.OrderID = I.OrderID
WHERE
    (LimitAvgCPV > 0 OR LimitAvgCPVCur > 0)
    AND (
        (NOT (I.CurrencyID != 0) AND $GetExceedRate(O.LimitAvgCPV, O.Cost, O.TrueViews) > $GetAcceptableExceedRate(O.TrueViews, O.StartTime, O.NextStartTime))
        OR ((I.CurrencyID != 0) AND $GetExceedRate(O.LimitAvgCPVCur, O.CostCur, O.TrueViews) > $GetAcceptableExceedRate(O.TrueViews, O.StartTime, O.NextStartTime))
    )
;

SELECT
    $ToSeconds($Now) AS RunTime,
    COUNT(*) AS TotalCheckedRows,
FROM
    $CPVOrderVersions
;
'''
