QUERY = '''
-- Daily budget audit --
------------------------
pragma yt.ForceInferSchema;

-- Helpers:

$ToDatetime = ($seconds) -> {
    RETURN DateTime::MakeTzDatetime(AddTimezone(DateTime::FromSeconds(CAST($seconds AS Uint32)), 'Europe/Moscow'));
};

$ToSeconds = ($timestamp) -> {
    RETURN DateTime::ToSeconds($timestamp);
};

$AddExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, COALESCE($a, 0) + COALESCE($b, 0));
};

$MinOfExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, MIN_OF(COALESCE($a, $b), COALESCE($b, $a)));
};

$MaxOfExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, MAX_OF(COALESCE($a, $b), COALESCE($b, $a)));
};

-- Arguments:

$Now = DateTime::MakeTzDatetime(TzDatetime('{{ Now }},Europe/Moscow'));

$Today = DateTime::MakeTzDatetime(DateTime::StartOfDay($Now));

$Yesterday = $Today - DateTime::IntervalFromDays(1);

-- Audit logic:

$MinExceedRate = 30.0;

$GetExceed = ($target, $real) -> {
    RETURN $real - $target;
};

$GetExceedRate = ($target, $real) -> {
    $fraction = IF($target > 0, 1.0 * $real / $target, 1) - 1;
    RETURN Math::Round($fraction * 100, -2);
};

$CheckIfLimitExceeded = ($abOrderLimit, $cost) -> {
    RETURN $GetExceedRate($abOrderLimit, $cost) > $MinExceedRate;
};

$IsFixPrice = ($row) -> {
    RETURN ($row.LimitDayMoney > 0 OR $row.LimitDayMoneyCur > 0) AND ($row.PeriodBudgetLimit > 0 OR $row.PeriodBudgetLimitCur > 0);
};

$FixPriceStrategyModifyTime = ($row) -> {
    RETURN IF($IsFixPrice($row), $row.ModifyTime, 0);
};

$FixPriceOrders = SELECT
    OrderID,
    MAX_BY(StartTime, $FixPriceStrategyModifyTime(TableRow())) AS StartTime,
    MAX_BY(LastUpdateTime, $FixPriceStrategyModifyTime(TableRow())) AS LastUpdateTime,
    MAX_BY(PeriodBudgetFinish, $FixPriceStrategyModifyTime(TableRow())) AS PeriodBudgetFinish,
    MAX_BY(PeriodBudgetLimit, $FixPriceStrategyModifyTime(TableRow())) AS PeriodBudgetLimit,
    MAX_BY(PeriodBudgetLimitCur, $FixPriceStrategyModifyTime(TableRow())) AS PeriodBudgetLimitCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE
    $ToDatetime(StartTime) < $Yesterday
GROUP BY
    OrderID
HAVING  -- Was fix-price order by the beginning of the day
    MAX_BY(
        $IsFixPrice(TableRow())
      , ModifyTime
    )
UNION ALL
SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    PeriodBudgetFinish,
    PeriodBudgetLimit,
    PeriodBudgetLimitCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE  -- Was fix-price order during the day
    $ToDatetime(StartTime) >= $Yesterday
    AND $ToDatetime(StartTime) < $Today
    AND $IsFixPrice(TableRow())
;


$CountDays = ($seconds) -> {
    $interval = DateTime::IntervalFromSeconds(CAST($seconds AS Int32));
    $days = DateTime::ToDays($interval);
    $days = IF(DateTime::ToSeconds($interval - DateTime::IntervalFromDays($days)) > 0, $days + 1, $days);
    RETURN IF($days = 0, 1, $days);
};

$DynamicFixPriceDailyLimit = ($budget, $spent, $periodFinish) -> {
    RETURN CAST(CAST($budget - COALESCE($spent, 0) AS Double) / $CountDays($periodFinish - $ToSeconds($Yesterday)) AS Int64);
};

$FixPriceOrders = SELECT
    OrderID,
    StartTime,
    LastUpdateTime,
    PeriodBudgetFinish,
    PeriodBudgetLimit,
    PeriodBudgetLimitCur,
    $DynamicFixPriceDailyLimit(PeriodBudgetLimit, SUM(S.Cost), PeriodBudgetFinish) AS LimitDayMoney,
    $DynamicFixPriceDailyLimit(PeriodBudgetLimitCur, SUM(S.CostCur), PeriodBudgetFinish) AS LimitDayMoneyCur,
FROM
    $FixPriceOrders AS O
LEFT JOIN
    `//home/yabs/dict/AutoBudgetStatExternal` AS S
ON
    S.OrderID = O.OrderID
    AND S.StartTime = O.StartTime
WHERE
    S.StatName = 'day'
    AND S.IntervalStartTime >= O.StartTime
    AND S.IntervalStartTime < $ToSeconds($Yesterday)
    AND $ToDatetime(PeriodBudgetFinish) > $Today
GROUP BY
    O.OrderID AS OrderID
  , O.StartTime AS StartTime
  , O.LastUpdateTime AS LastUpdateTime
  , O.PeriodBudgetFinish AS PeriodBudgetFinish
  , O.PeriodBudgetLimit AS PeriodBudgetLimit
  , O.PeriodBudgetLimitCur AS PeriodBudgetLimitCur
;

$IsDaily = ($row) -> {
    RETURN ($row.LimitDayMoney > 0 OR $row.LimitDayMoneyCur > 0) AND ($row.PeriodBudgetLimit = 0 AND $row.PeriodBudgetLimitCur = 0);
};

$DailyStrategyModifyTime = ($row) -> {
    RETURN IF($IsDaily($row), $row.ModifyTime, 0);
};

$DailyLimitDuringTheDay = SELECT
    OrderID,
    StartTime,
    MAX(MAX_OF(LimitDayMoney, MaxLimitDayMoney)) AS LimitDayMoney,
    MAX(MAX_OF(LimitDayMoneyCur, MaxLimitDayMoneyCur)) AS LimitDayMoneyCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE  -- Change daily limit during the day
    $ToDatetime(LastUpdateTime) >= $Yesterday
    AND $ToDatetime(LastUpdateTime) < $Today
    AND $IsDaily(TableRow())
GROUP BY
    OrderID, StartTime
;

$DailyLimitForTheBeginningOfDay = SELECT
    OrderID,
    MAX_BY(StartTime, $DailyStrategyModifyTime(TableRow())) AS StartTime,
    MAX_BY(LastUpdateTime, $DailyStrategyModifyTime(TableRow())) AS LastUpdateTime,
    MAX_BY(LimitDayMoney, $DailyStrategyModifyTime(TableRow())) AS LimitDayMoney,
    MAX_BY(LimitDayMoneyCur, $DailyStrategyModifyTime(TableRow())) AS LimitDayMoneyCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE
    $ToDatetime(StartTime) < $Yesterday
    AND $ToDatetime(LastUpdateTime) < $Yesterday
GROUP BY
    OrderID
HAVING  -- Was dailly order by the beginning of the day
    MAX_BY($IsDaily(TableRow()), ModifyTime)
;

$DailyOrders = SELECT
    COALESCE(B.OrderID, D.OrderID) AS OrderID,
    COALESCE(B.StartTime, D.StartTime) AS StartTime,
    $MaxOfExisted(B.LimitDayMoney, D.LimitDayMoney) AS LimitDayMoney,
    $MaxOfExisted(B.LimitDayMoneyCur, D.LimitDayMoneyCur) AS LimitDayMoneyCur,
FROM
    $DailyLimitForTheBeginningOfDay AS B
FULL JOIN
    $DailyLimitDuringTheDay AS D
ON
    D.OrderID = B.OrderID
    AND D.StartTime = B.StartTime
;


$Orders = SELECT
    'daily-fixprice' AS LimitType,
    OrderID,
    StartTime,
    LimitDayMoney,
    LimitDayMoneyCur,
FROM
    $FixPriceOrders
UNION ALL
SELECT
    'daily' AS LimitType,
    OrderID,
    StartTime,
    LimitDayMoney,
    LimitDayMoneyCur,
FROM
    $DailyOrders
;

$Orders = SELECT
    O.LimitType AS LimitType,
    O.OrderID AS OrderID,
    O.StartTime AS StartTime,
    O.LimitDayMoney AS LimitDayMoney,
    O.LimitDayMoneyCur AS LimitDayMoneyCur,
    CurrencyID != 0 AS IsCurrency,
FROM
    $Orders AS O
INNER JOIN
    `//home/yabs/dict/CaesarOrderInfo` AS I
ON
    O.OrderID = I.OrderID
;

$Logs = '//logs/bs-chevent-log/1d/' || DateTime::Format('%Y-%m-%d')($Yesterday);
$UndoLogs = '//logs/bs-undochevent-log/1d/' || DateTime::Format('%Y-%m-%d')($Yesterday);

$ChildLogs = SELECT
    orderid,
    autobudgetstarttime,
    eventcost,
    costcur,
    eventtime,
FROM
    $Logs
WHERE
    $ToDatetime(eventtime) >= $Yesterday
    AND $ToDatetime(eventtime) < $Today
    AND fraudbits = '0'
    AND autobudgetoptions LIKE '%optimize-manual%'
    AND autobudgetoptions LIKE '%limit-daily-budget%'
;

$GroupLogs = SELECT
    grouporderid,
    autobudgetgroupstarttime,
    eventcost,
    costcur,
    eventtime,
FROM
    $Logs
WHERE
    $ToDatetime(eventtime) >= $Yesterday
    AND $ToDatetime(eventtime) < $Today
    AND fraudbits = '0'
    AND autobudgetoptions LIKE '%limit-group-daily-budget%'
    AND autobudgetoptions NOT LIKE '%paid-actions%'
;

$ChildUndoLogs = SELECT
    orderid,
    autobudgetstarttime,
    oldeventcost,
    neweventcost,
    oldcostcur,
    newcostcur,
    eventtime,
FROM
    $UndoLogs
WHERE
    $ToDatetime(eventtime) >= $Yesterday
    AND $ToDatetime(eventtime) < $Today
    AND autobudgetoptions LIKE '%optimize-manual%'
    AND autobudgetoptions LIKE '%limit-daily-budget%'
;

$GroupUndoLogs = SELECT
    grouporderid,
    autobudgetgroupstarttime,
    oldeventcost,
    neweventcost,
    oldcostcur,
    newcostcur,
    eventtime,
FROM
    $UndoLogs
WHERE
    $ToDatetime(eventtime) >= $Yesterday
    AND $ToDatetime(eventtime) < $Today
    AND autobudgetoptions LIKE '%limit-group-daily-budget%'
    AND autobudgetoptions NOT LIKE '%paid-actions%'
;

$YesterdayAuditDailyLimits = SELECT
    LimitType,
    OrderID,
    StartTime,
    LimitDayMoney,
    LimitDayMoneyCur,
    IsCurrency,
    SUM(CAST(L.eventcost AS Int64)) AS Cost,
    SUM(CAST(L.costcur AS Int64)) AS CostCur,
    MIN(CAST(L.eventtime AS Int64)) AS FirstEvent,
    MAX(CAST(L.eventtime AS Int64)) AS LastEvent,
FROM
    $Orders AS O
LEFT JOIN
    $ChildLogs AS L
ON
    (
        CAST(L.orderid AS Int64) = O.OrderID
        AND CAST(L.autobudgetstarttime AS Int64) = O.StartTime
    )
GROUP BY
    O.LimitType AS LimitType,
    O.OrderID AS OrderID,
    O.StartTime AS StartTime,
    O.LimitDayMoney AS LimitDayMoney,
    O.LimitDayMoneyCur AS LimitDayMoneyCur,
    O.IsCurrency AS IsCurrency
;

$YesterdayAuditDailyLimits = SELECT
    LimitType,
    OrderID,
    StartTime,
    LimitDayMoney,
    LimitDayMoneyCur,
    IsCurrency,
    Cost - COALESCE(SUM(CAST(L.oldeventcost AS Int64) - CAST(L.neweventcost AS Int64)), 0l) AS Cost,
    CostCur - COALESCE(SUM(CAST(L.oldcostcur AS Int64) - CAST(L.newcostcur AS Int64)), 0l) AS CostCur,
    $MinOfExisted(FirstEvent, MIN(CAST(L.eventtime AS Int64))) AS FirstEvent,
    $MaxOfExisted(LastEvent, MAX(CAST(L.eventtime AS Int64))) AS LastEvent,
FROM
    $YesterdayAuditDailyLimits AS O
LEFT JOIN
    $ChildUndoLogs AS L
ON
    (
        CAST(L.orderid AS Int64) = O.OrderID
        AND CAST(L.autobudgetstarttime AS Int64) = O.StartTime
    )
GROUP BY
    O.LimitType AS LimitType,
    O.OrderID AS OrderID,
    O.StartTime AS StartTime,
    O.LimitDayMoney AS LimitDayMoney,
    O.LimitDayMoneyCur AS LimitDayMoneyCur,
    O.IsCurrency AS IsCurrency,
    O.Cost AS Cost,
    O.CostCur AS CostCur,
    O.FirstEvent AS FirstEvent,
    O.LastEvent AS LastEvent
;

$YesterdayAuditDailyLimits = SELECT
    LimitType,
    OrderID,
    StartTime,
    LimitDayMoney,
    LimitDayMoneyCur,
    IsCurrency,
    $AddExisted(Cost, SUM(CAST(L.eventcost AS Int64))) AS Cost,
    $AddExisted(CostCur, SUM(CAST(L.costcur AS Int64))) AS CostCur,
    $MinOfExisted(FirstEvent, MIN(CAST(L.eventtime AS Int64))) AS FirstEvent,
    $MaxOfExisted(LastEvent, MAX(CAST(L.eventtime AS Int64))) AS LastEvent,
FROM
    $YesterdayAuditDailyLimits AS O
LEFT JOIN
    $GroupLogs AS L
ON
    (
        CAST(L.grouporderid AS Int64) = O.OrderID
        AND CAST(L.autobudgetgroupstarttime AS Int64) = O.StartTime
    )
GROUP BY
    O.LimitType AS LimitType,
    O.OrderID AS OrderID,
    O.StartTime AS StartTime,
    O.LimitDayMoney AS LimitDayMoney,
    O.LimitDayMoneyCur AS LimitDayMoneyCur,
    O.IsCurrency AS IsCurrency,
    O.Cost AS Cost,
    O.CostCur AS CostCur,
    O.FirstEvent AS FirstEvent,
    O.LastEvent AS LastEvent
;

$YesterdayAuditDailyLimits = SELECT
    LimitType,
    OrderID,
    StartTime,
    LimitDayMoney,
    LimitDayMoneyCur,
    IsCurrency,
    Cost - COALESCE(SUM(CAST(L.oldeventcost AS Int64) - CAST(L.neweventcost AS Int64)), 0l) AS Cost,
    CostCur - COALESCE(SUM(CAST(L.oldcostcur AS Int64) - CAST(L.newcostcur AS Int64)), 0l) AS CostCur,
    $MinOfExisted(FirstEvent, MIN(CAST(L.eventtime AS Int64))) AS FirstEvent,
    $MaxOfExisted(LastEvent, MAX(CAST(L.eventtime AS Int64))) AS LastEvent,
FROM
    $YesterdayAuditDailyLimits AS O
LEFT JOIN
    $GroupUndoLogs AS L
ON
    (
        CAST(L.grouporderid AS Int64) = O.OrderID
        AND CAST(L.autobudgetgroupstarttime AS Int64) = O.StartTime  -- TODO: restore after
    )
GROUP BY
    O.LimitType AS LimitType,
    O.OrderID AS OrderID,
    O.StartTime AS StartTime,
    O.LimitDayMoney AS LimitDayMoney,
    O.LimitDayMoneyCur AS LimitDayMoneyCur,
    O.IsCurrency AS IsCurrency,
    O.Cost AS Cost,
    O.CostCur AS CostCur,
    O.FirstEvent AS FirstEvent,
    O.LastEvent AS LastEvent
;


SELECT
    $ToSeconds($Now) AS RunTime,
    OrderID,
    LimitType,
    LimitDayMoney AS LimitBudget,
    LimitDayMoneyCur AS LimitBudgetCur,
    Cost AS Budget,
    CostCur AS BudgetCur,
    $GetExceed(LimitDayMoney, Cost) as Exceed,
    $GetExceed(LimitDayMoneyCur, CostCur) as ExceedCur,
    $GetExceedRate(LimitDayMoney, Cost) as ExceedRate,
    $GetExceedRate(LimitDayMoneyCur, CostCur) as ExceedCurRate,
    StartTime AS AutobudgetStartTime,
    MAX_OF($ToSeconds($Yesterday), StartTime) AS PeriodStartTime,
    $ToSeconds($Today) AS PeriodFinishTime,
    FirstEvent,
    LastEvent,
FROM
    $YesterdayAuditDailyLimits
WHERE
    (
        (NOT IsCurrency AND $CheckIfLimitExceeded(LimitDayMoney, Cost))
        OR (IsCurrency AND $CheckIfLimitExceeded(LimitDayMoneyCur, CostCur))
    )
;

SELECT
    $ToSeconds($Now) as RunTime,
    COUNT(*) as TotalCheckedRows,
FROM
    $YesterdayAuditDailyLimits
;
'''
