QUERY = '''
-- Daily-week budget audit --
-----------------------------

-- Helpers:

$ToTimestamp = ($seconds) -> {
    RETURN DateTime::MakeTzTimestamp(AddTimezone(DateTime::FromSeconds(CAST($seconds AS Uint32)), 'Europe/Moscow'));
};

$ToSeconds = ($timestamp) -> {
    RETURN DateTime::ToSeconds($timestamp);
};

$RoundToDay = ($timestamp) -> {
    RETURN DateTime::MakeTzTimestamp(DateTime::StartOfDay($timestamp));
};

$RoundToWeekStart = ($timestamp) -> {
    RETURN DateTime::MakeTzTimestamp(DateTime::StartOfWeek($timestamp));
};

$MaxOfDates = ($a, $b) -> {
    RETURN IF($b > $a, $b, $a);
};

-- Arguments:

$Now = CurrentUtcTimestamp();
$Now = DateTime::MakeTzTimestamp(AddTimezone($Now, 'Europe/Moscow'));

$CurWeek = $RoundToWeekStart($Now);

$PrevWeek = $CurWeek - DateTime::IntervalFromDays(7);

-- Audit logic:

$MinExceedRate = 0.0;

$GetPeriodStart = ($autobudgetStartTime) -> {
    RETURN $MaxOfDates($RoundToDay($autobudgetStartTime) + DateTime::IntervalFromDays(1), $PrevWeek);
};

$GetPeriodEnd = ($autobudgetStartTime) -> {
    RETURN $RoundToWeekStart($GetPeriodStart($autobudgetStartTime)) + DateTime::IntervalFromDays(7);
};

$GetActiveDays = ($autobudgetStartTime) -> {
    RETURN DateTime::ToDays($GetPeriodEnd($autobudgetStartTime) - $GetPeriodStart($autobudgetStartTime));
};

$GetExceed = ($target, $real) -> {
    RETURN $real - $target;
};

$GetExceedRate = ($target, $real) -> {
    $fraction = IF($target > 0, 1.0 * $real / $target, 1) - 1;
    RETURN Math::Round($fraction * 100, -2);
};

$PrevWeekOrderVersion = (
    SELECT
        OrderID,
        $ToSeconds(WeekStart) as WeekStart,
        MAX(AutobudgetStartTime) as LatestAutobudgetStartTime
    FROM
        `//home/yabs/stat/AutobudgetAuditDailyLimit`
    GROUP BY
        OrderID,
        $RoundToWeekStart($ToTimestamp(Day)) as WeekStart
    HAVING
        WeekStart = $PrevWeek
);

$OrderExpenses = (
    SELECT
        OrderID,
        AutobudgetStartTime,
        IsCurrency,
        $ToSeconds(WeekStart) as WeekStart,

        DayLimit * $GetActiveDays($ToTimestamp(AutobudgetStartTime)) as LimitBudget,
        DayLimitCur * $GetActiveDays($ToTimestamp(AutobudgetStartTime)) as LimitBudgetCur,

        SUM(Cost) as Budget,
        SUM(CostCur) as BudgetCur,

        $GetExceed(DayLimit * $GetActiveDays($ToTimestamp(AutobudgetStartTime)), SUM(Cost)) as Exceed,
        $GetExceed(DayLimitCur * $GetActiveDays($ToTimestamp(AutobudgetStartTime)), SUM(CostCur)) as ExceedCur,
        $GetExceedRate(DayLimit * $GetActiveDays($ToTimestamp(AutobudgetStartTime)), SUM(Cost)) as ExceedRate,
        $GetExceedRate(DayLimitCur * $GetActiveDays($ToTimestamp(AutobudgetStartTime)), SUM(CostCur)) as ExceedCurRate,

        $GetActiveDays($ToTimestamp(AutobudgetStartTime)) as ActiveDays,
        $ToSeconds($GetPeriodStart($ToTimestamp(AutobudgetStartTime))) as PeriodStartTime,
        $ToSeconds($GetPeriodEnd($ToTimestamp(AutobudgetStartTime))) as PeriodFinishTime,

        MIN(FirstEvent) as FirstEvent,
        MAX(LastEvent) as LastEvent
    FROM
        `//home/yabs/stat/AutobudgetAuditDailyLimit`
    WHERE
        Day > AutobudgetStartTime
    GROUP BY
        OrderID,
        DayLimit,
        DayLimitCur,
        IsCurrency,
        AutobudgetStartTime,
        $RoundToWeekStart($ToTimestamp(Day)) as WeekStart
    HAVING
        (DayLimit > 0 OR DayLimitCur > 0)
        AND (
            (NOT IsCurrency AND $GetExceedRate(DayLimit * $GetActiveDays($ToTimestamp(AutobudgetStartTime)), SUM(Cost)) > $MinExceedRate)
            OR (IsCurrency AND $GetExceedRate(DayLimitCur * $GetActiveDays($ToTimestamp(AutobudgetStartTime)), SUM(CostCur)) > $MinExceedRate)
        )
        AND WeekStart = $PrevWeek
);

SELECT
    $ToSeconds($Now) as RunTime,
    O.OrderID as OrderID,
    "daily-week" as LimitType,

    O.LimitBudget as LimitBudget,
    O.LimitBudgetCur as LimitBudgetCur,

    O.AutobudgetStartTime as AutobudgetStartTime,
    O.PeriodStartTime as PeriodStartTime,
    O.PeriodFinishTime as PeriodFinishTime,

    O.Budget as Budget,
    O.BudgetCur as BudgetCur,

    O.Exceed as Exceed,
    O.ExceedCur as ExceedCur,

    O.ExceedRate as ExceedRate,
    O.ExceedCurRate as ExceedCurRate,

    O.FirstEvent as FirstEvent,
    O.LastEvent as LastEvent
FROM
    $PrevWeekOrderVersion as L
JOIN
    $OrderExpenses as O
ON
    L.OrderID = O.OrderID
    AND L.WeekStart = O.WeekStart
    AND L.LatestAutobudgetStartTime = O.AutobudgetStartTime
;

SELECT
    $ToSeconds($Now) as RunTime,
    COUNT(*) as TotalCheckedRows,
FROM
    $PrevWeekOrderVersion
;
'''
