QUERY = '''
-- Daily-week budget audit --
-----------------------------
pragma yt.ForceInferSchema;

-- Helpers:

$ToDatetime = ($seconds) -> {
    RETURN DateTime::MakeTzDatetime(AddTimezone(DateTime::FromSeconds(CAST($seconds AS Uint32)), 'Europe/Moscow'));
};

$ToSeconds = ($datetime) -> {
    RETURN DateTime::ToSeconds($datetime);
};

$RoundToDay = ($datetime) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfDay($datetime));
};

$RoundToWeekStart = ($datetime) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfWeek($datetime));
};

$MaxOfDates = ($a, $b) -> {
    RETURN IF($b > $a, $b, $a);
};

$AddExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, COALESCE($a, 0) + COALESCE($b, 0));
};

$MinOfExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, MIN_OF(COALESCE($a, $b), COALESCE($b, $a)));
};

$MaxOfExisted = ($a, $b) -> {
    RETURN IF($a IS NULL AND $b IS NULL, NULL, MAX_OF(COALESCE($a, $b), COALESCE($b, $a)));
};

-- Arguments:

$Now = DateTime::MakeTzDatetime(TzDatetime('{{ Now }},Europe/Moscow'));

$CurWeek = $RoundToWeekStart($Now);

$PrevWeek = $CurWeek - DateTime::IntervalFromDays(7);

-- Audit logic:

$MinExceedRate = 0.0;

$GetPeriodStart = ($softRestartTime) -> {
    RETURN $MaxOfDates($RoundToDay($softRestartTime) + DateTime::IntervalFromDays(1), $PrevWeek);
};

$GetPeriodEnd = ($softRestartTime) -> {
    RETURN $RoundToWeekStart($GetPeriodStart($softRestartTime)) + DateTime::IntervalFromDays(7);
};

$GetActiveDays = ($softRestartTime) -> {
    RETURN DateTime::ToDays($GetPeriodEnd($softRestartTime) - $GetPeriodStart($softRestartTime));
};

$GetExceed = ($target, $real) -> {
    RETURN $real - $target;
};

$GetExceedRate = ($target, $real) -> {
    $fraction = IF($target > 0, 1.0 * $real / $target, 1) - 1;
    RETURN Math::Round($fraction * 100, -2);
};

$IsDaily = ($row) -> {
    RETURN ($row.LimitDayMoney > 0 OR $row.LimitDayMoneyCur > 0) AND ($row.PeriodBudgetLimit = 0 AND $row.PeriodBudgetLimitCur = 0);
};

$DailyStrategyModifyTime = ($row) -> {
    RETURN IF($IsDaily($row), $row.ModifyTime, 0);
};


$DailyStrategies = SELECT
    StrategyID,
    MAX_BY(OrderID, $DailyStrategyModifyTime(TableRow())) as Some_OrderID,
    MAX_BY(StartTime, $DailyStrategyModifyTime(TableRow())) AS StartTime,
    MAX_BY(LastUpdateTime, $DailyStrategyModifyTime(TableRow())) AS LastUpdateTime,
    MAX_BY(LimitDayMoney, $DailyStrategyModifyTime(TableRow())) AS LimitDayMoney,
    MAX_BY(LimitDayMoneyCur, $DailyStrategyModifyTime(TableRow())) AS LimitDayMoneyCur,
    MAX_BY(LimitDayMoney, $DailyStrategyModifyTime(TableRow()))
    * $GetActiveDays($ToDatetime(MAX_BY(LastUpdateTime, $DailyStrategyModifyTime(TableRow())))) AS PeriodBudget,
    MAX_BY(LimitDayMoneyCur, $DailyStrategyModifyTime(TableRow()))
    * $GetActiveDays($ToDatetime(MAX_BY(LastUpdateTime, $DailyStrategyModifyTime(TableRow())))) AS PeriodBudgetCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE
    $ToDatetime(LastUpdateTime) < $CurWeek
GROUP BY
    StrategyID
HAVING  -- Last daily order limit for this week
    MAX_BY(
        $IsDaily(TableRow())
      , ModifyTime
    )
;

$Logs = SELECT
    autobudgetstrategyid,
    autobudgetstarttime,
    autobudgetsoftrestarttime,
    eventcost,
    costcur,
    eventtime,
FROM
    RANGE(
        `//logs/bs-chevent-log/1d`
      , DateTime::Format('%Y-%m-%d')($PrevWeek)
      , DateTime::Format('%Y-%m-%d')($CurWeek - DateTime::IntervalFromDays(1))
    )
WHERE
    fraudbits = '0'
    AND autobudgetoptions LIKE '%optimize-manual%'
    AND autobudgetoptions LIKE '%limit-daily-budget%'
;


$UndoLogs = SELECT
    autobudgetstrategyid,
    autobudgetstarttime,
    autobudgetsoftrestarttime,
    oldeventcost,
    neweventcost,
    oldcostcur,
    newcostcur,
    eventtime,
FROM
    RANGE(
        `//logs/bs-undochevent-log/1d`
      , DateTime::Format('%Y-%m-%d')($PrevWeek)
      , DateTime::Format('%Y-%m-%d')($CurWeek - DateTime::IntervalFromDays(1))
    )
WHERE
    autobudgetoptions LIKE '%optimize-manual%'
    AND autobudgetoptions LIKE '%limit-daily-budget%'
    AND $ToDatetime(eventtime) >= $PrevWeek
    AND $ToDatetime(eventtime) < $CurWeek
;


$DailyStrategyExpenses = SELECT
    StrategyID
  , Some_OrderID
  , StartTime
  , LastUpdateTime
  , LimitDayMoney
  , LimitDayMoneyCur
  , PeriodBudget
  , PeriodBudgetCur
  , SUM(CAST(L.eventcost AS Int64)) AS Cost
  , SUM(CAST(L.costcur AS Int64)) AS CostCur
  , MIN(CAST(L.eventtime AS Int64)) AS FirstEvent
  , MAX(CAST(L.eventtime AS Int64)) AS LastEvent
FROM
    $DailyStrategies AS S
LEFT JOIN
    $Logs AS L
ON
    (
        CAST(L.autobudgetstrategyid AS Int64) = S.StrategyID
        AND CAST(L.autobudgetstarttime AS Int64) = S.StartTime
        AND CAST(L.autobudgetsoftrestarttime AS Int64) = S.LastUpdateTime
    )
WHERE
    L.autobudgetstrategyid IS NULL OR (
        $ToDatetime(L.eventtime) >= $GetPeriodStart($ToDatetime(S.LastUpdateTime))
    )
GROUP BY
    StrategyID AS StrategyID
  , Some_OrderID AS Some_OrderID
  , StartTime AS StartTime
  , LastUpdateTime AS LastUpdateTime
  , LimitDayMoney AS LimitDayMoney
  , LimitDayMoneyCur AS LimitDayMoneyCur
  , PeriodBudget AS PeriodBudget
  , PeriodBudgetCur AS PeriodBudgetCur
;

$DailyStrategyExpenses = SELECT
    StrategyID
  , Some_OrderID
  , StartTime
  , LastUpdateTime
  , LimitDayMoney
  , LimitDayMoneyCur
  , PeriodBudget
  , PeriodBudgetCur
  , Cost - COALESCE(SUM(CAST(L.oldeventcost AS Int64) - CAST(L.neweventcost AS Int64)), 0l) AS Cost
  , CostCur - COALESCE(SUM(CAST(L.oldcostcur AS Int64) - CAST(L.newcostcur AS Int64)), 0l) AS CostCur
  , $MinOfExisted(FirstEvent, MIN(CAST(L.eventtime AS Int64))) AS FirstEvent
  , $MaxOfExisted(LastEvent, Max(CAST(L.eventtime AS Int64))) AS LastEvent
FROM
    $DailyStrategyExpenses AS S
LEFT JOIN
    $UndoLogs AS L
ON
    (
        CAST(L.autobudgetstrategyid AS Int64) = S.StrategyID
        AND CAST(L.autobudgetstarttime AS Int64) = S.StartTime
        AND CAST(L.autobudgetsoftrestarttime AS Int64) = S.LastUpdateTime
    )
WHERE
    L.autobudgetstrategyid IS NULL OR (
        $ToDatetime(L.eventtime) >= $GetPeriodStart($ToDatetime(S.LastUpdateTime))
    )
GROUP BY
    StrategyID AS StrategyID
  , Some_OrderID AS Some_OrderID
  , StartTime AS StartTime
  , LastUpdateTime AS LastUpdateTime
  , LimitDayMoney AS LimitDayMoney
  , LimitDayMoneyCur AS LimitDayMoneyCur
  , PeriodBudget AS PeriodBudget
  , PeriodBudgetCur AS PeriodBudgetCur
  , Cost AS Cost
  , CostCur AS CostCur
  , FirstEvent AS FirstEvent
  , LastEvent AS LastEvent
;


SELECT
    $ToSeconds($Now) AS RunTime,
    D.StrategyID AS StrategyID,
    'daily-week' AS LimitType,

    D.PeriodBudget AS LimitBudget,
    D.PeriodBudgetCur AS LimitBudgetCur,

    D.LastUpdateTime AS AutobudgetStartTime,
    $ToSeconds($GetPeriodStart($ToDatetime(D.LastUpdateTime))) AS PeriodStartTime,
    $ToSeconds($GetPeriodEnd($ToDatetime(D.LastUpdateTime))) AS PeriodFinishTime,

    D.Cost AS Budget,
    D.CostCur AS BudgetCur,
    O.CurrencyID as CurrencyID,

    $GetExceed(D.PeriodBudget, D.Cost) AS Exceed,
    $GetExceed(D.PeriodBudgetCur, D.CostCur) AS ExceedCur,

    $GetExceedRate(D.PeriodBudget, D.Cost) AS ExceedRate,
    $GetExceedRate(D.PeriodBudgetCur, D.CostCur) AS ExceedCurRate,

    D.FirstEvent AS FirstEvent,
    D.LastEvent AS LastEvent,
FROM
    $DailyStrategyExpenses AS D
INNER JOIN
    `//home/yabs/dict/CaesarOrderInfo` AS O
ON
    O.OrderID = D.Some_OrderID
WHERE
    (NOT (O.CurrencyID != 0) AND $GetExceedRate(D.PeriodBudget, D.Cost) > $MinExceedRate)
    OR (O.CurrencyID != 0 AND $GetExceedRate(D.PeriodBudgetCur, D.CostCur) > $MinExceedRate)
;

SELECT
    $ToSeconds($Now) AS RunTime,
    COUNT(*) AS TotalCheckedRows,
FROM
    $DailyStrategyExpenses
;
'''
