PRE_QUERY = '''
-- Period budget audit MinStartTime --
--------------------------------------

-- Helpers:

$ToDatetime = ($seconds) -> {
    RETURN DateTime::MakeTzDatetime(AddTimezone(DateTime::FromSeconds(CAST($seconds AS Uint32)), 'Europe/Moscow'));
};

$RoundToWeekStart = ($datetime) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfWeek($datetime));
};

-- Arguments:

$Now = DateTime::MakeTzDatetime(TzDatetime('{{ Now }},Europe/Moscow'));

$Today = DateTime::MakeTzDatetime(DateTime::StartOfDay($Now));

$Yesterday = $Today - DateTime::IntervalFromDays(1);

$CurWeek = $RoundToWeekStart($Now);

$PrevWeek = $CurWeek - DateTime::IntervalFromDays(7);

-- Audit logic:

$IsPeriod = ($row) -> {
    RETURN ($row.PeriodBudgetLimit > 0 OR $row.PeriodBudgetLimitCur > 0);
};

SELECT
    DateTime::Format('%Y-%m-%dT%H:%M:%S,%Z')(COALESCE($ToDatetime(MIN(StartTime)), $PrevWeek)) AS MinStartTime,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE
    $IsPeriod(TableRow())
    AND $ToDatetime(LastUpdateTime) < $Today
    AND $ToDatetime(PeriodBudgetFinish) < $Today
    AND $ToDatetime(PeriodBudgetFinish) >= $Yesterday
;
'''

QUERY = '''
-- Period budget audit with versioning --
-----------------------------------------

-- Helpers:

$ToDatetime = ($seconds) -> {
    RETURN DateTime::MakeTzDatetime(AddTimezone(DateTime::FromSeconds(CAST($seconds AS Uint32)), 'Europe/Moscow'));
};

$ToSeconds = ($datetime) -> {
    RETURN DateTime::ToSeconds($datetime);
};

$RoundToWeekStart = ($datetime) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfWeek($datetime));
};

$RoundToDayStart = ($datetime) -> {
    RETURN DateTime::MakeTzDatetime(DateTime::StartOfDay($datetime));
};

$script = @@
from yql.typing import List
from yql.typing import Optional
from yql.typing import Uint32


def range_days(fr: Optional[Uint32], to: Optional[Uint32]) -> List[Uint32]:
    result = []
    if fr is None or to is None:
        return result
    while fr <= to:
        result.append(fr)
        fr += 24 * 60 * 60
    return result
@@;

$RangeDays = Python3::range_days($script);

-- Arguments:

$MinStartTime = DateTime::MakeTzDatetime(TzDatetime('{{ MinStartTime }}'));

$MinStartDay = $RoundToDayStart($MinStartTime);

$Now = DateTime::MakeTzDatetime(TzDatetime('{{ Now }},Europe/Moscow'));

$Today = $RoundToDayStart($Now);

$Yesterday = $Today - DateTime::IntervalFromDays(1);

$CurWeek = $RoundToWeekStart($Now);

$PrevWeek = $CurWeek - DateTime::IntervalFromDays(7);

-- Audit logic:

$MinExceedRate = 0.0;

$GetExceed = ($target, $real) -> {
    RETURN $real - $target;
};

$GetExceedRate = ($target, $real) -> {
    $fraction = IF($target > 0, 1.0 * $real / $target, 1) - 1;
    RETURN Math::Round($fraction * 100, -2);
};

$GetLimitType = ($weekLimit, $periodLimit) -> {
    RETURN IF($weekLimit > 0, 'week', IF($periodLimit > 0, 'period', 'unknown'));
};

$GetLimitBudget = ($weekLimit, $periodLimit) -> {
    RETURN MAX_OF($weekLimit, $periodLimit);
};

$IsPeriod = ($row) -> {
    RETURN ($row.PeriodBudgetLimit > 0 OR $row.PeriodBudgetLimitCur > 0);
};

$IsWeekLimited = ($row) -> {
    RETURN ($row.LimitWeek > 0 OR $row.LimitWeekCur > 0);
};

$PeriodAndWeekLimitedStrategies = SELECT
    StrategyID,
    StartTime,
    MAX_BY(OrderID, ModifyTime) AS OrderID,
    MAX_BY(PeriodBudgetFinish, ModifyTime) AS PeriodBudgetFinish,
    MAX_BY(LastUpdateTime, ModifyTime) AS LastUpdateTime,
    MAX_BY(PeriodBudgetLimit, ModifyTime) AS PeriodBudgetLimit,
    MAX_BY(PeriodBudgetLimitCur, ModifyTime) AS PeriodBudgetLimitCur,
    MAX_BY(LimitWeek, ModifyTime) AS LimitWeek,
    MAX_BY(LimitWeekCur, ModifyTime) AS LimitWeekCur,
FROM
    `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
WHERE
    $IsPeriod(TableRow())
GROUP BY
    StrategyID, StartTime
HAVING
    $ToDatetime(MAX_BY(PeriodBudgetFinish, ModifyTime)) <= $Today
    AND $ToDatetime(MAX_BY(PeriodBudgetFinish, ModifyTime)) > $Yesterday
UNION ALL
SELECT
    StrategyID,
    MAX_BY(StartTime, ModifyTime) AS StartTime,
    MAX_BY(OrderID, ModifyTime) AS OrderID,
    MAX_BY(PeriodBudgetFinish, ModifyTime) AS PeriodBudgetFinish,
    MAX_BY(LastUpdateTime, ModifyTime) AS LastUpdateTime,
    MAX_BY(PeriodBudgetLimit, ModifyTime) AS PeriodBudgetLimit,
    MAX_BY(PeriodBudgetLimitCur, ModifyTime) AS PeriodBudgetLimitCur,
    MAX_BY(LimitWeek, ModifyTime) AS LimitWeek,
    MAX_BY(LimitWeekCur, ModifyTime) AS LimitWeekCur,
FROM (
    SELECT
        StrategyID,
        MAX(ModifyTime) AS ModifyTime,
        MAX_BY(StartTime, ModifyTime) AS StartTime,
        MAX_BY(OrderID, ModifyTime) AS OrderID,
        MAX_BY(PeriodBudgetFinish, ModifyTime) AS PeriodBudgetFinish,
        MAX_BY(LastUpdateTime, ModifyTime) AS LastUpdateTime,
        MAX_BY(PeriodBudgetLimit, ModifyTime) AS PeriodBudgetLimit,
        MAX_BY(PeriodBudgetLimitCur, ModifyTime) AS PeriodBudgetLimitCur,
        MAX_BY(LimitWeek, ModifyTime) AS LimitWeek,
        MAX_BY(LimitWeekCur, ModifyTime) AS LimitWeekCur,
    FROM
        `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
    WHERE
        $CurWeek = $Today
        AND $ToDatetime(StartTime) < $PrevWeek
    GROUP BY
        StrategyID
    HAVING
        MAX_BY(LimitWeek, ModifyTime) > 0
        OR MAX_BY(LimitWeekCur, ModifyTime) > 0
    UNION ALL
    SELECT
        StrategyID,
        MAX(ModifyTime) AS ModifyTime,
        MAX_BY(StartTime, ModifyTime) AS StartTime,
        MAX_BY(OrderID, ModifyTime) AS OrderID,
        MAX_BY(PeriodBudgetFinish, ModifyTime) AS PeriodBudgetFinish,
        MAX_BY(LastUpdateTime, ModifyTime) AS LastUpdateTime,
        MAX_BY(PeriodBudgetLimit, ModifyTime) AS PeriodBudgetLimit,
        MAX_BY(PeriodBudgetLimitCur, ModifyTime) AS PeriodBudgetLimitCur,
        MAX_BY(LimitWeek, ModifyTime) AS LimitWeek,
        MAX_BY(LimitWeekCur, ModifyTime) AS LimitWeekCur,
    FROM
        `//home/yabs/dict/CaesarAutoBudgetOrderWithHistory`
    WHERE
        $CurWeek = $Today
        AND $ToDatetime(StartTime) >= $PrevWeek
        AND $ToDatetime(StartTime) < $CurWeek
        AND $IsWeekLimited(TableRow())
    GROUP BY
        StrategyID, StartTime
)
GROUP BY
    StrategyID, StartTime
;


$PeriodAndWeekLimitedStrategies = SELECT
    S.StrategyID AS StrategyID,
    StartTime,
    IF($IsPeriod(TableRow()), StartTime, MAX_OF(StartTime, DateTime::ToSeconds($PrevWeek))) AS PeriodStartTime,
    IF($IsPeriod(TableRow()), PeriodBudgetFinish, DateTime::ToSeconds($CurWeek)) AS PeriodFinishTime,
    PeriodBudgetFinish,
    LastUpdateTime,
    PeriodBudgetLimit,
    PeriodBudgetLimitCur,
    LimitWeek,
    LimitWeekCur,
    CurrencyID,
    CurrencyID != 0 AS IsCurrency,
    $RangeDays(DateTime::ToSeconds($MinStartDay) , DateTime::ToSeconds($Now)) AS Day,
FROM
    $PeriodAndWeekLimitedStrategies AS S
INNER JOIN
    (SELECT OrderID, CurrencyID, EngineID FROM `//home/yabs/dict/CaesarOrderInfo`) AS I
ON
    S.OrderID = I.OrderID
WHERE
    EngineID = 7
;


$PeriodAndWeekLimitedStrategies = SELECT
    *
FROM
    $PeriodAndWeekLimitedStrategies
FLATTEN BY
    Day
;

$Logs = SELECT
    autobudgetstrategyid,
    autobudgetstarttime,
    autobudgetsoftrestarttime,
    eventcost,
    costcur,
    eventtime,
    $ToSeconds($RoundToDayStart($ToDatetime(eventtime))) AS Day,
FROM
    RANGE(
        `//cooked_logs/bs-chevent-cooked-log/1d`
      , DateTime::Format('%Y-%m-%d')($ToDatetime($MinStartTime))
      , DateTime::Format('%Y-%m-%d')($Yesterday)
    )
WHERE
    NOT `autobudgetoptions_paid-actions`
;

$PeriodAndWeekLimitedStrategiesExpenses = SELECT
    StrategyID,
    StartTime,
    PeriodStartTime,
    PeriodFinishTime,
    PeriodBudgetFinish,
    LastUpdateTime,
    PeriodBudgetLimit,
    PeriodBudgetLimitCur,
    LimitWeek,
    LimitWeekCur,
    CurrencyID,
    IsCurrency,
    SUM(L.eventcost) AS Cost,
    SUM(L.costcur) AS CostCur,
    MIN(L.eventtime) AS FirstEvent,
    MAX(L.eventtime) AS LastEvent,
    Day,
FROM
    $PeriodAndWeekLimitedStrategies AS S
LEFT JOIN
    $Logs AS L
ON
    (
        L.autobudgetstrategyid = S.StrategyID
        AND L.autobudgetstarttime = S.StartTime
        AND L.Day = S.Day
    )
WHERE
    L.autobudgetstrategyid IS NULL OR (
        $ToDatetime(L.eventtime) >= $ToDatetime(S.PeriodStartTime)
        AND $ToDatetime(L.eventtime) < $ToDatetime(S.PeriodFinishTime)
    )
GROUP BY
    StrategyID AS StrategyID
  , StartTime AS StartTime
  , PeriodStartTime AS PeriodStartTime
  , PeriodFinishTime AS PeriodFinishTime
  , PeriodBudgetLimit AS PeriodBudgetLimit
  , PeriodBudgetLimitCur AS PeriodBudgetLimitCur
  , PeriodBudgetFinish AS PeriodBudgetFinish
  , LastUpdateTime AS LastUpdateTime
  , CurrencyID AS CurrencyID
  , IsCurrency AS IsCurrency
  , LimitWeek AS LimitWeek
  , LimitWeekCur AS LimitWeekCur
  , S.Day AS Day
;


$PeriodAndWeekLimitedStrategiesExpenses = SELECT
    StrategyID,
    StartTime,
    PeriodStartTime,
    PeriodFinishTime,
    PeriodBudgetFinish,
    LastUpdateTime,
    PeriodBudgetLimit,
    PeriodBudgetLimitCur,
    LimitWeek,
    LimitWeekCur,
    CurrencyID,
    IsCurrency,
    SUM(Cost) AS Cost,
    SUM(CostCur) AS CostCur,
    MIN(FirstEvent) AS FirstEvent,
    MAX(LastEvent) AS LastEvent,
FROM
    $PeriodAndWeekLimitedStrategiesExpenses
GROUP BY
    StrategyID AS StrategyID
  , StartTime AS StartTime
  , PeriodStartTime AS PeriodStartTime
  , PeriodFinishTime AS PeriodFinishTime
  , PeriodBudgetLimit AS PeriodBudgetLimit
  , PeriodBudgetLimitCur AS PeriodBudgetLimitCur
  , PeriodBudgetFinish AS PeriodBudgetFinish
  , LastUpdateTime AS LastUpdateTime
  , CurrencyID AS CurrencyID
  , IsCurrency AS IsCurrency
  , LimitWeek AS LimitWeek
  , LimitWeekCur AS LimitWeekCur
;

SELECT
    $ToSeconds($Now) AS RunTime,
    StrategyID,
    $GetLimitType(LimitWeek, PeriodBudgetLimit) AS LimitType,
    $GetLimitBudget(LimitWeek, PeriodBudgetLimit) AS LimitBudget,
    $GetLimitBudget(LimitWeekCur, PeriodBudgetLimitCur) AS LimitBudgetCur,

    Cost AS Budget,
    CostCur AS BudgetCur,
    CurrencyID,

    $GetExceed($GetLimitBudget(LimitWeek, PeriodBudgetLimit), Cost) AS Exceed,
    $GetExceed($GetLimitBudget(LimitWeekCur, PeriodBudgetLimitCur), CostCur) AS ExceedCur,
    $GetExceedRate($GetLimitBudget(LimitWeek, PeriodBudgetLimit), Cost) AS ExceedRate,
    $GetExceedRate($GetLimitBudget(LimitWeekCur, PeriodBudgetLimitCur), CostCur) AS ExceedCurRate,

    StartTime AS AutobudgetStartTime,
    PeriodStartTime,
    PeriodFinishTime,
    FirstEvent,
    LastEvent,
FROM
    $PeriodAndWeekLimitedStrategiesExpenses
WHERE
    (
        (NOT IsCurrency AND $GetLimitBudget(LimitWeek, PeriodBudgetLimit) > 0)
        OR (IsCurrency AND $GetLimitBudget(LimitWeekCur, PeriodBudgetLimitCur) > 0)
    ) AND (
        (NOT IsCurrency AND $GetExceedRate($GetLimitBudget(LimitWeek, PeriodBudgetLimit), Cost) > $MinExceedRate)
        OR (IsCurrency AND $GetExceedRate($GetLimitBudget(LimitWeekCur, PeriodBudgetLimitCur), CostCur) > $MinExceedRate)
    )
;

SELECT
    $ToSeconds($Now) AS RunTime,
    COUNT(*) AS TotalCheckedRows,
FROM
    $PeriodAndWeekLimitedStrategiesExpenses
;
'''
