# -*- coding: utf-8 -*-

import datetime

from sandbox import common
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.autobudget.back_to_back.lib.mysql_backup import YabsAutobudgetPrepareMySQLBackup, COMMON_TABLES
from sandbox.projects.autobudget.back_to_back.lib.yt_backup import YabsAutobudgetPrepareYTBackup
import sandbox.common.types.task as ctt


class YabsAutobudgetPrepareDBBackup(sdk2.Task):
    """
        Prepare Autobudget database backup
    """

    class Parameters(sdk2.Parameters):
        use_date_of_execution = sdk2.parameters.Bool(
            "Set backup date on execution",
            default=True,
        )
        with use_date_of_execution.value[False]:
            backup_date = sdk2.parameters.String(
                "Backup date (YYYYMMDD)",
                default=None,
            )

        with sdk2.parameters.Group("YT backup parameters"):
            yt_base_path = sdk2.parameters.String(
                "Destination map node template",
                default="//home/autobudget/sandbox/back_to_back/backup/{task_id}",
            )

        with sdk2.parameters.Group("MySQL backup parameters"):
            basenos = sdk2.parameters.List(
                "Prepare backup for these basenos",
                sdk2.parameters.Integer,
                default=[11, 127, 130, 138],
            )
            common_tables_description = sdk2.parameters.JSON(
                "Common tables backup description",
                default=COMMON_TABLES,
            )

        with sdk2.parameters.Group("Internal parameters"):
            yt_proxy = sdk2.parameters.String(
                "YT proxy to prepare backup at",
                default="hahn",
            )
            sandbox_api_base_url = sdk2.parameters.String(
                "Sandbox API base URL",
                default="http://sandbox.yandex-team.ru/api/v1.0",
            )
            token_vault_name = sdk2.parameters.String(
                "Sandbox OAuth token vault name",
                default="sandbox-token",
            )
            myisamchk_binary = sdk2.parameters.Resource(
                "Myisamchk binary",
                required=True,
            )

    def on_execute(self):
        with self.memoize_stage.make_backup_date_fixed:
            if self.Parameters.use_date_of_execution:
                backup_date = datetime.datetime.now().strftime("%Y%m%d")
            else:
                backup_date = self.Parameters.backup_date
            self.Context.fixed_backup_date = backup_date

        backup_date = self.Context.fixed_backup_date
        common_tables_description = self.Parameters.common_tables_description
        myisamchk_binary = self.Parameters.myisamchk_binary
        sandbox_api_base_url = self.Parameters.sandbox_api_base_url
        token_vault_name = self.Parameters.token_vault_name
        yt_base_path = self.Parameters.yt_base_path
        yt_proxy = self.Parameters.yt_proxy

        with self.memoize_stage.run_subtasks:
            subtasks = []
            for baseno in self.Parameters.basenos:
                mysql_backup_task = YabsAutobudgetPrepareMySQLBackup(
                    self,
                    use_date_of_execution=False,
                    backup_date=backup_date,
                    baseno=baseno,
                    common_tables_description=common_tables_description,
                    sandbox_api_base_url=sandbox_api_base_url,
                    token_vault_name=token_vault_name,
                    myisamchk_binary=myisamchk_binary,
                )
                mysql_backup_task.Requirements.tasks_resource = self.Requirements.tasks_resource
                mysql_backup_task.save()
                subtasks.append(mysql_backup_task)

            yt_backup_task = YabsAutobudgetPrepareYTBackup(
                self,
                use_date_of_execution=False,
                backup_date=backup_date,
                yt_base_path=yt_base_path,
                yt_proxy=yt_proxy,
            )
            yt_backup_task.Requirements.tasks_resource = self.Requirements.tasks_resource
            yt_backup_task.save()
            subtasks.append(yt_backup_task)

            self.Context.subtask_ids = []
            for task in subtasks:
                task.enqueue()
                self.Context.subtask_ids.append(task.id)

            raise sdk2.WaitTask(
                subtasks,
                common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK),
            )

        subtasks = [sdk2.Task[task_id] for task_id in self.Context.subtask_ids]

        def succeed(task):
            return task.status == ctt.Status.SUCCESS

        if not all(succeed(subtask) for subtask in subtasks):
            raise TaskFailure("Some of backups has failed")
