import tarfile

from sandbox import sdk2
from sandbox.projects.sandbox.sandbox_lxc_image import SandboxLxcImage
from sandbox.projects.autobudget.back_to_back.lib.resources import (
    YabsOldAutobudgetMainPackage,
    YabsOldAutobudgetFunctionalTestsPackage,
    YabsOldAutobudgetParaspriteTestsPackage,
)


INSTALL_FT_DEPENDENCIES = "install_ft_dependencies.sh"
INSTALL_DEPENDENCIES = "install_dependencies.sh"
SETUP_SCRIPTS_PATH = "yabs/stat/yabs-autobudget2/setup-scripts"


class YabsOldAutobudgetBuildLxcImage(SandboxLxcImage):
    class Parameters(SandboxLxcImage.Parameters):
        main_package = sdk2.parameters.Resource(
            "Resource with yabs-autobudget2 package",
            resource_type=YabsOldAutobudgetMainPackage,
            required=True,
        )
        functional_tests_package = sdk2.parameters.Resource(
            "Resource with yabs-autobudget-ft package",
            resource_type=YabsOldAutobudgetFunctionalTestsPackage,
        )
        parasprite_tests_package = sdk2.parameters.Resource(
            "Resource with yabs-autobudget2-pt package",
            resource_type=YabsOldAutobudgetParaspriteTestsPackage,
        )
        with sdk2.parameters.Group("Arcadia parameters"):
            arcadia_url = sdk2.parameters.ArcadiaUrl('Arcadia Url')
            arcadia_patch = sdk2.parameters.String(
                "Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4",
                default="",
                multiline=True,
            )

    def cook_image(self, current_dir):
        super(YabsOldAutobudgetBuildLxcImage, self).cook_image(current_dir)
        main_package = self.Parameters.main_package
        functional_tests_package = self.Parameters.functional_tests_package
        parasprite_tests_package = self.Parameters.parasprite_tests_package
        arcadia_patch = self.Parameters.arcadia_patch
        arcadia_url = sdk2.svn.Arcadia.append(self.Parameters.arcadia_url, SETUP_SCRIPTS_PATH)

        fake_arcadia_root = self.path() / "arcadia"
        setup_scripts_dir = fake_arcadia_root / SETUP_SCRIPTS_PATH
        sdk2.svn.Arcadia.checkout(
            arcadia_url,
            setup_scripts_dir,
        )

        if arcadia_patch:
            sdk2.svn.Arcadia.apply_patch(fake_arcadia_root, arcadia_patch, self.path())

        for script in (INSTALL_DEPENDENCIES, INSTALL_FT_DEPENDENCIES):
            self.execute(
                " ".join(
                    [
                        "cp",
                        str(setup_scripts_dir / script),
                        self.rootfs + "/" + script,
                    ]
                )
            )

        with sdk2.helpers.ProgressMeter("Installing dependencies..."):
            self.execute("/" + INSTALL_DEPENDENCIES, chroot=True)
            if functional_tests_package is not None:
                # Need to provide /etc/hosts for new container as a workaround for local YT installation:
                # This task's parent cleans /etc/hosts in container (https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/sandbox/sandbox_lxc_image/__init__.py?rev=4243693#L393)
                # But YT needs successfull address resolution by local hostname
                # Therefore we need to have /etc/hosts filled during installation
                # For more information check out this thread -- https://ml.yandex-team.ru/thread/sandbox/167477611142889162/#message167477611142889162
                self.execute("cp /etc/hosts " + self.rootfs + "/etc/hosts")
                self.execute("/" + INSTALL_FT_DEPENDENCIES, chroot=True)
                self.execute("truncate " + self.rootfs + "/etc/hosts --size 0")

        with sdk2.helpers.ProgressMeter("Untaring debian packages..."):
            debian_packages_to_install = []
            for package in [main_package, functional_tests_package, parasprite_tests_package]:
                if package is None:
                    continue
                tarball_path = sdk2.ResourceData(package).path

                with tarfile.open(str(tarball_path)) as tarball:
                    for filename in tarball.getnames():
                        if filename.endswith(".deb"):
                            tarball.extract(filename, self.rootfs)
                            debian_packages_to_install.append(filename)

        with sdk2.helpers.ProgressMeter("Installing packages..."):
            self.execute(
                "dpkg --install " + " ".join(debian_packages_to_install) + " || true",
                chroot=True,
            )
            self.execute(
                "ASSUME_YES=1 DEBIAN_FRONTEND=noninteractive apt-get -f install --yes --force-yes --no-install-recommends",
                chroot=True,
            )
            self.execute(
                "dpkg --install " + " ".join(debian_packages_to_install),
                chroot=True,
            )
