# -*- coding: utf-8 -*-

from __future__ import print_function

import json
import textwrap
import tempfile

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.common.types.misc as ctm

from sandbox.projects.yabs.base_bin_task import BaseBinTask

from sandbox.projects.autobudget.back_to_back.lib.comparator.diff import Diff
from sandbox.projects.autobudget.back_to_back.lib.comparator.query_template import MAIN_QUERY, TABLE_QUERIES
from sandbox.projects.autobudget.back_to_back.lib.comparator.representation import DIFF_TEMPLATE
from sandbox.projects.autobudget.back_to_back.lib.comparator.table_schema import TABLE_SCHEMA
from sandbox.projects.autobudget.back_to_back.lib.stand import YabsOldAutobudgetBackToBackStand


def render_query(query_template, cluster, tables):
    from jinja2 import Environment

    env = Environment()
    template = env.from_string(query_template)
    return template.render(
        cluster=cluster,
        tables=tables,
        table_schema=TABLE_SCHEMA,
    )


def tables_diff(yql_token, cluster, tables, testing_stand, stable_stand):
    from yql.api.v1.client import YqlClient
    from yql.client.parameter_value_builder import YqlParameterValueBuilder
    from yql.config import config as yql_config

    progress_on_stderr = yql_config.progress_on_stderr
    yql_config.progress_on_stderr = False

    with tempfile.NamedTemporaryFile() as f:
        table_queries = render_query(TABLE_QUERIES, cluster, tables)
        f.write(table_queries)
        f.flush()
        with YqlClient(db=cluster, token=yql_token) as yql:
            yql.progress_on_stderr = False
            main_query = render_query(MAIN_QUERY, cluster, tables)
            query = yql.query(main_query, title="YQL Tables Comparison", syntax_version=1)
            query.attach_file(f.name, "table_queries.sql")
            query.run(
                parameters=YqlParameterValueBuilder.build_json_map(
                    {
                        "$testing_dir_path": YqlParameterValueBuilder.make_string("//home/autobudget/sandbox/back_to_back/stand/{dir}".format(dir=testing_stand)),
                        "$stable_dir_path": YqlParameterValueBuilder.make_string("//home/autobudget/sandbox/back_to_back/stand/{dir}".format(dir=stable_stand)),
                    },
                ),
            )
        return Diff(query)
    yql_config.progress_on_stderr = progress_on_stderr


class YabsOldAutobudgetBackToBackComparator(BaseBinTask):
    class Parameters(BaseBinTask.Parameters):
        testing_stand = sdk2.parameters.Task(
            "Testing stand task",
            task_type=YabsOldAutobudgetBackToBackStand,
            required=True,
        )
        stable_stand = sdk2.parameters.Task(
            "Stable stand task",
            task_type=YabsOldAutobudgetBackToBackStand,
            required=True,
        )

        tables_to_diff = sdk2.parameters.List(
            "List of tables to diff",
            sdk2.parameters.String,
            default=list(TABLE_SCHEMA.keys()),
        )

        cluster = sdk2.parameters.String(
            "Cluster to run queries on",
            default="hahn",
        )

        with sdk2.parameters.Group("Infrastructure parameters"):
            resource_attrs = sdk2.parameters.Dict(
                "Filter resource by",
                default={"name": "autobudget-back-to-back-binary"},
                description="Will be passed to 'attrs' search parameter",
            )

    class Requirements(BaseBinTask.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        from jinja2 import Environment, select_autoescape

        testing_stand = self.Parameters.testing_stand
        stable_stand = self.Parameters.stable_stand
        tables_to_diff = self.Parameters.tables_to_diff
        cluster = self.Parameters.cluster
        yql_token = sdk2.Vault.data(self.owner, "YQL_TOKEN")
        self.Context.has_diff = False

        if not testing_stand.Parameters.autobudget_succeed:
            self.set_info("Testing version failed")
            self.Context.has_diff = True
        if not stable_stand.Parameters.autobudget_succeed:
            self.set_info("Stable version failed")
            self.Context.has_diff = True

        if set(testing_stand.Parameters.missing_tables) != set(stable_stand.Parameters.missing_tables):
            self.set_info(
                textwrap.dedent(
                    """
                        Missing tables lists are not equal:
                        Testing: {testing_tables}
                        Stable: {stable_tables}
                    """
                ).format(
                    testing_tables=json.dumps(testing_stand.Parameters.missing_tables, indent=4, separators=(',', ': ')),
                    stable_tables=json.dumps(stable_stand.Parameters.missing_tables, indent=4, separators=(',', ': ')),
                ),
            )
            self.Context.has_diff = True
        elif testing_stand.Parameters.missing_tables:
            self.set_info(
                """Although sets of missing tables are equal, they aren't empty: {tables}""".format(
                    tables=json.dumps(testing_stand.Parameters.missing_tables, indent=4, separators=(',', ': ')),
                ),
            )
            self.Context.has_diff = True

        diff = tables_diff(yql_token, cluster, tables_to_diff, testing_stand.id, stable_stand.id)
        self.set_info("Ran query: {query}".format(query=diff.operation_url))

        if diff.has_diff:
            content = diff.content

            env = Environment(autoescape=select_autoescape(["html"]))
            template = env.from_string(DIFF_TEMPLATE)

            result = []
            for table in content:
                if table != "TablePresence":
                    table_schema = TABLE_SCHEMA[table]
                    result.append(
                        template.render(
                            diff_values=content[table]["diff_values"],
                            diff_columns=content[table]["diff_columns"],
                            table_schema=table_schema,
                        )
                    )

            self.Context.diff_content = "<br>".join(result)

            self.Context.has_diff = True
        elif not diff.query_succeed:
            raise TaskFailure("Query failed")

        if self.Context.has_diff:
            self.set_info("Test failed")
        else:
            self.set_info("All is OK")

    @sdk2.report(title="Diff")
    def diff(self):
        if self.Context.diff_content is not ctm.NotExists:
            return self.Context.diff_content
        else:
            return "<img src=\"https://jing.yandex-team.ru/files/snusmumrik/2019-01-12%2001.41.35.jpg\"/>"
