import os
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
import time
import uuid

from sandbox import common
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.autobudget.back_to_back.lib.comparator import YabsOldAutobudgetBackToBackComparator
from sandbox.projects.autobudget.back_to_back.lib.config import (
    MYSQL_TABLES_TO_PACK,
    TESTING_FILE_CONTENTS,
)
from sandbox.projects.autobudget.back_to_back.lib.resources import YabsOldAutobudgetLxcImage, YabsAutobudgetMysqlBackup
from sandbox.projects.autobudget.back_to_back.lib.stand import YabsOldAutobudgetBackToBackStand
from sandbox.projects.yabs.global_info.resources import YABS_APP_OPTIONS


def generate_unique_path(base_path):
    return os.path.join(base_path, uuid.uuid4().hex)


class YabsOldAutobudgetBackToBackController(sdk2.Task):
    class Parameters(sdk2.Parameters):
        baseno = sdk2.parameters.Integer(
            "Baseno to run tests for",
            required=True,
        )
        testing_package = sdk2.parameters.String(
            "Package to be tested",
            default="147.1",
            required=True,
        )
        stable_package = sdk2.parameters.String(
            "Stable package",
            default="148.1",
            required=True,
        )
        base_results_path = sdk2.parameters.String(
            "Base path to store stand outputs to",
            required=True,
        )
        testing_file_contents = sdk2.parameters.String(
            "Contents of testing file",
            default=TESTING_FILE_CONTENTS,
            multiline=True,
            required=True,
        )
        mysql_tables_to_pack = sdk2.parameters.List(
            "Result tables of autobudget to pack into resource",
            default=MYSQL_TABLES_TO_PACK,
            required=True,
        )

    def on_execute(self):
        baseno = self.Parameters.baseno
        testing_package = self.Parameters.testing_package
        stable_package = self.Parameters.stable_package
        mysql_tables_to_pack = self.Parameters.mysql_tables_to_pack
        testing_file_contents = self.Parameters.testing_file_contents

        with self.memoize_stage.run_stands:
            now = time.time()
            mysql_archive_resource = YabsAutobudgetMysqlBackup.find(state=ctr.State.READY, attrs={"baseno": self.Parameters.baseno}).order(-sdk2.Resource.id).first()
            host_options_resource = YABS_APP_OPTIONS.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()
            testing_container = YabsOldAutobudgetLxcImage.find(state=ctr.State.READY, attrs={"yabs_autobudget2_version": testing_package}).order(-sdk2.Resource.id).first()
            stable_container = YabsOldAutobudgetLxcImage.find(state=ctr.State.READY, attrs={"yabs_autobudget2_version": stable_package}).order(-sdk2.Resource.id).first()
            testing_stand = YabsOldAutobudgetBackToBackStand(
                self,
                baseno=baseno,
                container=testing_container,
                description="Test stand for package {package}".format(package=testing_package),
                host_options_resource=host_options_resource,
                mysql_archive_resource=mysql_archive_resource,
                mysql_tables_to_pack=mysql_tables_to_pack,
                notifications=[],
                now=now,
                testing_file_contents=testing_file_contents,
            ).enqueue()
            stable_stand = YabsOldAutobudgetBackToBackStand(
                self,
                baseno=baseno,
                container=stable_container,
                description="Test stand for package {package}".format(package=stable_package),
                host_options_resource=host_options_resource,
                mysql_archive_resource=mysql_archive_resource,
                mysql_tables_to_pack=mysql_tables_to_pack,
                notifications=[],
                now=now,
                testing_file_contents=testing_file_contents,
            ).enqueue()

            self.Context.testing_stand_id = testing_stand.id
            self.Context.stable_stand_id = stable_stand.id

            raise sdk2.WaitTask(
                [testing_stand, stable_stand],
                common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK),
            )

        testing_stand = sdk2.Task[self.Context.testing_stand_id]
        stable_stand = sdk2.Task[self.Context.stable_stand_id]
        if stable_stand.status != ctt.Status.SUCCESS or testing_stand.status != ctt.Status.SUCCESS:
            raise TaskFailure("Either testing or stable stand failed to run")

        with self.memoize_stage.compare_results:
            comparator_task = YabsOldAutobudgetBackToBackComparator(
                self,
                testing_stand=testing_stand,
                stable_stand=stable_stand,
            )
            comparator_task.Requirements.tasks_resource = self.Requirements.tasks_resource
            comparator_task.enqueue()

            self.Context.comparator_task_id = comparator_task.id

            raise sdk2.WaitTask(
                [comparator_task],
                common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK),
            )

        comparator_task_id = self.Context.comparator_task_id
        if self.__is_failed(comparator_task_id):
            raise TaskFailure("Stands successfully finished, but comparator task seems to be failed")

    def __is_failed(self, task_id):
        return not self.find(id=task_id, status=ctt.Status.SUCCESS).count
