import os
import shutil
import xml.etree.ElementTree as ET
from collections import Counter
from tempfile import NamedTemporaryFile

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.errors import TaskFailure

from sandbox.projects.autobudget.back_to_back.lib.resources import (
    YabsOldAutobudgetFunctionalTestsResult,
    YabsOldAutobudgetLxcImage,
)

FT_ROOT_PATH = '/home/yabs/stat/ft'


class YabsOldAutobudgetFunctionalTests(sdk2.Task):
    class Parameters(sdk2.Parameters):
        container = sdk2.parameters.Container(
            "Prebuilt environment",
            default_value=None,
            resource_type=YabsOldAutobudgetLxcImage,
            required=True,
        )
        yt_setup_retries = sdk2.parameters.Integer(
            "YT setup retries",
            default=5,
        )

    class Requirements(sdk2.Requirements):
        privileged = True

    def on_execute(self):
        with sdk2.helpers.ProgressMeter("Setting up environment..."):
            self._setup_environment()

        # Convert errors while running tests into failures
        self.Context.fail_on_any_error = True
        self.Context.save()

        with sdk2.helpers.ProgressMeter("Running tests..."):
            self._run_tests()

        self.Context.fail_on_any_error = False
        self.Context.save()

        with sdk2.helpers.ProgressMeter("Preparing test report..."):
            result = YabsOldAutobudgetFunctionalTestsResult(
                task=self,
                description="Autobudget FT results",
                path="report",
            )

            xsltproc_path = '/usr/bin/xsltproc'
            xmllog_xsl_path = os.path.join(FT_ROOT_PATH, 'xmllog.xsl')
            log_error_path = os.path.join(FT_ROOT_PATH, 'log_error')
            log_info_path = os.path.join(FT_ROOT_PATH, 'log_info')
            test_result_path = os.path.join(FT_ROOT_PATH, 'test-result')

            result_data = sdk2.ResourceData(result)

            os.makedirs('report')
            if os.path.isfile(log_info_path):
                shutil.copyfile(log_info_path, 'report/log_info.txt')
            if os.path.isfile(log_error_path):
                shutil.copyfile(log_error_path, 'report/log_error.txt')
            if os.path.isdir(test_result_path):
                shutil.copytree(test_result_path, 'report/test-result')

            if os.path.isfile(xmllog_xsl_path) and os.path.isfile(xsltproc_path):
                os.makedirs('report/html')
                for filename in os.listdir('report/test-result'):
                    orig_xml_path = os.path.join('report/test-result', filename)
                    if not os.path.isfile(orig_xml_path) or not orig_xml_path.endswith('.xml'):
                        continue

                    html_path = os.path.join('report/html', filename.replace('.xml', '.html'))
                    with NamedTemporaryFile('w+b', delete=False) as input_xml:
                        with open(orig_xml_path, 'r+b') as orig_xml:
                            xml_data = orig_xml.read()

                        xml_data = xml_data.replace('.xml', '.html')  # fix cross-links between pages
                        input_xml.write(xml_data)
                        input_xml.close()

                        with open(html_path, 'w+b') as output_html:
                            sp.check_call(
                                [xsltproc_path, xmllog_xsl_path, input_xml.name],
                                stdout=output_html,
                            )

            result_data.ready()

        test_report = self._find_test_report(test_result_path)
        if test_report is None:
            raise TaskFailure('Test report is not found')

        self.Context.test_report = test_report
        self.Context.report_url = result.http_proxy
        self.Context.save()

        is_result_ok = all(
            test.get('result') == 'passed'
            for test in test_report
        )
        if not is_result_ok:
            test_count = len(test_report)
            failed_tests = [
                test['name']
                for test in test_report
                if test['result'] != 'passed'
            ]
            raise TaskFailure('{failed} / {total} tests have failed:\n{tests}'.format(
                failed=len(failed_tests),
                total=test_count,
                tests='\n'.join(sorted(failed_tests)),
            ))

    def _setup_environment(self):
        yt_setup_retries = self.Parameters.yt_setup_retries

        with sdk2.helpers.ProcessLog(self, "environment_setup") as pl:
            sp.check_call(
                ["sudo", "service", "runsvdir", "stop"],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            )
            sp.check_call(
                ["sudo", "service", "mysql.yabs", "restart"],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            )
            self._retry_call(yt_setup_retries, lambda: sp.check_call(
                ["sudo", "service", "yt_local.yabs", "restart"],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ))

    def _run_tests(self):
        with sdk2.helpers.ProcessLog(self, "tests") as pl:
            sp.check_call(
                ["sudo", "./run.pl", "--junitlog", "1"],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
                cwd=FT_ROOT_PATH,
            )

    def _retry_call(self, retry_count, function):
        for retries_left in range(retry_count, 0, -1):
            try:
                return function()
            except Exception:
                if retries_left > 1:
                    continue
                raise

    def _find_test_report(self, results_dir):
        if not os.path.isdir(results_dir):
            return None

        for filename in os.listdir(results_dir):
            path = os.path.join(results_dir, filename)
            if not os.path.isfile(path) or not filename.endswith('.xml'):
                continue

            try:
                xml_data = ET.parse(path)
            except Exception:
                continue

            if xml_data.getroot().tag != 'log':
                continue

            return [dict(
                name=item.attrib.get('name', '???'),
                start=item.attrib.get('start', '???'),
                result=item.findtext('result', '???').strip(),
                duration=item.findtext('duration', '???').strip(),
            ) for item in xml_data.findall('.//test')]

        return None

    @sdk2.header()
    def render_test_report(self):
        test_report = self.Context.test_report
        if not test_report:
            return ''

        report_url = self.Context.report_url

        tests_by_status = Counter(_['result'] for _ in test_report)

        def iter_parts():
            yield '<style>'
            yield '.abft-table-summary {} '
            yield '.abft-table-results {width: 100%} '
            yield '.abft-status {text-transform: uppercase; font-weight: bold; letter-spacing: 1px; } '
            yield '.abft-status-passed {color: rgb(24, 166, 81); } '
            yield '.abft-status-failed {color: white; background-color: rgb(253, 13, 27); } '
            yield '</style>'

            if report_url:
                yield '<a href="{url}">View detailed HTML report</a>'.format(
                    url=report_url + '/html/report.html',
                )

            yield '<h3>Summary</h3>'
            yield '<table class="abft-table-summary">'
            yield '<tr><th>Total</th><td>{count} tests</td></tr>'.format(count=len(test_report))
            for status, count in sorted(tests_by_status.items()):
                yield '<tr><th class="abft-status abft-status-{status}">{status}</th><td>{count} tests</td></tr>'.format(
                    status=status,
                    count=count,
                )

            yield '</table>'

            yield '<h3>Test Results</h3>'
            yield '<table class="abft-table-results">'
            yield '<tr><th>Status</th><th>Test</th><th>Start time</th><th>Duration</th></tr>'
            for test in test_report:
                yield '<tr>'
                yield '<td class="abft-status abft-status-{status}">{status}</td>'.format(status=test['result'])
                yield '<td><a href="{url}">{name}</name></td>'.format(
                    name=test['name'],
                    url=report_url + '/test-result/' + test['name'] if report_url else '#',
                )
                yield '<td>{start}</td>'.format(start=test['start'])
                yield '<td>{duration}</td>'.format(duration=test['duration'])
                yield '</tr>'
            yield '</table>'

        return ''.join(iter_parts())
