# -*- coding: utf-8 -*-

import datetime
import json
import tarfile

from sandbox import common
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.proxy import OAuth
from sandbox.common.rest import Client
from sandbox.projects.autobudget.back_to_back.lib.compress import Compressor
from sandbox.projects.autobudget.back_to_back.lib.resources import YabsAutobudgetMysqlBackup
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.share as cs
import sandbox.common.types.task as ctt


COMMON_TABLES = {
    "yabsst{baseno}.yabsdb.ABBannerPhraseCurWeek": "bstat{baseno}",
    "yabsst{baseno}.yabsdb.ABExperimentCurWeek": "bstat{baseno}",
    "yabsst{baseno}.yabsdb.BroadMatchBanner": "bstat{baseno}",
    "yabsst{baseno}.yabsdb.BroadMatchBannerTime": "bstat{baseno}",
    "yabsst{baseno}.yabsdb.OrderTargetStat": "bstat{baseno}",
    "yabsst{baseno}.yabsdb.PhrasePrice": "bstat{baseno}",
    "yabs.yabsdb.Banner{baseno}": "bstat{baseno}",
    "yabs.yabsdb.Context{baseno}": "bstat{baseno}",
    "yabs.yabsdb.AutoBudgetCoeffs": "bsdb",
    "yabs.yabsdb.AutoBudgetOrder": "bsdb",
    "yabs.yabsdb.AutoBudgetStatCurWeek": "bsdb",
    "yabs.yabsdb.BroadmatchLimits": "bsdb",
    "yabs.yabsdb.ContextLimits": "bsdb",
    "yabs.yabsdb.Currency": "bsdb",
    "yabs.yabsdb.CurrencyRates": "bsdb",
    "yabs.yabsdb.DataTime": "bsdb",
    "yabs.yabsdb.Holiday": "bsdb",
    "yabs.yabsdb.OrderInfo": "bsdb",
    "yabs.yabsdb.OrderStat": "bsdb",
    "yabs.yabsdb.Pattern": "bsdb",
    "yabs.yabsdb.TaxHistory": "bsdb",
    "yabs.yabsdb.TimeZone": "bsdb",
    "yabs.yabsinfo.DLockConfig": "bsdb",
    "yabs.yabsinfo.Hosts": "bsdb",
}

DB_RESOURCE_PATH = "mysql.yabs"

TABLES_TO_UNPACK = [
    "Banner{baseno}",
]


class YabsAutobudgetPrepareMySQLBackup(sdk2.Task):
    """
        Prepare Autobudget MySQL backup
    """

    class Parameters(sdk2.Parameters):
        use_date_of_execution = sdk2.parameters.Bool(
            "Set backup date on execution",
            default=True,
        )
        with use_date_of_execution.value[False]:
            backup_date = sdk2.parameters.String(
                "Backup date (YYYYMMDD)",
                default=None,
            )

        baseno = sdk2.parameters.Integer(
            "Prepare backup for this baseno",
            required=True,
        )
        common_tables_description = sdk2.parameters.JSON(
            "Common tables backup description",
            default=COMMON_TABLES,
        )

        with sdk2.parameters.Group("Internal parameters"):
            sandbox_api_base_url = sdk2.parameters.String(
                "Sandbox API base URL",
                default="http://sandbox.yandex-team.ru/api/v1.0",
            )
            token_vault_name = sdk2.parameters.String(
                "Sandbox OAuth token vault name",
                default="sandbox-token",
            )
            myisamchk_binary = sdk2.parameters.Resource(
                "Myisamchk binary",
                required=True,
            )

    def on_execute(self):
        with self.memoize_stage.make_backup_date_fixed:
            if self.Parameters.use_date_of_execution:
                backup_date = datetime.datetime.now().strftime("%Y%m%d")
            else:
                backup_date = self.Parameters.backup_date
            self.Context.fixed_backup_date = backup_date

        self.prepare_mysql_backup()

    def prepare_mysql_backup(self):
        backup_date = self.Context.fixed_backup_date
        baseno = self.Parameters.baseno
        myisamchk = sdk2.ResourceData(sdk2.Resource[self.Parameters.myisamchk_binary]).path

        with self.memoize_stage.create_description(commit_on_entrance=False):
            restore_description = sdk2.Resource["YABS_MYSQL_RESTORE_DESCRIPTION"](
                self,
                description="Restore description resource with Autobudget tables",
                path="restore_description.json",
            )
            restore_description_data = sdk2.ResourceData(restore_description)
            tables_description = self.Parameters.common_tables_description

            full_restore_description = {}
            for table in tables_description.keys():
                full_restore_description[table.format(baseno=baseno)] = tables_description[table].format(baseno=baseno)

            self.Context.full_restore_description = full_restore_description
            restore_description_data.path.write_bytes(
                json.dumps(
                    full_restore_description,
                    sort_keys=True,
                    indent=4,
                    separators=(',', ': '),
                ),
            )
            restore_description_data.ready()
            self.Context.restore_description_resource_id = restore_description.id

        with self.memoize_stage.build_archive(commit_on_entrance=False):
            self.Context.restore_tables_task_id = self.start_subtask(
                task_type="YABS_SERVER_GET_SQL_ARCHIVE",
                description="Get Autobudget tables backup",
                date=backup_date,
                additional_restore_description_resource=self.Context.restore_description_resource_id,
                no_lm_dumps="no_lm_dumps",
                bin_db_list="",
            )

            raise sdk2.WaitTask(self.Context.restore_tables_task_id, common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))

        archive_task = self.find(status=ctt.Status.Group.SUCCEED, id=self.Context.restore_tables_task_id)
        if not archive_task.count:
            raise TaskFailure("Archive task failed")

        archive_resource_id = sdk2.Resource["YABS_MYSQL_ARCHIVE_CONTENTS"].find(
            task_id=self.Context.restore_tables_task_id,
        ).first()
        archive_contents = sdk2.ResourceData(archive_resource_id)
        tables = json.loads(archive_contents.path.read_text())["tables"]
        sandbox_client = Client(
            base_url="http://sandbox.yandex-team.ru/api/v1.0",
            auth=OAuth(sdk2.Vault.data(self.Parameters.token_vault_name)),
        )

        self.path().joinpath(DB_RESOURCE_PATH).mkdir()
        for table in self.Context.full_restore_description.keys():
            table_resource_id = tables[table]
            cs.skynet_get(
                skynet_id=sandbox_client.resource[table_resource_id].read()["skynet_id"],
                data_dir=str(self.path()),
                fallback_to_bb=True,
            )
            with tarfile.open(str(self.path() / "{table}.tar.gz".format(table=table))) as table_tar:
                table_tar.extractall(str(self.path() / DB_RESOURCE_PATH))

        with sdk2.helpers.ProcessLog(self, "unpack_tables") as pl:
            for table in TABLES_TO_UNPACK:
                table = table.format(baseno=baseno)
                sp.check_call(
                    [str(myisamchk), "--unpack", "--recover", str(self.path() / DB_RESOURCE_PATH / "yabsdb" / table)],
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                )

        result_resource = sdk2.ResourceData(
            YabsAutobudgetMysqlBackup(
                self,
                description="Tables pack for Autobudget",
                path="{path}.tar.zstd".format(path=DB_RESOURCE_PATH),
                baseno=baseno,
                date=backup_date,
            ),
        )
        compressor = Compressor()
        compressor.pack(
            source_path=DB_RESOURCE_PATH,
            destination_path=str(result_resource.path),
        )
        result_resource.ready()

    def start_subtask(self, task_type, description, **parameters):
        task_id = self.server.task({
            "children": True,
            "context": parameters,
            "description": description,
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.cls,
            },
            "notifications": [],
            "owner": self.owner,
            "type": task_type,
        })["id"]
        self.server.batch.tasks.start.update([task_id])
        return task_id
